% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lineqAGP.R
\name{predict.lineqAGP}
\alias{predict.lineqAGP}
\title{Prediction Method for the \code{"lineqAGP"} S3 Class}
\usage{
\method{predict}{lineqAGP}(object, xtest, ...)
}
\arguments{
\item{object}{an object with class \code{"lineqAGP"}.}

\item{xtest}{a vector (or matrix) with the test input design.}

\item{...}{further arguments passed to or from other methods.}
}
\value{
A \code{"lineqAGP"} object with the following elements.
\item{Lambda}{a matrix corresponding to the linear set of inequality constraints.}
\item{lb}{the lower bound vector of the inequalities constraints.}
\item{ub}{the upper bound vector of the inequalities constraints.}
\item{Phi.test}{a matrix corresponding to the hat basis functions evaluated
at \code{xtest}. The basis functions are indexed by rows.}
\item{mu}{the unconstrained GP mean predictor.}
\item{Sigma}{the unconstrained GP prediction conditional covariance matrix.}
\item{xi.map}{the GP maximum a posteriori (MAP) predictor given the inequality constraints.}
}
\description{
Prediction method for the \code{"lineqAGP"} S3 class.
}
\details{
The posterior paramaters of the finite-dimensional GP with linear inequality
constraints are computed. Here, \eqn{\boldsymbol{\xi}}{\xi} is a centred Gaussian
vector with covariance \eqn{\boldsymbol{\Gamma}}{\Gamma}, s.t.
\eqn{\boldsymbol{\Phi} \boldsymbol{\xi} = \boldsymbol{y}}{\Phi \xi = y}
(interpolation constraints) and
\eqn{\boldsymbol{l} \leq \boldsymbol{\Lambda} \boldsymbol{\xi} \leq \boldsymbol{u}}{lb \le \Lambda \xi \le ub}
(inequality constraints).
}
\examples{
library(plot3D)
# creating the model
d <- 2
fun1 <- function(x) return(4*(x-0.5)^2)
fun2 <- function(x) return(2*x)
targetFun <- function(x) return(fun1(x[, 1]) + fun2(x[, 2])) 
xgrid <- expand.grid(seq(0, 1, 0.01), seq(0, 1, 0.01))
ygrid <- targetFun(xgrid)
xdesign <- rbind(c(0.5, 0), c(0.5, 0.5), c(0.5, 1), c(0, 0.5), c(1, 0.5))
ydesign <- targetFun(xdesign)
model <- create(class = "lineqAGP", x = xdesign, y = ydesign,
                constrType = c("convexity", "monotonicity"))

# updating and expanding the model
model$localParam$m <- rep(10, d)
model$kernParam[[1]]$type <- "matern52"
model$kernParam[[2]]$type <- "matern52"
model$kernParam[[1]]$par <- c(1, 0.2)
model$kernParam[[2]]$par <- c(1, 0.3)
model$nugget <- 1e-9
model$varnoise <- 1e-5
model <- augment(model)

# predictions from the model
ntest <- 25
xtest  <- cbind(seq(0, 1, length = ntest), seq(0, 1, length = ntest))
ytest <- targetFun(xtest)
pred <- predict(model, xtest)
persp3D(x = unique(xtest[, 1]), y = unique(xtest[, 2]),
        z = outer(c(pred$Phi.test[[1]] \%*\% pred$xi.map[, 1]),
                  c(pred$Phi.test[[2]] \%*\% pred$xi.map[, 2]), "+"),
        xlab = "x1", ylab = "x2", zlab = "mode(x1,x2)", zlim = c(0, 3),
        phi = 20, theta = -30, alpha = 1, colkey = FALSE)
points3D(x = xdesign[,1], y = xdesign[,2], z = ydesign, col = "black", pch = 19, add = TRUE)

}
\references{
Lopez-Lopera, A. F., Bachoc, F., Durrande, N., and Roustant, O. (2017),
"Finite-dimensional Gaussian approximation with linear inequality constraints".
\emph{ArXiv e-prints}
\href{https://arxiv.org/abs/1710.07453}{[link]}
}
\seealso{
\code{\link{create.lineqAGP}}, \code{\link{augment.lineqAGP}},
         \code{\link{simulate.lineqAGP}}
}
\author{
A. F. Lopez-Lopera.
}
