% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simple_api.R
\name{link_plots}
\alias{link_plots}
\title{Simple Plot Linking Function for Non-Modular 'shiny' Apps}
\usage{
link_plots(
  session,
  ...,
  shared_id_column,
  leaflet_lng_col = "longitude",
  leaflet_lat_col = "latitude",
  leaflet_click_handler = NULL,
  dt_click_handler = NULL,
  on_selection_change = NULL
)
}
\arguments{
\item{session}{The 'shiny' session object}

\item{...}{Named arguments where names are component output IDs and values are
reactive data frames. Each data frame must contain the shared_id_column.
For leaflet maps: can be sf objects (coordinates auto-extracted) or regular
data frames with longitude/latitude columns.}

\item{shared_id_column}{Character string naming the column that contains unique
identifiers present in all linked components.}

\item{leaflet_lng_col}{Character string naming the longitude column for leaflet maps.
Defaults to "longitude". For sf objects, this will be the name of the created column.}

\item{leaflet_lat_col}{Character string naming the latitude column for leaflet maps.
Defaults to "latitude". For sf objects, this will be the name of the created column.}

\item{leaflet_click_handler}{Optional function that handles leaflet marker clicks.
This will be used for both direct clicks and when other components select this marker.
Function should accept (map_proxy, selected_data, session).}

\item{dt_click_handler}{Optional function that handles DT row selections.
This will be used for both direct clicks and when other components select this row.
Function should accept (dt_proxy, selected_data, session).}

\item{on_selection_change}{Optional callback function that gets called when selection changes.
Function should accept parameters: (selected_id, selected_data, source_component_id, session)}
}
\value{
Invisibly returns the created registry object
}
\description{
\code{link_plots} provides a simple, one-line interface to link interactive
components in a \strong{single-file or non-modular 'shiny' application}. It
automatically detects component types and sets up bidirectional linking.
}
\details{
This function is the fastest way to get started with \code{linkeR} and is ideal
for straightforward dashboards.

For more complex applications that use \strong{'shiny' Modules}, you should use the
more robust pattern of creating a central registry with \code{\link[=create_link_registry]{create_link_registry()}}
and passing it to your modules, where you will call \code{\link[=register_leaflet]{register_leaflet()}} or
\code{\link[=register_dt]{register_dt()}} directly. This preserves module encapsulation and leads to
more maintainable code. See the \code{modularized_example} for a complete example of this pattern.
}
\examples{
# This example is for a single-file app.
# For modular apps, please see the "Using linkeR with Modules" vignette.
if (interactive()) {
  library(shiny)
  library(leaflet)
  library(DT)

  # Sample data
  sample_data <- data.frame(
    id = 1:10,
    name = paste("Location", 1:10),
    latitude = runif(10, 40.7, 40.8),
    longitude = runif(10, -111.95, -111.85),
    value = round(runif(10, 100, 1000))
  )

  ui <- fluidPage(
    titlePanel("linkeR Example"),
    fluidRow(
      column(6, leafletOutput("my_map")),
      column(6, DTOutput("my_table"))
    )
  )

  server <- function(input, output, session) {
    my_data <- reactive({
      sample_data
    })

    output$my_map <- renderLeaflet({
      leaflet(my_data()) \%>\%
        addTiles() \%>\%
        addMarkers(
          lng = ~longitude,
          lat = ~latitude,
          layerId = ~id,
          popup = ~name
        )
    })

    output$my_table <- renderDT({
      datatable(my_data()[, c("name", "value")], selection = "single")
    })

    link_plots(
      session,
      my_map = my_data,
      my_table = my_data,
      shared_id_column = "id"
    )
  }

  shinyApp(ui, server)
}
}
