#' Print method for 'loo' objects
#'
#' @export
#' @param x A list with class \code{'loo'} (as returned by \code{\link{loo}} or
#'   \code{\link{waic}}).
#' @param digits An integer passed to \code{\link[base]{round}}.
#' @param plot_k Logical. If \code{TRUE} the estimates of the Pareto shape
#'   parameter \eqn{k} are plotted. Ignored if \code{x} was generated by
#'   \code{\link{waic}}. To just plot \eqn{k} without printing use the
#'   \code{\link[=pareto-k-diagnostic]{plot method}}.
#' @param ... Arguments passed to \code{\link{plot.loo}} if \code{plot_k} is
#'   \code{TRUE}.
#'
#' @return \code{x}, invisibly.
#'
#' @seealso \code{\link{pareto-k-diagnostic}}
#'
print.loo <- function(x, digits = 1, plot_k = FALSE, ...) {
  lldims <- paste(attr(x, "log_lik_dim"), collapse = " by ")
  cat("Computed from", lldims, "log-likelihood matrix\n\n")
  z <- x[-grep("pointwise|pareto_k", names(x))]
  uz <- unlist(z)
  nms <- names(uz)
  ses <- grepl("se", nms)
  out <- data.frame(Estimate = uz[!ses], SE = uz[ses])
  print(.fr(out, digits), quote = FALSE)
  if ("p_waic" %in% colnames(x[["pointwise"]])) {
    pwaic_warnings(x$pointwise[, "p_waic"], digits)
  } else if ("pareto_k" %in% names(x)) {
    print(pareto_k_table(x), digits = digits)
    cat(.k_help())
    if (plot_k)
      plot(x, ...)
  }
  invisible(x)
}
