#' Provide pre-extracted names for Provider of Service file, years 2000-2010
#' 
#' @param year A year in the range 2000-2010
#'   
#' @return A list of names for the POS dataset in the year specified, in the 
#'   order that the raw data lists them
#'   
#'   This function returns the results of running \code{pos_names_extract} on 
#'   the layout and raw data files for the chosen year. For years 2000-2010, the
#'   raw data had unhelpful, generic, sequentially numbered variable names.
#'   These can be calculated fresh by calling \code{pos_names_extract} on the
#'   imported dataframe and its corresponding layout .txt file. Alternately,
#'   this function returns the names compiled by that function for years
#'   2000-2010, saving the user a step.
#'   
#' @examples
#' 
#' # get the list
#' pos_names(2005)
#' pos_names(2010)
#' 
#' @export

pos_names <- function(year) {
  if (year == 2000) {
    return(
      c("CATEGORY_SUBTYPE_IND", "CATEGORY", "CHOW_CNT", "CHOW_DT", 
        "CITY", "COMPL_ACCEPT_PLAN_COR", "STATUS_COMPL", "SSA_COUNTY", 
        "CROSS_REF_PROV_NUM", "FMS_SURVEY_DT_1", "SURVEY_DT_1", "ELIG_CD", 
        "FACILITY_NAME", "INTER_CARRIER_NUM", "PARTCI_DT", "PRIOR_CHOW_DT", 
        "PRIOR_INTER_CARRIER_NUM", "PROV_NUM", "RECORD_TYPE", "REGION", 
        "SKELETON_IND", "STATE_ABBREV", "SSA_STATE", "STATE_REGION_CD", 
        "STREET_ADDRESS", "PHONE_NUM", "TERM_CD_1", "EXP_DT_1", "TYPE_ACTION", 
        "TYPE_CONTROL", "ZIP_CD", "WS_FIPS_STATE", "WS_FIPS_CNTY", "WS_SSA_MSA_CD", 
        "WS_SSA_MSA_SIZE_CD", "ACCRED_EFF_DT", "ACCRED_EXP_DT", "ACCRED_STAT", 
        "ALCOH_DRG_UNIT_BED_SZ", "ALCOH_DRG_UNIT_EFF_DT", "ALCOH_DRG_UNIT_IND", 
        "ALCOH_DRG_UNIT_TERM_CD", "ALCOH_DRG_UNIT_TERM_DT", "NUM_BEDS", 
        "NUM_CERT_BEDS", "NUM_CERT_RN_ANEST", "CLIA_ID_NUM_A", "CLIA_ID_NUM_B", 
        "CLIA_ID_NUM_C", "CLIA_ID_NUM_D", "CLIA_ID_NUM_E", "COMPL_LSC", 
        "COMPL_SCOPE_OF_SERV", "COMPL_TECH_PERSNL", "COMPL_24_HR_RN", 
        "CURRENT_EVER_ACCRED", "CURRENT_EVER_NON_ACCRED", "CURRENT_EVER_SWINGBED", 
        "DT_VALID_SURVEY", "NUM_DIETICIANS", "FISC_YR_END_DT", "NUM_INHAL_THERAPY", 
        "NUM_LPN_LVN", "MED_SCHL_AFF", "MEDICAID_VEND_NUM", "MEETS_1861", 
        "NUM_OCCUP_THERAPISTS", "NUM_OTHER_PERSNL", "PARTICIPATING_CD", 
        "NUM_PHYS_THERAPY", "NUM_PHYS_ASSIST", "OLD_PROV_NUM", "PROG_PARTCI", 
        "PSY_UNIT_BED_SZ", "PSY_UNIT_EFF_DT", "PSY_UNIT_IND", "PSY_UNIT_TERM_CD", 
        "PSY_UNIT_TERM_DT", "OVERRIDE_1", "OVERRIDE_2", "OVERRIDE_3", 
        "NUM_REG_NURS", "NUM_PHARMACIST_REG", "REHAB_UNIT_BED_SZ", "REHAB_UNIT_EFF_DT", 
        "REHAB_UNIT_IND", "REHAB_UNIT_TERM_CD", "REHAB_UNIT_TERM_DT", 
        "RELATED_PROV_NUM", "RES_PGM_APPR_ADA", "RES_PGM_APPR_AMA", "RES_PGM_APPR_AOA", 
        "RES_PGM_APPR_OTHER", "NUM_RESID_PHYS", "SEP_COST_ENTITY_IND", 
        "SP_ACUTE_REN_DIAL", "SP_ALCOH_DRUG", "SP_ANESTH", "SP_BLOOD_BANK", 
        "SP_BURN_UNIT", "SP_CHIROPRATIC", "SP_CORONARY_CARE", "SP_DENTAL", 
        "SP_DIETARY", "SP_EMERG_DEPT", "SP_HOME_CARE_UNIT", "SP_HOSPICE", 
        "SP_INPAT_SURG", "SP_ICU", "SP_LABORATORY_ANATOM", "SP_LABORATORY_CLINIC", 
        "SP_LTC_UNIT", "SP_NEONATAL_NURS", "SP_NUCLEAR_MED", "SP_OBSTETRICS", 
        "SP_OCCUP_THERAPY", "SP_OPEN_HEART_SURG", "SP_OR_ROOMS", "SP_OPTOMETRIC", 
        "SP_ORGAN_BANK", "SP_ORGAN_TRANS", "SP_OUTPAT", "SP_OUTPAT_SURG", 
        "SP_PEDIATRIC", "SP_PHARMACY", "SP_PHYSICAL_THERAPY", "SP_POSTOP_REC_RM", 
        "SP_PSYCHIATRIC", "SP_RADIOLOGY_DIAG", "SP_RADIOLOGY_THERAPY", 
        "SP_REHABIL", "SP_SELF_CARE", "SP_SHOCK_TRAUMA", "SP_SOCIAL", 
        "SP_SPEECH_PATH", "SWINGBED_IND", "SWINGBED_SIZE_CD", "TYPE_FACILITY", 
        "NON_PARTICIPATING_TYPE", "NUM_ACT_THER_CONTRACT", "NUM_ACT_THER_FULL_TIME", 
        "NUM_ACT_THER_PART_TIME", "NUM_ADMN_CONTRACT", "NUM_ADMN_FULL_TIME", 
        "NUM_ADMN_PART_TIME", "ADMIN_SUSP_DT", "NUM_T18_SNF_BEDS", "NUM_T19_SNF_BEDS", 
        "NUM_T1819_SNF_BEDS", "NUM_NURSE_AID_CONTRACT", "NUM_NURSE_AID_FULL_TIME", 
        "NUM_NURSE_AID_PART_TIME", "CHRISTIAN_SCIENCE_IND", "COMPL_BEDS_PER_ROOM", 
        "COMPL_PATIENT_ROOM_SZ", "COMPL_7_DAY_RN", "NUM_DENTIST_CONTRACT", 
        "NUM_DENTIST_FULL_TIME", "NUM_DENTIST_PART_TIME", "NUM_DIET_CONTRACT", 
        "NUM_DIET_FULL_TIME", "NUM_DIET_PART_TIME", "EXPER_RESEARCH", 
        "NUM_FOOD_SRV_CONTRACT", "NUM_FOOD_SRV_FULL_TIME", "NUM_FOOD_SRV_PART_TIME", 
        "NUM_HOUSE_CONTRACT", "NUM_HOUSE_FULL_TIME", "NUM_HOUSE_PART_TIME", 
        "NUM_VOC_NURSE_CONTRACT", "NUM_VOC_NURSE_FULL_TIME", "NUM_VOC_NURSE_PART_TIME", 
        "LTC_CROSS_REF_PROV_NUM", "NUM_MED_CONTRACT", "NUM_MED_FULL_TIME", 
        "NUM_MED_PART_TIME", "NUM_MED_AID_CONTRACT", "NUM_MED_AID_FULL_TIME", 
        "NUM_MED_AID_PART_TIME", "NUM_MEN_HLTH_CONTRACT", "NUM_MEN_HLTH_FULL_TIME", 
        "NUM_MEN_HLTH_PART_TIME", "NAME_MULT_FACL_ORG", "MULT_FACL_ORG", 
        "NUM_AID_TRNG_CONTRACT", "NUM_AID_TRNG_FULL_TIME", "NUM_AID_TRNG_PART_TIME", 
        "NUM_NURSE_ADM_CONTRACT", "NUM_NURSE_ADM_FULL_TIME", "NUM_NURSE_ADM_PART_TIME", 
        "NUM_OCC_AID_CONTRACT", "NUM_OCC_AID_FULL_TIME", "NUM_OCC_AID_PART_TIME", 
        "NUM_OCC_ASST_CONTRACT", "NUM_OCC_ASST_FULL_TIME", "NUM_OCC_ASST_PART_TIME", 
        "NUM_OCC_THER_CONTRACT", "NUM_OCC_THER_FULL_TIME", "NUM_OCC_THER_PART_TIME", 
        "ORG_FAMILY_GRP", "ORG_RESID_GRP", "NUM_OTH_CONTRACT", "NUM_OTH_FULL_TIME", 
        "NUM_OTH_PART_TIME", "NUM_OTH_ACT_CONTRACT", "NUM_OTH_ACT_FULL_TIME", 
        "NUM_OTH_ACT_PART_TIME", "NUM_OTH_PHY_CONTRACT", "NUM_OTH_PHY_FULL_TIME", 
        "NUM_OTH_PHY_PART_TIME", "NUM_OTH_SOC_CONTRACT", "NUM_OTH_SOC_FULL_TIME", 
        "NUM_OTH_SOC_PART_TIME", "NUM_PHAR_CONTRACT", "NUM_PHAR_FULL_TIME", 
        "NUM_PHAR_PART_TIME", "NUM_THER_ASST_CONTRACT", "NUM_THER_ASST_FULL_TIME", 
        "NUM_THER_ASST_PART_TIME", "NUM_THER_CONTRACT", "NUM_THER_FULL_TIME", 
        "NUM_THER_PART_TIME", "NUM_THER_AID_CONTRACT", "NUM_THER_AID_FULL_TIME", 
        "NUM_THER_AID_PART_TIME", "NUM_PHYS_EXT_CONTRACT", "NUM_PHYS_EXT_FULL_TIME", 
        "NUM_PHYS_EXT_PART_TIME", "NUM_POD_CONTRACT", "NUM_POD_FULL_TIME", 
        "NUM_POD_PART_TIME", "PRIOR_ADMIN_SUSP_DT", "PRIOR_RESC_SUSP_DT", 
        "PROV_BASED_FACILITY", "NUM_REG_NURSE_CONTRACT", "NUM_REG_NURSE_FULL_TIME", 
        "NUM_REG_NURSE_PART_TIME", "RESC_SUSP_DT", "NUM_RN_DON_CONTRACT", 
        "NUM_RN_DON_FULL_TIME", "NUM_RN_DON_PART_TIME", "NUM_SOCIAL_CONTRACT", 
        "NUM_SOCIAL_FULL_TIME", "NUM_SOCIAL_PART_TIME", "NUM_AIDS_BEDS", 
        "NUM_ALZHEIMERS_BEDS", "NUM_DIAL_BEDS", "NUM_DIS_CHILD_BEDS", 
        "NUM_HEAD_TRAUMA_BEDS", "NUM_HOSPICE_BEDS", "NUM_HUNTING_DIS_BEDS", 
        "NUM_SPEC_REHAB_BEDS", "NUM_VENT_RESP_BEDS", "NUM_SPCH_PATH_CONTRACT", 
        "NUM_SPCH_PATH_FULL_TIME", "NUM_SPCH_PATH_PART_TIME", "SP_ACT_THER_OFF_RES", 
        "SP_ACT_THER_ON_NON_RES", "SP_ACT_THER_ON_RES", "SP_ADM_BLOOD_OFF_RES", 
        "SP_ADM_BLOOD_ON_NON_RES", "SP_ADM_BLOOD_ON_RES", "SP_CLIN_LAB_OFF_RES", 
        "SP_CLIN_LAB_ON_NON_RES", "SP_CLIN_LAB_ON_RES", "SP_DENTAL_OFF_RES", 
        "SP_DENTAL_ON_NON_RES", "SP_DENTAL_ON_RES", "SP_DIETARY_OFF_RES", 
        "SP_DIETARY_ON_NON_RES", "SP_DIETARY_ON_RES", "SP_HOUSE_KP_ON_NON_RES", 
        "SP_HOUSE_KP_OFF_RES", "SP_HOUSE_KP_ON_RES", "SP_MEN_HLTH_OFF_RES", 
        "SP_MEN_HLTH_ON_NON_RES", "SP_MEN_HLTH_ON_RES", "SP_NURSING_OFF_RES", 
        "SP_NURSING_ON_NON_RES", "SP_NURSING_ON_RES", "SP_OCC_THER_OFF_RES", 
        "SP_OCC_THER_ON_NON_RES", "SP_OCC_THER_ON_RES", "SP_OTH_ACT_OFF_RES", 
        "SP_OTH_ACT_ON_NON_RES", "SP_OTH_ACT_ON_RES", "SP_OTH_SOC_OFF_RES", 
        "SP_OTH_SOC_ON_NON_RES", "SP_OTH_SOC_ON_RES", "SP_PHARMACY_OFF_RES", 
        "SP_PHARMACY_ON_NON_RES", "SP_PHARMACY_ON_RES", "SP_PHYS_EXT_OFF_RES", 
        "SP_PHYS_EXT_ON_NON_RES", "SP_PHYS_EXT_ON_RES", "SP_PHYS_THER_OFF_RES", 
        "SP_PHYS_THER_ON_NON_RES", "SP_PHYS_THER_ON_RES", "SP_PHYS_OFF_RES", 
        "SP_PHYS_ON_NON_RES", "SP_PHYS_ON_RES", "SP_PODIATRY_OFF_RES", 
        "SP_PODIATRY_ON_NON_RES", "SP_PODIATRY_ON_RES", "SP_MED_SOC_OFF_RES", 
        "SP_MED_SOC_ON_NON_RES", "SP_MED_SOC_ON_RES", "SP_SPEECH_PH_OFF_RES", 
        "SP_SPEECH_PH_ON_NON_RES", "SP_SPEECH_PH_ON_RES", "SP_THER_REC_OFF_RES", 
        "SP_THER_REC_ON_NON_RES", "SP_THER_REC_ON_RES", "SP_VOC_GUID_OFF_RES", 
        "SP_VOC_GUID_ON_NON_RES", "SP_VOC_GUID_ON_RES", "SP_DIAG_XRAY_OFF_RES", 
        "SP_DIAG_XRAY_ON_NON_RES", "SP_DIAG_XRAY_ON_RES", "NUM_THER_REC_CONTRACT", 
        "NUM_THER_REC_FULL_TIME", "NUM_THER_REC_PART_TIME", "HHA_PROVIDES_DIRECT", 
        "OPERS_BRANCHES", "NUM_BRANCHES", "CHOW_IND", "HHA_QUAL_FOR_OPT", 
        "NUM_HOME_HEALTH_AIDES", "MEDICARE_CERT_HOSPICE", "HOSPICE_PROV_NUM", 
        "MEDICAID_CARE_VEND_NUM", "NUM_SOCIAL_WRKS", "NUM_SPEECH_PATH_AUDIO", 
        "SP_APPLIANCE_EQUIP", "SP_HH_AIDE_HOMEMAKER", "SP_INTERNS_RESIDENTS", 
        "SP_MEDICAL_SOCIAL", "SP_NURSING", "SP_NUTRITION_GUIDANCE", "SP_OTHER", 
        "SP_SPEECH_THERAPY", "SP_VOCAT_GUIDANCE", "SUBUNIT_IND", "OPERS_SUBUNITS", 
        "NUM_SUBUNITS", "SURETY_BOND_IND", "QUAL_OF_DIRECTOR", "NUM_AS_RADIO_TECH", 
        "NUM_BS_BA_RAD_TECH", "NUM_2YR_RADIO_TECH", "PROVIDES_OCCUP_THERAPY", 
        "NUM_PHY_THER_ARGNM", "NUM_PHYS_THERAPISTS", "NUM_SPEECH_PATH_AR", 
        "NUM_SPEECH_PATH", "SP_SPEECH_AND_PATH", "NETWORK_NUM", "NUM_PATIENT_TUE_SHIFT_4", 
        "NUM_STATIONS_HEMO", "TOT_STATIONS", "NUM_ICF_MR_BEDS", "NUM_DCARE_PERSNL", 
        "LTC_AGREE_BEGIN_DT", "LTC_AGREE_END_DT", "LTC_AGREE_EXT_DT", 
        "PRIOR_LTC_END_DT", "PRIOR_LTC_EXT_DT", "TOT_EMPLOYEES", "FED_PROG_SUPPORT", 
        "NUM_NURSE_PRACT", "PARENT_PROV_NUM", "NUM_PHYS", "TITL_FED_PROGR", 
        "PARTIC_OPT_SP", "SP_OCCUP_THERAPY_2", "SP_OCCUP_THERAPY_3", 
        "SP_ORTHOTIC_PROSTHET", "SP_ORTHOTIC_PROSTHET_2", "SP_ORTHOTIC_PROSTHET_3", 
        "SP_PHYSICAL_THERAPY_2", "SP_PHYSICAL_THERAPY_3", "SP_PHYSICIAN", 
        "SP_PHYSICIAN_2", "SP_PHYSICIAN_3", "SP_PSYCHOLOGICAL", "SP_PSYCHOLOGICAL_2", 
        "SP_PSYCHOLOGICAL_3", "SP_RESP_CARE", "SP_RESP_CARE_2", "SP_RESP_CARE_3", 
        "SP_SOCIAL_2", "SP_SOCIAL_3", "SP_SPEECH_PATH_2", "SP_SPEECH_PATH_3", 
        "DT_SERVICE_BEGAN", "FREE_STAND_IND", "HOSP_BASED_IND", "NUM_OPERATING_ROOMS", 
        "SP_CARDIOVASCULAR", "SP_FOOT", "SP_GENERAL", "SP_NEUROLOGICAL", 
        "SP_OBSTETR_GYNECOL", "SP_OPTHAMOLOGY_SURG", "SP_ORAL", "SP_ORTHOPEDIC", 
        "SP_OTOLARYRGOLOGY", "SP_PLASTIC", "SP_THORACIC", "SP_UROLOGY", 
        "SP_EKG", "SP_LABORATORY", "SP_RADIOLOGY", "ACUTE_RESPITE", "NUM_STAFF_COUNSL", 
        "NUM_VOL_COUNSL", "NUM_VOL_HHA", "NUM_HOMEMAKERS", "NUM_VOL_HOMEMKR", 
        "NUM_VOL_LPN_LVN", "NUM_MED_SOCIAL_WRKS", "NUM_VOL_SOC_WORK", 
        "NUM_VOL_PHYS", "NUM_VOL_REG_NURS", "SP_COUNSELING", "SP_HOME_HEALTH_AIDE", 
        "SP_HOMEMAKER", "SP_MEDICAL_SUPPLIES", "SP_SHORT_TERM_INCARE", 
        "NUM_OTHER_VOLS", "TOT_VOLS", "FED_FUNDED_FFHC", "APPROVED_RHC_PROV_NUM", 
        "APPROVED_MEDICARE_RHC", "ACCRED_AABB_IND", "ACCRED_AOA_IND", 
        "ACCRED_ASHI_IND", "ACCRED_CAP_IND", "ACCRED_COLA_IND", "ACCRED_JCAHO_IND", 
        "ACCRED_AABB_DT", "ACCRED_AOA_DT", "ACCRED_ASHI_DT", "ACCRED_CAP_DT", 
        "ACCRED_COLA_DT", "ACCRED_JCAHO_DT", "ACCRED_AABB_MATCH_IND", 
        "ACCRED_AOA_MATCH_IND", "ACCRED_ASHI_MATCH_IND", "ACCRED_CAP_MATCH_IND", 
        "ACCRED_COLA_MATCH_IND", "ACCRED_JCAHO_MATCH_IND", "AFFIL_PROV_NUM_1", 
        "AFFIL_PROV_NUM_2", "AFFIL_PROV_NUM_3", "AFFIL_PROV_NUM_4", "AFFIL_PROV_NUM_5", 
        "AFFIL_PROV_NUM_6", "AFFIL_PROV_NUM_7", "AFFIL_PROV_NUM_8", "APPL_ACCR_ANN_TEST_VOL", 
        "APPL_ACCRED_SCHED_CD", "APPL_RECEIVED_DT", "APPL_TOT_ANN_TEST_VOL", 
        "TYPE_APPLICATION", "CERT_TYPE_CD_1", "CERT_MAILED_DT_1", "EFF_DT_1", 
        "CLIA_MEDICARE_NUM", "FAX_NUM", "CLIA_LAB_CLASS_CD", "MULTI_SITE_IND", 
        "NON_PROFIT_IND", "TOT_NUM_NON_WAIVED_IND", "NUM_CLIN_CONSULT", 
        "NUM_DIRECTORS", "NUM_GEN_SUPER", "TOT_NUM_SITES", "NUM_AFFIL_LABS", 
        "NUM_TECH_CONSULT", "NUM_TECH_SUPER", "NUM_TEST_PERSONNEL", "TOT_NUM_WAIVED_IND", 
        "CLIA67_IND", "SHARED_LAB_XREF_NUM", "SHARED_LAB_IND", "SURV_CERT_SCHED_CD", 
        "SURV_COMPL_SCHED_CD", "SURV_TOT_ANN_TEST_VOL", "TERM_CD", "TOT_ANN_TEST_VOL_WAIVED"
      )
    )
  } else if (year == 2001) {
    return(
      c("CATEGORY_SUBTYPE_IND", "CATEGORY", "CHOW_CNT", "CHOW_DT", 
        "CITY", "COMPL_ACCEPT_PLAN_COR", "STATUS_COMPL", "SSA_COUNTY", 
        "CROSS_REF_PROV_NUM", "FMS_SURVEY_DT_1", "SURVEY_DT_1", "ELIG_CD", 
        "FACILITY_NAME", "INTER_CARRIER_NUM", "PARTCI_DT", "PRIOR_CHOW_DT", 
        "PRIOR_INTER_CARRIER_NUM", "PROV_NUM", "RECORD_TYPE", "REGION", 
        "SKELETON_IND", "STATE_ABBREV", "SSA_STATE", "STATE_REGION_CD", 
        "STREET_ADDRESS", "PHONE_NUM", "TERM_CD_1", "EXP_DT_1", "TYPE_ACTION", 
        "TYPE_CONTROL", "ZIP_CD", "WS_FIPS_STATE", "WS_FIPS_CNTY", "WS_SSA_MSA_CD", 
        "WS_SSA_MSA_SIZE_CD", "ACCRED_EFF_DT", "ACCRED_EXP_DT", "ACCRED_STAT", 
        "ALCOH_DRG_UNIT_BED_SZ", "ALCOH_DRG_UNIT_EFF_DT", "ALCOH_DRG_UNIT_IND", 
        "ALCOH_DRG_UNIT_TERM_CD", "ALCOH_DRG_UNIT_TERM_DT", "NUM_BEDS", 
        "NUM_CERT_BEDS", "NUM_CERT_RN_ANEST", "CLIA_ID_NUM_A", "CLIA_ID_NUM_B", 
        "CLIA_ID_NUM_C", "CLIA_ID_NUM_D", "CLIA_ID_NUM_E", "COMPL_LSC", 
        "COMPL_SCOPE_OF_SERV", "COMPL_TECH_PERSNL", "COMPL_24_HR_RN", 
        "CURRENT_EVER_ACCRED", "CURRENT_EVER_NON_ACCRED", "CURRENT_EVER_SWINGBED", 
        "DT_VALID_SURVEY", "NUM_DIETICIANS", "FISC_YR_END_DT", "NUM_INHAL_THERAPY", 
        "NUM_LPN_LVN", "MED_SCHL_AFF", "MEDICAID_VEND_NUM", "MEETS_1861", 
        "NUM_OCCUP_THERAPISTS", "NUM_OTHER_PERSNL", "PARTICIPATING_CD", 
        "NUM_PHYS_THERAPY", "NUM_PHYS_ASSIST", "OLD_PROV_NUM", "PROG_PARTCI", 
        "PSY_UNIT_BED_SZ", "PSY_UNIT_EFF_DT", "PSY_UNIT_IND", "PSY_UNIT_TERM_CD", 
        "PSY_UNIT_TERM_DT", "OVERRIDE_1", "OVERRIDE_2", "OVERRIDE_3", 
        "NUM_REG_NURS", "NUM_PHARMACIST_REG", "REHAB_UNIT_BED_SZ", "REHAB_UNIT_EFF_DT", 
        "REHAB_UNIT_IND", "REHAB_UNIT_TERM_CD", "REHAB_UNIT_TERM_DT", 
        "RELATED_PROV_NUM", "RES_PGM_APPR_ADA", "RES_PGM_APPR_AMA", "RES_PGM_APPR_AOA", 
        "RES_PGM_APPR_OTHER", "NUM_RESID_PHYS", "SEP_COST_ENTITY_IND", 
        "SP_ACUTE_REN_DIAL", "SP_ALCOH_DRUG", "SP_ANESTH", "SP_BLOOD_BANK", 
        "SP_BURN_UNIT", "SP_CHIROPRATIC", "SP_CORONARY_CARE", "SP_DENTAL", 
        "SP_DIETARY", "SP_EMERG_DEPT", "SP_HOME_CARE_UNIT", "SP_HOSPICE", 
        "SP_INPAT_SURG", "SP_ICU", "SP_LABORATORY_ANATOM", "SP_LABORATORY_CLINIC", 
        "SP_LTC_UNIT", "SP_NEONATAL_NURS", "SP_NUCLEAR_MED", "SP_OBSTETRICS", 
        "SP_OCCUP_THERAPY", "SP_OPEN_HEART_SURG", "SP_OR_ROOMS", "SP_OPTOMETRIC", 
        "SP_ORGAN_BANK", "SP_ORGAN_TRANS", "SP_OUTPAT", "SP_OUTPAT_SURG", 
        "SP_PEDIATRIC", "SP_PHARMACY", "SP_PHYSICAL_THERAPY", "SP_POSTOP_REC_RM", 
        "SP_PSYCHIATRIC", "SP_RADIOLOGY_DIAG", "SP_RADIOLOGY_THERAPY", 
        "SP_REHABIL", "SP_SELF_CARE", "SP_SHOCK_TRAUMA", "SP_SOCIAL", 
        "SP_SPEECH_PATH", "SWINGBED_IND", "SWINGBED_SIZE_CD", "TYPE_FACILITY", 
        "NON_PARTICIPATING_TYPE", "NUM_ACT_THER_CONTRACT", "NUM_ACT_THER_FULL_TIME", 
        "NUM_ACT_THER_PART_TIME", "NUM_ADMN_CONTRACT", "NUM_ADMN_FULL_TIME", 
        "NUM_ADMN_PART_TIME", "NUM_T18_SNF_BEDS", "NUM_T19_SNF_BEDS", 
        "NUM_T1819_SNF_BEDS", "NUM_NURSE_AID_CONTRACT", "NUM_NURSE_AID_FULL_TIME", 
        "NUM_NURSE_AID_PART_TIME", "CHRISTIAN_SCIENCE_IND", "COMPL_BEDS_PER_ROOM", 
        "COMPL_PATIENT_ROOM_SZ", "COMPL_7_DAY_RN", "NUM_DENTIST_CONTRACT", 
        "NUM_DENTIST_FULL_TIME", "NUM_DENTIST_PART_TIME", "NUM_DIET_CONTRACT", 
        "NUM_DIET_FULL_TIME", "NUM_DIET_PART_TIME", "EXPER_RESEARCH", 
        "NUM_FOOD_SRV_CONTRACT", "NUM_FOOD_SRV_FULL_TIME", "NUM_FOOD_SRV_PART_TIME", 
        "NUM_HOUSE_CONTRACT", "NUM_HOUSE_FULL_TIME", "NUM_HOUSE_PART_TIME", 
        "NUM_VOC_NURSE_CONTRACT", "NUM_VOC_NURSE_FULL_TIME", "NUM_VOC_NURSE_PART_TIME", 
        "LTC_CROSS_REF_PROV_NUM", "NUM_MED_CONTRACT", "NUM_MED_FULL_TIME", 
        "NUM_MED_PART_TIME", "NUM_MED_AID_CONTRACT", "NUM_MED_AID_FULL_TIME", 
        "NUM_MED_AID_PART_TIME", "NUM_MEN_HLTH_CONTRACT", "NUM_MEN_HLTH_FULL_TIME", 
        "NUM_MEN_HLTH_PART_TIME", "NAME_MULT_FACL_ORG", "MULT_FACL_ORG", 
        "NUM_AID_TRNG_CONTRACT", "NUM_AID_TRNG_FULL_TIME", "NUM_AID_TRNG_PART_TIME", 
        "NUM_NURSE_ADM_CONTRACT", "NUM_NURSE_ADM_FULL_TIME", "NUM_NURSE_ADM_PART_TIME", 
        "NUM_OCC_AID_CONTRACT", "NUM_OCC_AID_FULL_TIME", "NUM_OCC_AID_PART_TIME", 
        "NUM_OCC_ASST_CONTRACT", "NUM_OCC_ASST_FULL_TIME", "NUM_OCC_ASST_PART_TIME", 
        "NUM_OCC_THER_CONTRACT", "NUM_OCC_THER_FULL_TIME", "NUM_OCC_THER_PART_TIME", 
        "ORG_FAMILY_GRP", "ORG_RESID_GRP", "NUM_OTH_CONTRACT", "NUM_OTH_FULL_TIME", 
        "NUM_OTH_PART_TIME", "NUM_OTH_ACT_CONTRACT", "NUM_OTH_ACT_FULL_TIME", 
        "NUM_OTH_ACT_PART_TIME", "NUM_OTH_PHY_CONTRACT", "NUM_OTH_PHY_FULL_TIME", 
        "NUM_OTH_PHY_PART_TIME", "NUM_OTH_SOC_CONTRACT", "NUM_OTH_SOC_FULL_TIME", 
        "NUM_OTH_SOC_PART_TIME", "NUM_PHAR_CONTRACT", "NUM_PHAR_FULL_TIME", 
        "NUM_PHAR_PART_TIME", "NUM_THER_ASST_CONTRACT", "NUM_THER_ASST_FULL_TIME", 
        "NUM_THER_ASST_PART_TIME", "NUM_THER_CONTRACT", "NUM_THER_FULL_TIME", 
        "NUM_THER_PART_TIME", "NUM_THER_AID_CONTRACT", "NUM_THER_AID_FULL_TIME", 
        "NUM_THER_AID_PART_TIME", "NUM_PHYS_EXT_CONTRACT", "NUM_PHYS_EXT_FULL_TIME", 
        "NUM_PHYS_EXT_PART_TIME", "NUM_POD_CONTRACT", "NUM_POD_FULL_TIME", 
        "NUM_POD_PART_TIME", "PRIOR_ADMIN_SUSP_DT", "PRIOR_RESC_SUSP_DT", 
        "PROV_BASED_FACILITY", "NUM_REG_NURSE_CONTRACT", "NUM_REG_NURSE_FULL_TIME", 
        "NUM_REG_NURSE_PART_TIME", "RESC_SUSP_DT", "NUM_RN_DON_CONTRACT", 
        "NUM_RN_DON_FULL_TIME", "NUM_RN_DON_PART_TIME", "NUM_SOCIAL_CONTRACT", 
        "NUM_SOCIAL_FULL_TIME", "NUM_SOCIAL_PART_TIME", "NUM_AIDS_BEDS", 
        "NUM_ALZHEIMERS_BEDS", "NUM_DIAL_BEDS", "NUM_DIS_CHILD_BEDS", 
        "NUM_HEAD_TRAUMA_BEDS", "NUM_HOSPICE_BEDS", "NUM_HUNTING_DIS_BEDS", 
        "NUM_SPEC_REHAB_BEDS", "NUM_VENT_RESP_BEDS", "NUM_SPCH_PATH_CONTRACT", 
        "NUM_SPCH_PATH_FULL_TIME", "NUM_SPCH_PATH_PART_TIME", "SP_ACT_THER_OFF_RES", 
        "SP_ACT_THER_ON_NON_RES", "SP_ACT_THER_ON_RES", "SP_ADM_BLOOD_OFF_RES", 
        "SP_ADM_BLOOD_ON_NON_RES", "SP_ADM_BLOOD_ON_RES", "SP_CLIN_LAB_OFF_RES", 
        "SP_CLIN_LAB_ON_NON_RES", "SP_CLIN_LAB_ON_RES", "SP_DENTAL_OFF_RES", 
        "SP_DENTAL_ON_NON_RES", "SP_DENTAL_ON_RES", "SP_DIETARY_OFF_RES", 
        "SP_DIETARY_ON_NON_RES", "SP_DIETARY_ON_RES", "SP_HOUSE_KP_ON_NON_RES", 
        "SP_HOUSE_KP_OFF_RES", "SP_HOUSE_KP_ON_RES", "SP_MEN_HLTH_OFF_RES", 
        "SP_MEN_HLTH_ON_NON_RES", "SP_MEN_HLTH_ON_RES", "SP_NURSING_OFF_RES", 
        "SP_NURSING_ON_NON_RES", "SP_NURSING_ON_RES", "SP_OCC_THER_OFF_RES", 
        "SP_OCC_THER_ON_NON_RES", "SP_OCC_THER_ON_RES", "SP_OTH_ACT_OFF_RES", 
        "SP_OTH_ACT_ON_NON_RES", "SP_OTH_ACT_ON_RES", "SP_OTH_SOC_OFF_RES", 
        "SP_OTH_SOC_ON_NON_RES", "SP_OTH_SOC_ON_RES", "SP_PHARMACY_OFF_RES", 
        "SP_PHARMACY_ON_NON_RES", "SP_PHARMACY_ON_RES", "SP_PHYS_EXT_OFF_RES", 
        "SP_PHYS_EXT_ON_NON_RES", "SP_PHYS_EXT_ON_RES", "SP_PHYS_THER_OFF_RES", 
        "SP_PHYS_THER_ON_NON_RES", "SP_PHYS_THER_ON_RES", "SP_PHYS_OFF_RES", 
        "SP_PHYS_ON_NON_RES", "SP_PHYS_ON_RES", "SP_PODIATRY_OFF_RES", 
        "SP_PODIATRY_ON_NON_RES", "SP_PODIATRY_ON_RES", "SP_MED_SOC_OFF_RES", 
        "SP_MED_SOC_ON_NON_RES", "SP_MED_SOC_ON_RES", "SP_SPEECH_PH_OFF_RES", 
        "SP_SPEECH_PH_ON_NON_RES", "SP_SPEECH_PH_ON_RES", "SP_THER_REC_OFF_RES", 
        "SP_THER_REC_ON_NON_RES", "SP_THER_REC_ON_RES", "SP_VOC_GUID_OFF_RES", 
        "SP_VOC_GUID_ON_NON_RES", "SP_VOC_GUID_ON_RES", "SP_DIAG_XRAY_OFF_RES", 
        "SP_DIAG_XRAY_ON_NON_RES", "SP_DIAG_XRAY_ON_RES", "NUM_THER_REC_CONTRACT", 
        "NUM_THER_REC_FULL_TIME", "NUM_THER_REC_PART_TIME", "HHA_PROVIDES_DIRECT", 
        "OPERS_BRANCHES", "NUM_BRANCHES", "CHOW_IND", "HHA_QUAL_FOR_OPT", 
        "NUM_HOME_HEALTH_AIDES", "MEDICARE_CERT_HOSPICE", "HOSPICE_PROV_NUM", 
        "MEDICAID_CARE_VEND_NUM", "NUM_SOCIAL_WRKS", "NUM_SPEECH_PATH_AUDIO", 
        "SP_APPLIANCE_EQUIP", "SP_HH_AIDE_HOMEMAKER", "SP_INTERNS_RESIDENTS", 
        "SP_MEDICAL_SOCIAL", "SP_NURSING", "SP_NUTRITION_GUIDANCE", "SP_OTHER", 
        "SP_SPEECH_THERAPY", "SP_VOCAT_GUIDANCE", "SUBUNIT_IND", "OPERS_SUBUNITS", 
        "NUM_SUBUNITS", "SURETY_BOND_IND", "QUAL_OF_DIRECTOR", "NUM_AS_RADIO_TECH", 
        "NUM_BS_BA_RAD_TECH", "NUM_2YR_RADIO_TECH", "PROVIDES_OCCUP_THERAPY", 
        "NUM_PHY_THER_ARGNM", "NUM_PHYS_THERAPISTS", "NUM_SPEECH_PATH_AR", 
        "NUM_SPEECH_PATH", "SP_SPEECH_AND_PATH", "NETWORK_NUM", "NUM_PATIENT_TUE_SHIFT_4", 
        "NUM_STATIONS_HEMO", "TOT_STATIONS", "ADMIN_SUSP_DT", "NUM_ICF_MR_BEDS", 
        "NUM_DCARE_PERSNL", "LTC_AGREE_BEGIN_DT", "LTC_AGREE_END_DT", 
        "LTC_AGREE_EXT_DT", "PRIOR_LTC_END_DT", "PRIOR_LTC_EXT_DT", "TOT_EMPLOYEES", 
        "FED_PROG_SUPPORT", "NUM_NURSE_PRACT", "PARENT_PROV_NUM", "NUM_PHYS", 
        "TITL_FED_PROGR", "PARTIC_OPT_SP", "SP_OCCUP_THERAPY_2", "SP_OCCUP_THERAPY_3", 
        "SP_ORTHOTIC_PROSTHET", "SP_ORTHOTIC_PROSTHET_2", "SP_ORTHOTIC_PROSTHET_3", 
        "SP_PHYSICAL_THERAPY_2", "SP_PHYSICAL_THERAPY_3", "SP_PHYSICIAN", 
        "SP_PHYSICIAN_2", "SP_PHYSICIAN_3", "SP_PSYCHOLOGICAL", "SP_PSYCHOLOGICAL_2", 
        "SP_PSYCHOLOGICAL_3", "SP_RESP_CARE", "SP_RESP_CARE_2", "SP_RESP_CARE_3", 
        "SP_SOCIAL_2", "SP_SOCIAL_3", "SP_SPEECH_PATH_2", "SP_SPEECH_PATH_3", 
        "DT_SERVICE_BEGAN", "FREE_STAND_IND", "HOSP_BASED_IND", "NUM_OPERATING_ROOMS", 
        "SP_CARDIOVASCULAR", "SP_FOOT", "SP_GENERAL", "SP_NEUROLOGICAL", 
        "SP_OBSTETR_GYNECOL", "SP_OPTHAMOLOGY_SURG", "SP_ORAL", "SP_ORTHOPEDIC", 
        "SP_OTOLARYRGOLOGY", "SP_PLASTIC", "SP_THORACIC", "SP_UROLOGY", 
        "SP_EKG", "SP_LABORATORY", "SP_RADIOLOGY", "ACUTE_RESPITE", "NUM_STAFF_COUNSL", 
        "NUM_VOL_COUNSL", "NUM_VOL_HHA", "NUM_HOMEMAKERS", "NUM_VOL_HOMEMKR", 
        "NUM_VOL_LPN_LVN", "NUM_MED_SOCIAL_WRKS", "NUM_VOL_SOC_WORK", 
        "NUM_VOL_PHYS", "NUM_VOL_REG_NURS", "SP_COUNSELING", "SP_HOME_HEALTH_AIDE", 
        "SP_HOMEMAKER", "SP_MEDICAL_SUPPLIES", "SP_SHORT_TERM_INCARE", 
        "NUM_OTHER_VOLS", "TOT_VOLS", "FED_FUNDED_FFHC", "APPROVED_RHC_PROV_NUM", 
        "APPROVED_MEDICARE_RHC", "ACCRED_AABB_IND", "ACCRED_AOA_IND", 
        "ACCRED_ASHI_IND", "ACCRED_CAP_IND", "ACCRED_COLA_IND", "ACCRED_JCAHO_IND", 
        "ACCRED_AABB_DT", "ACCRED_AOA_DT", "ACCRED_ASHI_DT", "ACCRED_CAP_DT", 
        "ACCRED_COLA_DT", "ACCRED_JCAHO_DT", "ACCRED_AABB_MATCH_IND", 
        "ACCRED_AOA_MATCH_IND", "ACCRED_ASHI_MATCH_IND", "ACCRED_CAP_MATCH_IND", 
        "ACCRED_COLA_MATCH_IND", "ACCRED_JCAHO_MATCH_IND", "AFFIL_PROV_NUM_1", 
        "AFFIL_PROV_NUM_2", "AFFIL_PROV_NUM_3", "AFFIL_PROV_NUM_4", "AFFIL_PROV_NUM_5", 
        "AFFIL_PROV_NUM_6", "AFFIL_PROV_NUM_7", "AFFIL_PROV_NUM_8", "APPL_ACCR_ANN_TEST_VOL", 
        "APPL_ACCRED_SCHED_CD", "APPL_RECEIVED_DT", "APPL_TOT_ANN_TEST_VOL", 
        "TYPE_APPLICATION", "CERT_TYPE_CD_1", "CERT_MAILED_DT_1", "EFF_DT_1", 
        "CLIA_MEDICARE_NUM", "FAX_NUM", "CLIA_LAB_CLASS_CD", "MULTI_SITE_IND", 
        "NON_PROFIT_IND", "TOT_NUM_NON_WAIVED_IND", "NUM_CLIN_CONSULT", 
        "NUM_DIRECTORS", "NUM_GEN_SUPER", "TOT_NUM_SITES", "NUM_AFFIL_LABS", 
        "NUM_TECH_CONSULT", "NUM_TECH_SUPER", "NUM_TEST_PERSONNEL", "TOT_NUM_WAIVED_IND", 
        "CLIA67_IND", "SHARED_LAB_XREF_NUM", "SHARED_LAB_IND", "SURV_CERT_SCHED_CD", 
        "SURV_COMPL_SCHED_CD", "SURV_TOT_ANN_TEST_VOL", "TERM_CD", "TOT_ANN_TEST_VOL_WAIVED"
      )
    )
  } else if (year == 2002) {
    return(
      c("CATEGORY_SUBTYPE_IND", "CATEGORY", "CHOW_CNT", "CHOW_DT", 
        "CITY", "COMPL_ACCEPT_PLAN_COR", "STATUS_COMPL", "SSA_COUNTY", 
        "CROSS_REF_PROV_NUM", "FMS_SURVEY_DT_1", "SURVEY_DT_1", "ELIG_CD", 
        "FACILITY_NAME", "INTER_CARRIER_NUM", "PARTCI_DT", "PRIOR_CHOW_DT", 
        "PRIOR_INTER_CARRIER_NUM", "PROV_NUM", "RECORD_TYPE", "REGION", 
        "SKELETON_IND", "STATE_ABBREV", "SSA_STATE", "STATE_REGION_CD", 
        "STREET_ADDRESS", "PHONE_NUM", "TERM_CD_1", "EXP_DT_1", "TYPE_ACTION", 
        "TYPE_CONTROL", "ZIP_CD", "WS_FIPS_STATE", "WS_FIPS_CNTY", "WS_SSA_MSA_CD", 
        "WS_SSA_MSA_SIZE_CD", "ACCRED_EFF_DT", "ACCRED_EXP_DT", "ACCRED_STAT", 
        "ALCOH_DRG_UNIT_BED_SZ", "ALCOH_DRG_UNIT_EFF_DT", "ALCOH_DRG_UNIT_IND", 
        "ALCOH_DRG_UNIT_TERM_CD", "ALCOH_DRG_UNIT_TERM_DT", "NUM_BEDS", 
        "NUM_CERT_BEDS", "NUM_CERT_RN_ANEST", "CLIA_ID_NUM_A", "CLIA_ID_NUM_B", 
        "CLIA_ID_NUM_C", "CLIA_ID_NUM_D", "CLIA_ID_NUM_E", "COMPL_LSC", 
        "COMPL_SCOPE_OF_SERV", "COMPL_TECH_PERSNL", "COMPL_24_HR_RN", 
        "CURRENT_EVER_ACCRED", "CURRENT_EVER_NON_ACCRED", "CURRENT_EVER_SWINGBED", 
        "DT_VALID_SURVEY", "NUM_DIETICIANS", "FISC_YR_END_DT", "NUM_INHAL_THERAPY", 
        "NUM_LPN_LVN", "MED_SCHL_AFF", "MEDICAID_VEND_NUM", "MEETS_1861", 
        "NUM_OCCUP_THERAPISTS", "NUM_OTHER_PERSNL", "PARTICIPATING_CD", 
        "NUM_PHYS_THERAPY", "NUM_PHYS_ASSIST", "OLD_PROV_NUM", "PROG_PARTCI", 
        "PSY_UNIT_BED_SZ", "PSY_UNIT_EFF_DT", "PSY_UNIT_IND", "PSY_UNIT_TERM_CD", 
        "PSY_UNIT_TERM_DT", "OVERRIDE_1", "OVERRIDE_2", "OVERRIDE_3", 
        "NUM_REG_NURS", "NUM_PHARMACIST_REG", "REHAB_UNIT_BED_SZ", "REHAB_UNIT_EFF_DT", 
        "REHAB_UNIT_IND", "REHAB_UNIT_TERM_CD", "REHAB_UNIT_TERM_DT", 
        "RELATED_PROV_NUM", "RES_PGM_APPR_ADA", "RES_PGM_APPR_AMA", "RES_PGM_APPR_AOA", 
        "RES_PGM_APPR_OTHER", "NUM_RESID_PHYS", "SEP_COST_ENTITY_IND", 
        "SP_ACUTE_REN_DIAL", "SP_ALCOH_DRUG", "SP_ANESTH", "SP_BLOOD_BANK", 
        "SP_BURN_UNIT", "SP_CHIROPRATIC", "SP_CORONARY_CARE", "SP_DENTAL", 
        "SP_DIETARY", "SP_EMERG_DEPT", "SP_HOME_CARE_UNIT", "SP_HOSPICE", 
        "SP_INPAT_SURG", "SP_ICU", "SP_LABORATORY_ANATOM", "SP_LABORATORY_CLINIC", 
        "SP_LTC_UNIT", "SP_NEONATAL_NURS", "SP_NUCLEAR_MED", "SP_OBSTETRICS", 
        "SP_OCCUP_THERAPY", "SP_OPEN_HEART_SURG", "SP_OR_ROOMS", "SP_OPTOMETRIC", 
        "SP_ORGAN_BANK", "SP_ORGAN_TRANS", "SP_OUTPAT", "SP_OUTPAT_SURG", 
        "SP_PEDIATRIC", "SP_PHARMACY", "SP_PHYSICAL_THERAPY", "SP_POSTOP_REC_RM", 
        "SP_PSYCHIATRIC", "SP_RADIOLOGY_DIAG", "SP_RADIOLOGY_THERAPY", 
        "SP_REHABIL", "SP_SELF_CARE", "SP_SHOCK_TRAUMA", "SP_SOCIAL", 
        "SP_SPEECH_PATH", "SWINGBED_IND", "SWINGBED_SIZE_CD", "TYPE_FACILITY", 
        "NON_PARTICIPATING_TYPE", "NUM_ACT_THER_CONTRACT", "NUM_ACT_THER_FULL_TIME", 
        "NUM_ACT_THER_PART_TIME", "NUM_ADMN_CONTRACT", "NUM_ADMN_FULL_TIME", 
        "NUM_ADMN_PART_TIME", "NUM_T18_SNF_BEDS", "NUM_T19_SNF_BEDS", 
        "NUM_T1819_SNF_BEDS", "NUM_NURSE_AID_CONTRACT", "NUM_NURSE_AID_FULL_TIME", 
        "NUM_NURSE_AID_PART_TIME", "CHRISTIAN_SCIENCE_IND", "COMPL_BEDS_PER_ROOM", 
        "COMPL_PATIENT_ROOM_SZ", "COMPL_7_DAY_RN", "NUM_DENTIST_CONTRACT", 
        "NUM_DENTIST_FULL_TIME", "NUM_DENTIST_PART_TIME", "NUM_DIET_CONTRACT", 
        "NUM_DIET_FULL_TIME", "NUM_DIET_PART_TIME", "EXPER_RESEARCH", 
        "NUM_FOOD_SRV_CONTRACT", "NUM_FOOD_SRV_FULL_TIME", "NUM_FOOD_SRV_PART_TIME", 
        "NUM_HOUSE_CONTRACT", "NUM_HOUSE_FULL_TIME", "NUM_HOUSE_PART_TIME", 
        "NUM_VOC_NURSE_CONTRACT", "NUM_VOC_NURSE_FULL_TIME", "NUM_VOC_NURSE_PART_TIME", 
        "LTC_CROSS_REF_PROV_NUM", "NUM_MED_CONTRACT", "NUM_MED_FULL_TIME", 
        "NUM_MED_PART_TIME", "NUM_MED_AID_CONTRACT", "NUM_MED_AID_FULL_TIME", 
        "NUM_MED_AID_PART_TIME", "NUM_MEN_HLTH_CONTRACT", "NUM_MEN_HLTH_FULL_TIME", 
        "NUM_MEN_HLTH_PART_TIME", "NAME_MULT_FACL_ORG", "MULT_FACL_ORG", 
        "NUM_AID_TRNG_CONTRACT", "NUM_AID_TRNG_FULL_TIME", "NUM_AID_TRNG_PART_TIME", 
        "NUM_NURSE_ADM_CONTRACT", "NUM_NURSE_ADM_FULL_TIME", "NUM_NURSE_ADM_PART_TIME", 
        "NUM_OCC_AID_CONTRACT", "NUM_OCC_AID_FULL_TIME", "NUM_OCC_AID_PART_TIME", 
        "NUM_OCC_ASST_CONTRACT", "NUM_OCC_ASST_FULL_TIME", "NUM_OCC_ASST_PART_TIME", 
        "NUM_OCC_THER_CONTRACT", "NUM_OCC_THER_FULL_TIME", "NUM_OCC_THER_PART_TIME", 
        "ORG_FAMILY_GRP", "ORG_RESID_GRP", "NUM_OTH_CONTRACT", "NUM_OTH_FULL_TIME", 
        "NUM_OTH_PART_TIME", "NUM_OTH_ACT_CONTRACT", "NUM_OTH_ACT_FULL_TIME", 
        "NUM_OTH_ACT_PART_TIME", "NUM_OTH_PHY_CONTRACT", "NUM_OTH_PHY_FULL_TIME", 
        "NUM_OTH_PHY_PART_TIME", "NUM_OTH_SOC_CONTRACT", "NUM_OTH_SOC_FULL_TIME", 
        "NUM_OTH_SOC_PART_TIME", "NUM_PHAR_CONTRACT", "NUM_PHAR_FULL_TIME", 
        "NUM_PHAR_PART_TIME", "NUM_THER_ASST_CONTRACT", "NUM_THER_ASST_FULL_TIME", 
        "NUM_THER_ASST_PART_TIME", "NUM_THER_CONTRACT", "NUM_THER_FULL_TIME", 
        "NUM_THER_PART_TIME", "NUM_THER_AID_CONTRACT", "NUM_THER_AID_FULL_TIME", 
        "NUM_THER_AID_PART_TIME", "NUM_PHYS_EXT_CONTRACT", "NUM_PHYS_EXT_FULL_TIME", 
        "NUM_PHYS_EXT_PART_TIME", "NUM_POD_CONTRACT", "NUM_POD_FULL_TIME", 
        "NUM_POD_PART_TIME", "PRIOR_ADMIN_SUSP_DT", "PRIOR_RESC_SUSP_DT", 
        "PROV_BASED_FACILITY", "NUM_REG_NURSE_CONTRACT", "NUM_REG_NURSE_FULL_TIME", 
        "NUM_REG_NURSE_PART_TIME", "RESC_SUSP_DT", "NUM_RN_DON_CONTRACT", 
        "NUM_RN_DON_FULL_TIME", "NUM_RN_DON_PART_TIME", "NUM_SOCIAL_CONTRACT", 
        "NUM_SOCIAL_FULL_TIME", "NUM_SOCIAL_PART_TIME", "NUM_AIDS_BEDS", 
        "NUM_ALZHEIMERS_BEDS", "NUM_DIAL_BEDS", "NUM_DIS_CHILD_BEDS", 
        "NUM_HEAD_TRAUMA_BEDS", "NUM_HOSPICE_BEDS", "NUM_HUNTING_DIS_BEDS", 
        "NUM_SPEC_REHAB_BEDS", "NUM_VENT_RESP_BEDS", "NUM_SPCH_PATH_CONTRACT", 
        "NUM_SPCH_PATH_FULL_TIME", "NUM_SPCH_PATH_PART_TIME", "SP_ACT_THER_OFF_RES", 
        "SP_ACT_THER_ON_NON_RES", "SP_ACT_THER_ON_RES", "SP_ADM_BLOOD_OFF_RES", 
        "SP_ADM_BLOOD_ON_NON_RES", "SP_ADM_BLOOD_ON_RES", "SP_CLIN_LAB_OFF_RES", 
        "SP_CLIN_LAB_ON_NON_RES", "SP_CLIN_LAB_ON_RES", "SP_DENTAL_OFF_RES", 
        "SP_DENTAL_ON_NON_RES", "SP_DENTAL_ON_RES", "SP_DIETARY_OFF_RES", 
        "SP_DIETARY_ON_NON_RES", "SP_DIETARY_ON_RES", "SP_HOUSE_KP_ON_NON_RES", 
        "SP_HOUSE_KP_OFF_RES", "SP_HOUSE_KP_ON_RES", "SP_MEN_HLTH_OFF_RES", 
        "SP_MEN_HLTH_ON_NON_RES", "SP_MEN_HLTH_ON_RES", "SP_NURSING_OFF_RES", 
        "SP_NURSING_ON_NON_RES", "SP_NURSING_ON_RES", "SP_OCC_THER_OFF_RES", 
        "SP_OCC_THER_ON_NON_RES", "SP_OCC_THER_ON_RES", "SP_OTH_ACT_OFF_RES", 
        "SP_OTH_ACT_ON_NON_RES", "SP_OTH_ACT_ON_RES", "SP_OTH_SOC_OFF_RES", 
        "SP_OTH_SOC_ON_NON_RES", "SP_OTH_SOC_ON_RES", "SP_PHARMACY_OFF_RES", 
        "SP_PHARMACY_ON_NON_RES", "SP_PHARMACY_ON_RES", "SP_PHYS_EXT_OFF_RES", 
        "SP_PHYS_EXT_ON_NON_RES", "SP_PHYS_EXT_ON_RES", "SP_PHYS_THER_OFF_RES", 
        "SP_PHYS_THER_ON_NON_RES", "SP_PHYS_THER_ON_RES", "SP_PHYS_OFF_RES", 
        "SP_PHYS_ON_NON_RES", "SP_PHYS_ON_RES", "SP_PODIATRY_OFF_RES", 
        "SP_PODIATRY_ON_NON_RES", "SP_PODIATRY_ON_RES", "SP_MED_SOC_OFF_RES", 
        "SP_MED_SOC_ON_NON_RES", "SP_MED_SOC_ON_RES", "SP_SPEECH_PH_OFF_RES", 
        "SP_SPEECH_PH_ON_NON_RES", "SP_SPEECH_PH_ON_RES", "SP_THER_REC_OFF_RES", 
        "SP_THER_REC_ON_NON_RES", "SP_THER_REC_ON_RES", "SP_VOC_GUID_OFF_RES", 
        "SP_VOC_GUID_ON_NON_RES", "SP_VOC_GUID_ON_RES", "SP_DIAG_XRAY_OFF_RES", 
        "SP_DIAG_XRAY_ON_NON_RES", "SP_DIAG_XRAY_ON_RES", "NUM_THER_REC_CONTRACT", 
        "NUM_THER_REC_FULL_TIME", "NUM_THER_REC_PART_TIME", "HHA_PROVIDES_DIRECT", 
        "OPERS_BRANCHES", "NUM_BRANCHES", "CHOW_IND", "HHA_QUAL_FOR_OPT", 
        "NUM_HOME_HEALTH_AIDES", "MEDICARE_CERT_HOSPICE", "HOSPICE_PROV_NUM", 
        "MEDICAID_CARE_VEND_NUM", "NUM_SOCIAL_WRKS", "NUM_SPEECH_PATH_AUDIO", 
        "SP_APPLIANCE_EQUIP", "SP_HH_AIDE_HOMEMAKER", "SP_INTERNS_RESIDENTS", 
        "SP_MEDICAL_SOCIAL", "SP_NURSING", "SP_NUTRITION_GUIDANCE", "SP_OTHER", 
        "SP_SPEECH_THERAPY", "SP_VOCAT_GUIDANCE", "SUBUNIT_IND", "OPERS_SUBUNITS", 
        "NUM_SUBUNITS", "SURETY_BOND_IND", "QUAL_OF_DIRECTOR", "NUM_AS_RADIO_TECH", 
        "NUM_BS_BA_RAD_TECH", "NUM_2YR_RADIO_TECH", "PROVIDES_OCCUP_THERAPY", 
        "NUM_PHY_THER_ARGNM", "NUM_PHYS_THERAPISTS", "NUM_SPEECH_PATH_AR", 
        "NUM_SPEECH_PATH", "SP_SPEECH_AND_PATH", "NETWORK_NUM", "NUM_PATIENT_TUE_SHIFT_4", 
        "NUM_STATIONS_HEMO", "TOT_STATIONS", "ADMIN_SUSP_DT", "NUM_ICF_MR_BEDS", 
        "NUM_DCARE_PERSNL", "LTC_AGREE_BEGIN_DT", "LTC_AGREE_END_DT", 
        "LTC_AGREE_EXT_DT", "PRIOR_LTC_END_DT", "PRIOR_LTC_EXT_DT", "TOT_EMPLOYEES", 
        "FED_PROG_SUPPORT", "NUM_NURSE_PRACT", "PARENT_PROV_NUM", "NUM_PHYS", 
        "TITL_FED_PROGR", "PARTIC_OPT_SP", "SP_OCCUP_THERAPY_2", "SP_OCCUP_THERAPY_3", 
        "SP_ORTHOTIC_PROSTHET", "SP_ORTHOTIC_PROSTHET_2", "SP_ORTHOTIC_PROSTHET_3", 
        "SP_PHYSICAL_THERAPY_2", "SP_PHYSICAL_THERAPY_3", "SP_PHYSICIAN", 
        "SP_PHYSICIAN_2", "SP_PHYSICIAN_3", "SP_PSYCHOLOGICAL", "SP_PSYCHOLOGICAL_2", 
        "SP_PSYCHOLOGICAL_3", "SP_RESP_CARE", "SP_RESP_CARE_2", "SP_RESP_CARE_3", 
        "SP_SOCIAL_2", "SP_SOCIAL_3", "SP_SPEECH_PATH_2", "SP_SPEECH_PATH_3", 
        "DT_SERVICE_BEGAN", "FREE_STAND_IND", "HOSP_BASED_IND", "NUM_OPERATING_ROOMS", 
        "SP_CARDIOVASCULAR", "SP_FOOT", "SP_GENERAL", "SP_NEUROLOGICAL", 
        "SP_OBSTETR_GYNECOL", "SP_OPTHAMOLOGY_SURG", "SP_ORAL", "SP_ORTHOPEDIC", 
        "SP_OTOLARYRGOLOGY", "SP_PLASTIC", "SP_THORACIC", "SP_UROLOGY", 
        "SP_EKG", "SP_LABORATORY", "SP_RADIOLOGY", "ACUTE_RESPITE", "NUM_STAFF_COUNSL", 
        "NUM_VOL_COUNSL", "NUM_VOL_HHA", "NUM_HOMEMAKERS", "NUM_VOL_HOMEMKR", 
        "NUM_VOL_LPN_LVN", "NUM_MED_SOCIAL_WRKS", "NUM_VOL_SOC_WORK", 
        "NUM_VOL_PHYS", "NUM_VOL_REG_NURS", "SP_COUNSELING", "SP_HOME_HEALTH_AIDE", 
        "SP_HOMEMAKER", "SP_MEDICAL_SUPPLIES", "SP_SHORT_TERM_INCARE", 
        "NUM_OTHER_VOLS", "TOT_VOLS", "FED_FUNDED_FFHC", "APPROVED_RHC_PROV_NUM", 
        "APPROVED_MEDICARE_RHC", "ACCRED_AABB_IND", "ACCRED_AOA_IND", 
        "ACCRED_ASHI_IND", "ACCRED_CAP_IND", "ACCRED_COLA_IND", "ACCRED_JCAHO_IND", 
        "ACCRED_AABB_DT", "ACCRED_AOA_DT", "ACCRED_ASHI_DT", "ACCRED_CAP_DT", 
        "ACCRED_COLA_DT", "ACCRED_JCAHO_DT", "ACCRED_AABB_MATCH_IND", 
        "ACCRED_AOA_MATCH_IND", "ACCRED_ASHI_MATCH_IND", "ACCRED_CAP_MATCH_IND", 
        "ACCRED_COLA_MATCH_IND", "ACCRED_JCAHO_MATCH_IND", "AFFIL_PROV_NUM_1", 
        "AFFIL_PROV_NUM_2", "AFFIL_PROV_NUM_3", "AFFIL_PROV_NUM_4", "AFFIL_PROV_NUM_5", 
        "AFFIL_PROV_NUM_6", "AFFIL_PROV_NUM_7", "AFFIL_PROV_NUM_8", "APPL_ACCR_ANN_TEST_VOL", 
        "APPL_ACCRED_SCHED_CD", "APPL_RECEIVED_DT", "APPL_TOT_ANN_TEST_VOL", 
        "TYPE_APPLICATION", "CERT_TYPE_CD_1", "CERT_MAILED_DT_1", "EFF_DT_1", 
        "CLIA_MEDICARE_NUM", "FAX_NUM", "CLIA_LAB_CLASS_CD", "MULTI_SITE_IND", 
        "NON_PROFIT_IND", "TOT_NUM_NON_WAIVED_IND", "NUM_CLIN_CONSULT", 
        "NUM_DIRECTORS", "NUM_GEN_SUPER", "TOT_NUM_SITES", "NUM_AFFIL_LABS", 
        "NUM_TECH_CONSULT", "NUM_TECH_SUPER", "NUM_TEST_PERSONNEL", "TOT_NUM_WAIVED_IND", 
        "CLIA67_IND", "SHARED_LAB_XREF_NUM", "SHARED_LAB_IND", "SURV_CERT_SCHED_CD", 
        "SURV_COMPL_SCHED_CD", "SURV_TOT_ANN_TEST_VOL", "TERM_CD", "TOT_ANN_TEST_VOL_WAIVED"
      )
    )
  } else if (year == 2003) {
    return(
      c("CATEGORY_SUBTYPE_IND", "CATEGORY", "CHOW_CNT", "CHOW_DT", 
        "CITY", "COMPL_ACCEPT_PLAN_COR", "STATUS_COMPL", "SSA_COUNTY", 
        "CROSS_REF_PROV_NUM", "FMS_SURVEY_DT_1", "SURVEY_DT_1", "ELIG_CD", 
        "FACILITY_NAME", "INTER_CARRIER_NUM", "MEDICAID_VEND_NUM", "PARTCI_DT", 
        "PRIOR_CHOW_DT", "PRIOR_INTER_CARRIER_NUM", "PROV_NUM", "RECORD_TYPE", 
        "REGION", "SKELETON_IND", "STATE_ABBREV", "SSA_STATE", "STATE_REGION_CD", 
        "STREET_ADDRESS", "PHONE_NUM", "TERM_CD_1", "EXP_DT_1", "TYPE_ACTION", 
        "TYPE_CONTROL", "ZIP_CD", "WS_FIPS_STATE", "WS_FIPS_CNTY", "WS_SSA_MSA_CD", 
        "WS_SSA_MSA_SIZE_CD", "ACCRED_EFF_DT", "ACCRED_EXP_DT", "ACCRED_STAT", 
        "ALCOH_DRG_UNIT_BED_SZ", "ALCOH_DRG_UNIT_EFF_DT", "ALCOH_DRG_UNIT_IND", 
        "ALCOH_DRG_UNIT_TERM_CD", "ALCOH_DRG_UNIT_TERM_DT", "NUM_BEDS", 
        "NUM_CERT_BEDS", "NUM_CERT_RN_ANEST", "CLIA_ID_NUM_A", "CLIA_ID_NUM_B", 
        "CLIA_ID_NUM_C", "CLIA_ID_NUM_D", "CLIA_ID_NUM_E", "COMPL_LSC", 
        "COMPL_SCOPE_OF_SERV", "COMPL_TECH_PERSNL", "COMPL_24_HR_RN", 
        "CURRENT_EVER_ACCRED", "CURRENT_EVER_NON_ACCRED", "CURRENT_EVER_SWINGBED", 
        "DT_VALID_SURVEY", "NUM_DIETICIANS", "FISC_YR_END_DT", "NUM_INHAL_THERAPY", 
        "NUM_LPN_LVN", "MED_SCHL_AFF", "MEETS_1861", "NUM_OCCUP_THERAPISTS", 
        "NUM_OTHER_PERSNL", "PARTICIPATING_CD", "NUM_PHYS_THERAPY", "NUM_PHYS_ASSIST", 
        "OLD_PROV_NUM", "PROG_PARTCI", "PSY_UNIT_BED_SZ", "PSY_UNIT_EFF_DT", 
        "PSY_UNIT_IND", "PSY_UNIT_TERM_CD", "PSY_UNIT_TERM_DT", "OVERRIDE_1", 
        "OVERRIDE_2", "OVERRIDE_3", "NUM_REG_NURS", "NUM_PHARMACIST_REG", 
        "REHAB_UNIT_BED_SZ", "REHAB_UNIT_EFF_DT", "REHAB_UNIT_IND", "REHAB_UNIT_TERM_CD", 
        "REHAB_UNIT_TERM_DT", "RELATED_PROV_NUM", "RES_PGM_APPR_ADA", 
        "RES_PGM_APPR_AMA", "RES_PGM_APPR_AOA", "RES_PGM_APPR_OTHER", 
        "NUM_RESID_PHYS", "SEP_COST_ENTITY_IND", "SP_ACUTE_REN_DIAL", 
        "SP_ALCOH_DRUG", "SP_ANESTH", "SP_BLOOD_BANK", "SP_BURN_UNIT", 
        "SP_CHIROPRATIC", "SP_CORONARY_CARE", "SP_DENTAL", "SP_DIETARY", 
        "SP_EMERG_DEPT", "SP_HOME_CARE_UNIT", "SP_HOSPICE", "SP_INPAT_SURG", 
        "SP_ICU", "SP_LABORATORY_ANATOM", "SP_LABORATORY_CLINIC", "SP_LTC_UNIT", 
        "SP_NEONATAL_NURS", "SP_NUCLEAR_MED", "SP_OBSTETRICS", "SP_OCCUP_THERAPY", 
        "SP_OPEN_HEART_SURG", "SP_OR_ROOMS", "SP_OPTOMETRIC", "SP_ORGAN_BANK", 
        "SP_ORGAN_TRANS", "SP_OUTPAT", "SP_OUTPAT_SURG", "SP_PEDIATRIC", 
        "SP_PHARMACY", "SP_PHYSICAL_THERAPY", "SP_POSTOP_REC_RM", "SP_PSYCHIATRIC", 
        "SP_RADIOLOGY_DIAG", "SP_RADIOLOGY_THERAPY", "SP_REHABIL", "SP_SELF_CARE", 
        "SP_SHOCK_TRAUMA", "SP_SOCIAL", "SP_SPEECH_PATH", "SWINGBED_IND", 
        "SWINGBED_SIZE_CD", "TYPE_FACILITY", "NON_PARTICIPATING_TYPE", 
        "NUM_ACT_THER_CONTRACT", "NUM_ACT_THER_FULL_TIME", "NUM_ACT_THER_PART_TIME", 
        "NUM_ADMN_CONTRACT", "NUM_ADMN_FULL_TIME", "NUM_ADMN_PART_TIME", 
        "NUM_T18_SNF_BEDS", "NUM_T19_SNF_BEDS", "NUM_T1819_SNF_BEDS", 
        "NUM_NURSE_AID_CONTRACT", "NUM_NURSE_AID_FULL_TIME", "NUM_NURSE_AID_PART_TIME", 
        "CHRISTIAN_SCIENCE_IND", "COMPL_BEDS_PER_ROOM", "COMPL_PATIENT_ROOM_SZ", 
        "COMPL_7_DAY_RN", "NUM_DENTIST_CONTRACT", "NUM_DENTIST_FULL_TIME", 
        "NUM_DENTIST_PART_TIME", "NUM_DIET_CONTRACT", "NUM_DIET_FULL_TIME", 
        "NUM_DIET_PART_TIME", "EXPER_RESEARCH", "NUM_FOOD_SRV_CONTRACT", 
        "NUM_FOOD_SRV_FULL_TIME", "NUM_FOOD_SRV_PART_TIME", "NUM_HOUSE_CONTRACT", 
        "NUM_HOUSE_FULL_TIME", "NUM_HOUSE_PART_TIME", "NUM_VOC_NURSE_CONTRACT", 
        "NUM_VOC_NURSE_FULL_TIME", "NUM_VOC_NURSE_PART_TIME", "LTC_CROSS_REF_PROV_NUM", 
        "NUM_MED_CONTRACT", "NUM_MED_FULL_TIME", "NUM_MED_PART_TIME", 
        "NUM_MED_AID_CONTRACT", "NUM_MED_AID_FULL_TIME", "NUM_MED_AID_PART_TIME", 
        "NUM_MEN_HLTH_CONTRACT", "NUM_MEN_HLTH_FULL_TIME", "NUM_MEN_HLTH_PART_TIME", 
        "NAME_MULT_FACL_ORG", "MULT_FACL_ORG", "NUM_AID_TRNG_CONTRACT", 
        "NUM_AID_TRNG_FULL_TIME", "NUM_AID_TRNG_PART_TIME", "NUM_NURSE_ADM_CONTRACT", 
        "NUM_NURSE_ADM_FULL_TIME", "NUM_NURSE_ADM_PART_TIME", "NUM_OCC_AID_CONTRACT", 
        "NUM_OCC_AID_FULL_TIME", "NUM_OCC_AID_PART_TIME", "NUM_OCC_ASST_CONTRACT", 
        "NUM_OCC_ASST_FULL_TIME", "NUM_OCC_ASST_PART_TIME", "NUM_OCC_THER_CONTRACT", 
        "NUM_OCC_THER_FULL_TIME", "NUM_OCC_THER_PART_TIME", "ORG_FAMILY_GRP", 
        "ORG_RESID_GRP", "NUM_OTH_CONTRACT", "NUM_OTH_FULL_TIME", "NUM_OTH_PART_TIME", 
        "NUM_OTH_ACT_CONTRACT", "NUM_OTH_ACT_FULL_TIME", "NUM_OTH_ACT_PART_TIME", 
        "NUM_OTH_PHY_CONTRACT", "NUM_OTH_PHY_FULL_TIME", "NUM_OTH_PHY_PART_TIME", 
        "NUM_OTH_SOC_CONTRACT", "NUM_OTH_SOC_FULL_TIME", "NUM_OTH_SOC_PART_TIME", 
        "NUM_PHAR_CONTRACT", "NUM_PHAR_FULL_TIME", "NUM_PHAR_PART_TIME", 
        "NUM_THER_ASST_CONTRACT", "NUM_THER_ASST_FULL_TIME", "NUM_THER_ASST_PART_TIME", 
        "NUM_THER_CONTRACT", "NUM_THER_FULL_TIME", "NUM_THER_PART_TIME", 
        "NUM_THER_AID_CONTRACT", "NUM_THER_AID_FULL_TIME", "NUM_THER_AID_PART_TIME", 
        "NUM_PHYS_EXT_CONTRACT", "NUM_PHYS_EXT_FULL_TIME", "NUM_PHYS_EXT_PART_TIME", 
        "NUM_POD_CONTRACT", "NUM_POD_FULL_TIME", "NUM_POD_PART_TIME", 
        "PROV_BASED_FACILITY", "NUM_REG_NURSE_CONTRACT", "NUM_REG_NURSE_FULL_TIME", 
        "NUM_REG_NURSE_PART_TIME", "RESC_SUSP_DT", "NUM_RN_DON_CONTRACT", 
        "NUM_RN_DON_FULL_TIME", "NUM_RN_DON_PART_TIME", "NUM_SOCIAL_CONTRACT", 
        "NUM_SOCIAL_FULL_TIME", "NUM_SOCIAL_PART_TIME", "NUM_AIDS_BEDS", 
        "NUM_ALZHEIMERS_BEDS", "NUM_DIAL_BEDS", "NUM_DIS_CHILD_BEDS", 
        "NUM_HEAD_TRAUMA_BEDS", "NUM_HOSPICE_BEDS", "NUM_HUNTING_DIS_BEDS", 
        "NUM_SPEC_REHAB_BEDS", "NUM_VENT_RESP_BEDS", "NUM_SPCH_PATH_CONTRACT", 
        "NUM_SPCH_PATH_FULL_TIME", "NUM_SPCH_PATH_PART_TIME", "SP_ACT_THER_OFF_RES", 
        "SP_ACT_THER_ON_NON_RES", "SP_ACT_THER_ON_RES", "SP_ADM_BLOOD_OFF_RES", 
        "SP_ADM_BLOOD_ON_NON_RES", "SP_ADM_BLOOD_ON_RES", "SP_CLIN_LAB_OFF_RES", 
        "SP_CLIN_LAB_ON_NON_RES", "SP_CLIN_LAB_ON_RES", "SP_DENTAL_OFF_RES", 
        "SP_DENTAL_ON_NON_RES", "SP_DENTAL_ON_RES", "SP_DIETARY_OFF_RES", 
        "SP_DIETARY_ON_NON_RES", "SP_DIETARY_ON_RES", "SP_HOUSE_KP_ON_NON_RES", 
        "SP_HOUSE_KP_OFF_RES", "SP_HOUSE_KP_ON_RES", "SP_MEN_HLTH_OFF_RES", 
        "SP_MEN_HLTH_ON_NON_RES", "SP_MEN_HLTH_ON_RES", "SP_NURSING_OFF_RES", 
        "SP_NURSING_ON_NON_RES", "SP_NURSING_ON_RES", "SP_OCC_THER_OFF_RES", 
        "SP_OCC_THER_ON_NON_RES", "SP_OCC_THER_ON_RES", "SP_OTH_ACT_OFF_RES", 
        "SP_OTH_ACT_ON_NON_RES", "SP_OTH_ACT_ON_RES", "SP_OTH_SOC_OFF_RES", 
        "SP_OTH_SOC_ON_NON_RES", "SP_OTH_SOC_ON_RES", "SP_PHARMACY_OFF_RES", 
        "SP_PHARMACY_ON_NON_RES", "SP_PHARMACY_ON_RES", "SP_PHYS_EXT_OFF_RES", 
        "SP_PHYS_EXT_ON_NON_RES", "SP_PHYS_EXT_ON_RES", "SP_PHYS_THER_OFF_RES", 
        "SP_PHYS_THER_ON_NON_RES", "SP_PHYS_THER_ON_RES", "SP_PHYS_OFF_RES", 
        "SP_PHYS_ON_NON_RES", "SP_PHYS_ON_RES", "SP_PODIATRY_OFF_RES", 
        "SP_PODIATRY_ON_NON_RES", "SP_PODIATRY_ON_RES", "SP_MED_SOC_OFF_RES", 
        "SP_MED_SOC_ON_NON_RES", "SP_MED_SOC_ON_RES", "SP_SPEECH_PH_OFF_RES", 
        "SP_SPEECH_PH_ON_NON_RES", "SP_SPEECH_PH_ON_RES", "SP_THER_REC_OFF_RES", 
        "SP_THER_REC_ON_NON_RES", "SP_THER_REC_ON_RES", "SP_VOC_GUID_OFF_RES", 
        "SP_VOC_GUID_ON_NON_RES", "SP_VOC_GUID_ON_RES", "SP_DIAG_XRAY_OFF_RES", 
        "SP_DIAG_XRAY_ON_NON_RES", "SP_DIAG_XRAY_ON_RES", "NUM_THER_REC_CONTRACT", 
        "NUM_THER_REC_FULL_TIME", "NUM_THER_REC_PART_TIME", "HHA_PROVIDES_DIRECT", 
        "OPERS_BRANCHES", "NUM_BRANCHES", "CHOW_IND", "HHA_QUAL_FOR_OPT", 
        "NUM_HOME_HEALTH_AIDES", "MEDICARE_CERT_HOSPICE", "HOSPICE_PROV_NUM", 
        "MEDICAID_CARE_VEND_NUM", "NUM_SOCIAL_WRKS", "NUM_SPEECH_PATH_AUDIO", 
        "SP_APPLIANCE_EQUIP", "SP_HH_AIDE_HOMEMAKER", "SP_INTERNS_RESIDENTS", 
        "SP_MEDICAL_SOCIAL", "SP_NURSING", "SP_NUTRITION_GUIDANCE", "SP_OTHER", 
        "SP_SPEECH_THERAPY", "SP_VOCAT_GUIDANCE", "SUBUNIT_IND", "OPERS_SUBUNITS", 
        "NUM_SUBUNITS", "SURETY_BOND_IND", "QUAL_OF_DIRECTOR", "NUM_AS_RADIO_TECH", 
        "NUM_BS_BA_RAD_TECH", "NUM_2YR_RADIO_TECH", "PROVIDES_OCCUP_THERAPY", 
        "NUM_PHY_THER_ARGNM", "NUM_PHYS_THERAPISTS", "NUM_SPEECH_PATH_AR", 
        "NUM_SPEECH_PATH", "SP_SPEECH_AND_PATH", "NETWORK_NUM", "NUM_PATIENT_TUE_SHIFT_4", 
        "NUM_STATIONS_HEMO", "TOT_STATIONS", "ADMIN_SUSP_DT", "NUM_ICF_MR_BEDS", 
        "NUM_DCARE_PERSNL", "LTC_AGREE_BEGIN_DT", "LTC_AGREE_END_DT", 
        "LTC_AGREE_EXT_DT", "PRIOR_ADMIN_SUSP_DT", "PRIOR_LTC_END_DT", 
        "PRIOR_LTC_EXT_DT", "PRIOR_RESC_SUSP_DT", "TOT_EMPLOYEES", "FED_PROG_SUPPORT", 
        "NUM_NURSE_PRACT", "PARENT_PROV_NUM", "NUM_PHYS", "TITL_FED_PROGR", 
        "PARTIC_OPT_SP", "SP_NURSING_2", "SP_NURSING_3", "SP_OCCUP_THERAPY_2", 
        "SP_OCCUP_THERAPY_3", "SP_ORTHOTIC_PROSTHET", "SP_ORTHOTIC_PROSTHET_2", 
        "SP_ORTHOTIC_PROSTHET_3", "SP_PHYSICAL_THERAPY_2", "SP_PHYSICAL_THERAPY_3", 
        "SP_PHYSICIAN", "SP_PHYSICIAN_2", "SP_PHYSICIAN_3", "SP_PSYCHOLOGICAL", 
        "SP_PSYCHOLOGICAL_2", "SP_PSYCHOLOGICAL_3", "SP_RESP_CARE", "SP_RESP_CARE_2", 
        "SP_RESP_CARE_3", "SP_SOCIAL_2", "SP_SOCIAL_3", "SP_SPEECH_PATH_2", 
        "SP_SPEECH_PATH_3", "DT_SERVICE_BEGAN", "FREE_STAND_IND", "HOSP_BASED_IND", 
        "NUM_OPERATING_ROOMS", "SP_CARDIOVASCULAR", "SP_FOOT", "SP_GENERAL", 
        "SP_NEUROLOGICAL", "SP_OBSTETR_GYNECOL", "SP_OPTHAMOLOGY_SURG", 
        "SP_ORAL", "SP_ORTHOPEDIC", "SP_OTOLARYRGOLOGY", "SP_PLASTIC", 
        "SP_THORACIC", "SP_UROLOGY", "SP_EKG", "SP_LABORATORY", "SP_RADIOLOGY", 
        "ACUTE_RESPITE", "NUM_STAFF_COUNSL", "NUM_VOL_COUNSL", "NUM_VOL_HHA", 
        "NUM_HOMEMAKERS", "NUM_VOL_HOMEMKR", "NUM_VOL_LPN_LVN", "NUM_MED_SOCIAL_WRKS", 
        "NUM_VOL_SOC_WORK", "NUM_VOL_PHYS", "NUM_VOL_REG_NURS", "SP_COUNSELING", 
        "SP_HOME_HEALTH_AIDE", "SP_HOMEMAKER", "SP_MEDICAL_SUPPLIES", 
        "SP_SHORT_TERM_INCARE", "NUM_OTHER_VOLS", "TOT_VOLS", "FED_FUNDED_FFHC", 
        "APPROVED_RHC_PROV_NUM", "APPROVED_MEDICARE_RHC", "ACCRED_AABB_IND", 
        "ACCRED_AOA_IND", "ACCRED_ASHI_IND", "ACCRED_CAP_IND", "ACCRED_COLA_IND", 
        "ACCRED_JCAHO_IND", "ACCRED_AABB_DT", "ACCRED_AOA_DT", "ACCRED_ASHI_DT", 
        "ACCRED_CAP_DT", "ACCRED_COLA_DT", "ACCRED_JCAHO_DT", "ACCRED_AABB_MATCH_IND", 
        "ACCRED_AOA_MATCH_IND", "ACCRED_ASHI_MATCH_IND", "ACCRED_CAP_MATCH_IND", 
        "ACCRED_COLA_MATCH_IND", "ACCRED_JCAHO_MATCH_IND", "AFFIL_PROV_NUM_1", 
        "AFFIL_PROV_NUM_2", "AFFIL_PROV_NUM_3", "AFFIL_PROV_NUM_4", "AFFIL_PROV_NUM_5", 
        "AFFIL_PROV_NUM_6", "AFFIL_PROV_NUM_7", "AFFIL_PROV_NUM_8", "APPL_ACCR_ANN_TEST_VOL", 
        "APPL_ACCRED_SCHED_CD", "APPL_RECEIVED_DT", "APPL_TOT_ANN_TEST_VOL", 
        "TYPE_APPLICATION", "CERT_TYPE_CD_1", "CERT_MAILED_DT_1", "EFF_DT_1", 
        "CLIA_MEDICARE_NUM", "FAX_NUM", "CLIA_LAB_CLASS_CD", "MULTI_SITE_IND", 
        "NON_PROFIT_IND", "TOT_NUM_NON_WAIVED_IND", "NUM_CLIN_CONSULT", 
        "NUM_DIRECTORS", "NUM_GEN_SUPER", "TOT_NUM_SITES", "NUM_AFFIL_LABS", 
        "NUM_TECH_CONSULT", "NUM_TECH_SUPER", "NUM_TEST_PERSONNEL", "TOT_NUM_WAIVED_IND", 
        "CLIA67_IND", "SHARED_LAB_XREF_NUM", "SHARED_LAB_IND", "SURV_CERT_SCHED_CD", 
        "SURV_COMPL_SCHED_CD", "SURV_TOT_ANN_TEST_VOL", "TERM_CD", "TOT_ANN_TEST_VOL_WAIVED"
      )
    )
  } else if (year == 2004) {
    return(
      c("CATEGORY_SUBTYPE_IND", "CATEGORY", "CHOW_CNT", "CHOW_DT", 
        "CITY", "COMPL_ACCEPT_PLAN_COR", "STATUS_COMPL", "SSA_COUNTY", 
        "CROSS_REF_PROV_NUM", "FMS_SURVEY_DT_1", "SURVEY_DT_1", "ELIG_CD", 
        "FACILITY_NAME", "INTER_CARRIER_NUM", "MEDICAID_VEND_NUM", "PARTCI_DT", 
        "PRIOR_CHOW_DT", "PRIOR_INTER_CARRIER_NUM", "PROV_NUM", "RECORD_TYPE", 
        "REGION", "SKELETON_IND", "STATE_ABBREV", "SSA_STATE", "STATE_REGION_CD", 
        "STREET_ADDRESS", "PHONE_NUM", "TERM_CD_1", "EXP_DT_1", "TYPE_ACTION", 
        "TYPE_CONTROL", "ZIP_CD", "WS_FIPS_STATE", "WS_FIPS_CNTY", "WS_SSA_MSA_CD", 
        "WS_SSA_MSA_SIZE_CD", "ACCRED_EFF_DT", "ACCRED_EXP_DT", "ACCRED_STAT", 
        "ALCOH_DRG_UNIT_BED_SZ", "ALCOH_DRG_UNIT_EFF_DT", "ALCOH_DRG_UNIT_IND", 
        "ALCOH_DRG_UNIT_TERM_CD", "ALCOH_DRG_UNIT_TERM_DT", "NUM_BEDS", 
        "NUM_CERT_BEDS", "NUM_CERT_RN_ANEST", "CLIA_ID_NUM_A", "CLIA_ID_NUM_B", 
        "CLIA_ID_NUM_C", "CLIA_ID_NUM_D", "CLIA_ID_NUM_E", "COMPL_LSC", 
        "COMPL_SCOPE_OF_SERV", "COMPL_TECH_PERSNL", "COMPL_24_HR_RN", 
        "CURRENT_EVER_ACCRED", "CURRENT_EVER_NON_ACCRED", "CURRENT_EVER_SWINGBED", 
        "DT_VALID_SURVEY", "NUM_DIETICIANS", "FISC_YR_END_DT", "NUM_INHAL_THERAPY", 
        "NUM_LPN_LVN", "MED_SCHL_AFF", "MEETS_1861", "NUM_OCCUP_THERAPISTS", 
        "NUM_OTHER_PERSNL", "PARTICIPATING_CD", "NUM_PHYS_THERAPY", "NUM_PHYS_ASSIST", 
        "OLD_PROV_NUM", "PROG_PARTCI", "PSY_UNIT_BED_SZ", "PSY_UNIT_EFF_DT", 
        "PSY_UNIT_IND", "PSY_UNIT_TERM_CD", "PSY_UNIT_TERM_DT", "OVERRIDE_1", 
        "OVERRIDE_2", "OVERRIDE_3", "NUM_REG_NURS", "NUM_PHARMACIST_REG", 
        "REHAB_UNIT_BED_SZ", "REHAB_UNIT_EFF_DT", "REHAB_UNIT_IND", "REHAB_UNIT_TERM_CD", 
        "REHAB_UNIT_TERM_DT", "RELATED_PROV_NUM", "RES_PGM_APPR_ADA", 
        "RES_PGM_APPR_AMA", "RES_PGM_APPR_AOA", "RES_PGM_APPR_OTHER", 
        "NUM_RESID_PHYS", "SEP_COST_ENTITY_IND", "SP_ACUTE_REN_DIAL", 
        "SP_ALCOH_DRUG", "SP_ANESTH", "SP_BLOOD_BANK", "SP_BURN_UNIT", 
        "SP_CHIROPRATIC", "SP_CORONARY_CARE", "SP_DENTAL", "SP_DIETARY", 
        "SP_EMERG_DEPT", "SP_HOME_CARE_UNIT", "SP_HOSPICE", "SP_INPAT_SURG", 
        "SP_ICU", "SP_LABORATORY_ANATOM", "SP_LABORATORY_CLINIC", "SP_LTC_UNIT", 
        "SP_NEONATAL_NURS", "SP_NUCLEAR_MED", "SP_OBSTETRICS", "SP_OCCUP_THERAPY", 
        "SP_OPEN_HEART_SURG", "SP_OR_ROOMS", "SP_OPTOMETRIC", "SP_ORGAN_BANK", 
        "SP_ORGAN_TRANS", "SP_OUTPAT", "SP_OUTPAT_SURG", "SP_PEDIATRIC", 
        "SP_PHARMACY", "SP_PHYSICAL_THERAPY", "SP_POSTOP_REC_RM", "SP_PSYCHIATRIC", 
        "SP_RADIOLOGY_DIAG", "SP_RADIOLOGY_THERAPY", "SP_REHABIL", "SP_SELF_CARE", 
        "SP_SHOCK_TRAUMA", "SP_SOCIAL", "SP_SPEECH_PATH", "SWINGBED_IND", 
        "SWINGBED_SIZE_CD", "TYPE_FACILITY", "NON_PARTICIPATING_TYPE", 
        "NUM_ACT_THER_CONTRACT", "NUM_ACT_THER_FULL_TIME", "NUM_ACT_THER_PART_TIME", 
        "NUM_ADMN_CONTRACT", "NUM_ADMN_FULL_TIME", "NUM_ADMN_PART_TIME", 
        "NUM_T18_SNF_BEDS", "NUM_T19_SNF_BEDS", "NUM_T1819_SNF_BEDS", 
        "NUM_NURSE_AID_CONTRACT", "NUM_NURSE_AID_FULL_TIME", "NUM_NURSE_AID_PART_TIME", 
        "CHRISTIAN_SCIENCE_IND", "COMPL_BEDS_PER_ROOM", "COMPL_PATIENT_ROOM_SZ", 
        "COMPL_7_DAY_RN", "NUM_DENTIST_CONTRACT", "NUM_DENTIST_FULL_TIME", 
        "NUM_DENTIST_PART_TIME", "NUM_DIET_CONTRACT", "NUM_DIET_FULL_TIME", 
        "NUM_DIET_PART_TIME", "EXPER_RESEARCH", "NUM_FOOD_SRV_CONTRACT", 
        "NUM_FOOD_SRV_FULL_TIME", "NUM_FOOD_SRV_PART_TIME", "NUM_HOUSE_CONTRACT", 
        "NUM_HOUSE_FULL_TIME", "NUM_HOUSE_PART_TIME", "NUM_VOC_NURSE_CONTRACT", 
        "NUM_VOC_NURSE_FULL_TIME", "NUM_VOC_NURSE_PART_TIME", "LTC_CROSS_REF_PROV_NUM", 
        "NUM_MED_CONTRACT", "NUM_MED_FULL_TIME", "NUM_MED_PART_TIME", 
        "NUM_MED_AID_CONTRACT", "NUM_MED_AID_FULL_TIME", "NUM_MED_AID_PART_TIME", 
        "NUM_MEN_HLTH_CONTRACT", "NUM_MEN_HLTH_FULL_TIME", "NUM_MEN_HLTH_PART_TIME", 
        "NAME_MULT_FACL_ORG", "MULT_FACL_ORG", "NUM_AID_TRNG_CONTRACT", 
        "NUM_AID_TRNG_FULL_TIME", "NUM_AID_TRNG_PART_TIME", "NUM_NURSE_ADM_CONTRACT", 
        "NUM_NURSE_ADM_FULL_TIME", "NUM_NURSE_ADM_PART_TIME", "NUM_OCC_AID_CONTRACT", 
        "NUM_OCC_AID_FULL_TIME", "NUM_OCC_AID_PART_TIME", "NUM_OCC_ASST_CONTRACT", 
        "NUM_OCC_ASST_FULL_TIME", "NUM_OCC_ASST_PART_TIME", "NUM_OCC_THER_CONTRACT", 
        "NUM_OCC_THER_FULL_TIME", "NUM_OCC_THER_PART_TIME", "ORG_FAMILY_GRP", 
        "ORG_RESID_GRP", "NUM_OTH_CONTRACT", "NUM_OTH_FULL_TIME", "NUM_OTH_PART_TIME", 
        "NUM_OTH_ACT_CONTRACT", "NUM_OTH_ACT_FULL_TIME", "NUM_OTH_ACT_PART_TIME", 
        "NUM_OTH_PHY_CONTRACT", "NUM_OTH_PHY_FULL_TIME", "NUM_OTH_PHY_PART_TIME", 
        "NUM_OTH_SOC_CONTRACT", "NUM_OTH_SOC_FULL_TIME", "NUM_OTH_SOC_PART_TIME", 
        "NUM_PHAR_CONTRACT", "NUM_PHAR_FULL_TIME", "NUM_PHAR_PART_TIME", 
        "NUM_THER_ASST_CONTRACT", "NUM_THER_ASST_FULL_TIME", "NUM_THER_ASST_PART_TIME", 
        "NUM_THER_CONTRACT", "NUM_THER_FULL_TIME", "NUM_THER_PART_TIME", 
        "NUM_THER_AID_CONTRACT", "NUM_THER_AID_FULL_TIME", "NUM_THER_AID_PART_TIME", 
        "NUM_PHYS_EXT_CONTRACT", "NUM_PHYS_EXT_FULL_TIME", "NUM_PHYS_EXT_PART_TIME", 
        "NUM_POD_CONTRACT", "NUM_POD_FULL_TIME", "NUM_POD_PART_TIME", 
        "PROV_BASED_FACILITY", "NUM_REG_NURSE_CONTRACT", "NUM_REG_NURSE_FULL_TIME", 
        "NUM_REG_NURSE_PART_TIME", "RESC_SUSP_DT", "NUM_RN_DON_CONTRACT", 
        "NUM_RN_DON_FULL_TIME", "NUM_RN_DON_PART_TIME", "NUM_SOCIAL_CONTRACT", 
        "NUM_SOCIAL_FULL_TIME", "NUM_SOCIAL_PART_TIME", "NUM_AIDS_BEDS", 
        "NUM_ALZHEIMERS_BEDS", "NUM_DIAL_BEDS", "NUM_DIS_CHILD_BEDS", 
        "NUM_HEAD_TRAUMA_BEDS", "NUM_HOSPICE_BEDS", "NUM_HUNTING_DIS_BEDS", 
        "NUM_SPEC_REHAB_BEDS", "NUM_VENT_RESP_BEDS", "NUM_SPCH_PATH_CONTRACT", 
        "NUM_SPCH_PATH_FULL_TIME", "NUM_SPCH_PATH_PART_TIME", "SP_ACT_THER_OFF_RES", 
        "SP_ACT_THER_ON_NON_RES", "SP_ACT_THER_ON_RES", "SP_ADM_BLOOD_OFF_RES", 
        "SP_ADM_BLOOD_ON_NON_RES", "SP_ADM_BLOOD_ON_RES", "SP_CLIN_LAB_OFF_RES", 
        "SP_CLIN_LAB_ON_NON_RES", "SP_CLIN_LAB_ON_RES", "SP_DENTAL_OFF_RES", 
        "SP_DENTAL_ON_NON_RES", "SP_DENTAL_ON_RES", "SP_DIETARY_OFF_RES", 
        "SP_DIETARY_ON_NON_RES", "SP_DIETARY_ON_RES", "SP_HOUSE_KP_ON_NON_RES", 
        "SP_HOUSE_KP_OFF_RES", "SP_HOUSE_KP_ON_RES", "SP_MEN_HLTH_OFF_RES", 
        "SP_MEN_HLTH_ON_NON_RES", "SP_MEN_HLTH_ON_RES", "SP_NURSING_OFF_RES", 
        "SP_NURSING_ON_NON_RES", "SP_NURSING_ON_RES", "SP_OCC_THER_OFF_RES", 
        "SP_OCC_THER_ON_NON_RES", "SP_OCC_THER_ON_RES", "SP_OTH_ACT_OFF_RES", 
        "SP_OTH_ACT_ON_NON_RES", "SP_OTH_ACT_ON_RES", "SP_OTH_SOC_OFF_RES", 
        "SP_OTH_SOC_ON_NON_RES", "SP_OTH_SOC_ON_RES", "SP_PHARMACY_OFF_RES", 
        "SP_PHARMACY_ON_NON_RES", "SP_PHARMACY_ON_RES", "SP_PHYS_EXT_OFF_RES", 
        "SP_PHYS_EXT_ON_NON_RES", "SP_PHYS_EXT_ON_RES", "SP_PHYS_THER_OFF_RES", 
        "SP_PHYS_THER_ON_NON_RES", "SP_PHYS_THER_ON_RES", "SP_PHYS_OFF_RES", 
        "SP_PHYS_ON_NON_RES", "SP_PHYS_ON_RES", "SP_PODIATRY_OFF_RES", 
        "SP_PODIATRY_ON_NON_RES", "SP_PODIATRY_ON_RES", "SP_MED_SOC_OFF_RES", 
        "SP_MED_SOC_ON_NON_RES", "SP_MED_SOC_ON_RES", "SP_SPEECH_PH_OFF_RES", 
        "SP_SPEECH_PH_ON_NON_RES", "SP_SPEECH_PH_ON_RES", "SP_THER_REC_OFF_RES", 
        "SP_THER_REC_ON_NON_RES", "SP_THER_REC_ON_RES", "SP_VOC_GUID_OFF_RES", 
        "SP_VOC_GUID_ON_NON_RES", "SP_VOC_GUID_ON_RES", "SP_DIAG_XRAY_OFF_RES", 
        "SP_DIAG_XRAY_ON_NON_RES", "SP_DIAG_XRAY_ON_RES", "NUM_THER_REC_CONTRACT", 
        "NUM_THER_REC_FULL_TIME", "NUM_THER_REC_PART_TIME", "HHA_PROVIDES_DIRECT", 
        "OPERS_BRANCHES", "NUM_BRANCHES", "CHOW_IND", "HHA_QUAL_FOR_OPT", 
        "NUM_HOME_HEALTH_AIDES", "MEDICARE_CERT_HOSPICE", "HOSPICE_PROV_NUM", 
        "MEDICAID_CARE_VEND_NUM", "NUM_SOCIAL_WRKS", "NUM_SPEECH_PATH_AUDIO", 
        "SP_APPLIANCE_EQUIP", "SP_HH_AIDE_HOMEMAKER", "SP_INTERNS_RESIDENTS", 
        "SP_MEDICAL_SOCIAL", "SP_NURSING", "SP_NUTRITION_GUIDANCE", "SP_OTHER", 
        "SP_SPEECH_THERAPY", "SP_VOCAT_GUIDANCE", "SUBUNIT_IND", "OPERS_SUBUNITS", 
        "NUM_SUBUNITS", "SURETY_BOND_IND", "QUAL_OF_DIRECTOR", "NUM_AS_RADIO_TECH", 
        "NUM_BS_BA_RAD_TECH", "NUM_2YR_RADIO_TECH", "PROVIDES_OCCUP_THERAPY", 
        "NUM_PHY_THER_ARGNM", "NUM_PHYS_THERAPISTS", "NUM_SPEECH_PATH_AR", 
        "NUM_SPEECH_PATH", "SP_SPEECH_AND_PATH", "NETWORK_NUM", "NUM_PATIENT_TUE_SHIFT_4", 
        "NUM_STATIONS_HEMO", "TOT_STATIONS", "ADMIN_SUSP_DT", "NUM_ICF_MR_BEDS", 
        "NUM_DCARE_PERSNL", "LTC_AGREE_BEGIN_DT", "LTC_AGREE_END_DT", 
        "LTC_AGREE_EXT_DT", "PRIOR_ADMIN_SUSP_DT", "PRIOR_LTC_END_DT", 
        "PRIOR_LTC_EXT_DT", "PRIOR_RESC_SUSP_DT", "TOT_EMPLOYEES", "FED_PROG_SUPPORT", 
        "NUM_NURSE_PRACT", "PARENT_PROV_NUM", "NUM_PHYS", "TITL_FED_PROGR", 
        "PARTIC_OPT_SP", "SP_NURSING_2", "SP_NURSING_3", "SP_OCCUP_THERAPY_2", 
        "SP_OCCUP_THERAPY_3", "SP_ORTHOTIC_PROSTHET", "SP_ORTHOTIC_PROSTHET_2", 
        "SP_ORTHOTIC_PROSTHET_3", "SP_PHYSICAL_THERAPY_2", "SP_PHYSICAL_THERAPY_3", 
        "SP_PHYSICIAN", "SP_PHYSICIAN_2", "SP_PHYSICIAN_3", "SP_PSYCHOLOGICAL", 
        "SP_PSYCHOLOGICAL_2", "SP_PSYCHOLOGICAL_3", "SP_RESP_CARE", "SP_RESP_CARE_2", 
        "SP_RESP_CARE_3", "SP_SOCIAL_2", "SP_SOCIAL_3", "SP_SPEECH_PATH_2", 
        "SP_SPEECH_PATH_3", "DT_SERVICE_BEGAN", "FREE_STAND_IND", "HOSP_BASED_IND", 
        "NUM_OPERATING_ROOMS", "SP_CARDIOVASCULAR", "SP_FOOT", "SP_GENERAL", 
        "SP_NEUROLOGICAL", "SP_OBSTETR_GYNECOL", "SP_OPTHAMOLOGY_SURG", 
        "SP_ORAL", "SP_ORTHOPEDIC", "SP_OTOLARYRGOLOGY", "SP_PLASTIC", 
        "SP_THORACIC", "SP_UROLOGY", "SP_EKG", "SP_LABORATORY", "SP_RADIOLOGY", 
        "ACUTE_RESPITE", "NUM_STAFF_COUNSL", "NUM_VOL_COUNSL", "NUM_VOL_HHA", 
        "NUM_HOMEMAKERS", "NUM_VOL_HOMEMKR", "NUM_VOL_LPN_LVN", "NUM_MED_SOCIAL_WRKS", 
        "NUM_VOL_SOC_WORK", "NUM_VOL_PHYS", "NUM_VOL_REG_NURS", "SP_COUNSELING", 
        "SP_HOME_HEALTH_AIDE", "SP_HOMEMAKER", "SP_MEDICAL_SUPPLIES", 
        "SP_SHORT_TERM_INCARE", "NUM_OTHER_VOLS", "TOT_VOLS", "FED_FUNDED_FFHC", 
        "APPROVED_RHC_PROV_NUM", "APPROVED_MEDICARE_RHC", "ACCRED_AABB_IND", 
        "ACCRED_AOA_IND", "ACCRED_ASHI_IND", "ACCRED_CAP_IND", "ACCRED_COLA_IND", 
        "ACCRED_JCAHO_IND", "ACCRED_AABB_DT", "ACCRED_AOA_DT", "ACCRED_ASHI_DT", 
        "ACCRED_CAP_DT", "ACCRED_COLA_DT", "ACCRED_JCAHO_DT", "ACCRED_AABB_MATCH_IND", 
        "ACCRED_AOA_MATCH_IND", "ACCRED_ASHI_MATCH_IND", "ACCRED_CAP_MATCH_IND", 
        "ACCRED_COLA_MATCH_IND", "ACCRED_JCAHO_MATCH_IND", "AFFIL_PROV_NUM_1", 
        "AFFIL_PROV_NUM_2", "AFFIL_PROV_NUM_3", "AFFIL_PROV_NUM_4", "AFFIL_PROV_NUM_5", 
        "AFFIL_PROV_NUM_6", "AFFIL_PROV_NUM_7", "AFFIL_PROV_NUM_8", "APPL_ACCR_ANN_TEST_VOL", 
        "APPL_ACCRED_SCHED_CD", "APPL_RECEIVED_DT", "APPL_TOT_ANN_TEST_VOL", 
        "TYPE_APPLICATION", "CERT_TYPE_CD_1", "CERT_MAILED_DT_1", "EFF_DT_1", 
        "CLIA_MEDICARE_NUM", "FAX_NUM", "CLIA_LAB_CLASS_CD", "MULTI_SITE_IND", 
        "NON_PROFIT_IND", "TOT_NUM_NON_WAIVED_IND", "NUM_CLIN_CONSULT", 
        "NUM_DIRECTORS", "NUM_GEN_SUPER", "TOT_NUM_SITES", "NUM_AFFIL_LABS", 
        "NUM_TECH_CONSULT", "NUM_TECH_SUPER", "NUM_TEST_PERSONNEL", "TOT_NUM_WAIVED_IND", 
        "CLIA67_IND", "SHARED_LAB_XREF_NUM", "SHARED_LAB_IND", "SURV_CERT_SCHED_CD", 
        "SURV_COMPL_SCHED_CD", "SURV_TOT_ANN_TEST_VOL", "TERM_CD", "TOT_ANN_TEST_VOL_WAIVED"
      )
    )
  } else if (year == 2005) {
    return(
      c("CATEGORY_SUBTYPE_IND", "CATEGORY", "CHOW_CNT", "CHOW_DT", 
        "CITY", "COMPL_ACCEPT_PLAN_COR", "STATUS_COMPL", "SSA_COUNTY", 
        "CROSS_REF_PROV_NUM", "FMS_SURVEY_DT_1", "SURVEY_DT_1", "ELIG_CD", 
        "FACILITY_NAME", "INTER_CARRIER_NUM", "MEDICAID_VEND_NUM", "PARTCI_DT", 
        "PRIOR_CHOW_DT", "PRIOR_INTER_CARRIER_NUM", "PROV_NUM", "RECORD_TYPE", 
        "REGION", "SKELETON_IND", "STATE_ABBREV", "SSA_STATE", "STATE_REGION_CD", 
        "STREET_ADDRESS", "PHONE_NUM", "TERM_CD_1", "EXP_DT_1", "TYPE_ACTION", 
        "TYPE_CONTROL", "ZIP_CD", "WS_FIPS_STATE", "WS_FIPS_CNTY", "WS_SSA_MSA_CD", 
        "WS_SSA_MSA_SIZE_CD", "ACCRED_EFF_DT", "ACCRED_EXP_DT", "ACCRED_STAT", 
        "NUM_BEDS", "NUM_CERT_BEDS", "NUM_CERT_RN_ANEST", "CLIA_ID_NUM_A", 
        "CLIA_ID_NUM_B", "CLIA_ID_NUM_C", "CLIA_ID_NUM_D", "CLIA_ID_NUM_E", 
        "COMPL_LSC", "COMPL_SCOPE_OF_SERV", "COMPL_TECH_PERSNL", "COMPL_24_HR_RN", 
        "CURRENT_EVER_ACCRED", "CURRENT_EVER_NON_ACCRED", "CURRENT_EVER_SWINGBED", 
        "DT_VALID_SURVEY", "NUM_DIETICIANS", "FISC_YR_END_DT", "NUM_LPN_LVN", 
        "MED_SCHL_AFF", "NUM_LAB_MED_TECHS", "MEETS_1861", "NUM_NUCL_MED_TECHS", 
        "NUM_OCCUP_THERAPISTS", "NUM_OTHER_PERSNL", "PARTICIPATING_CD", 
        "NUM_PHYS_THERAPY", "NUM_PHYS_ASSIST", "PROG_PARTCI", "PSY_UNIT_BED_SZ", 
        "PSY_UNIT_EFF_DT", "PSY_UNIT_IND", "PSY_UNIT_TERM_CD", "PSY_UNIT_TERM_DT", 
        "NUM_PSYCHOL", "NUM_RADIO_TECHS", "OVERRIDE_1", "OVERRIDE_2", 
        "OVERRIDE_3", "NUM_REG_NURS", "NUM_PHARMACIST_REG", "REHAB_UNIT_BED_SZ", 
        "REHAB_UNIT_EFF_DT", "REHAB_UNIT_IND", "REHAB_UNIT_TERM_CD", 
        "REHAB_UNIT_TERM_DT", "RES_PGM_APPR_ADA", "RES_PGM_APPR_AMA", 
        "RES_PGM_APPR_AOA", "RES_PGM_APPR_OTHER", "NUM_RESID_PHYS", "NUM_INHAL_THERAPY", 
        "SP_ACUTE_REN_DIAL", "SP_ALCOH_DRUG", "SP_AMBUL_OWNED", "SP_ANESTH", 
        "SP_AUDIO", "SP_BLOOD_BANK", "SP_BURN_UNIT", "SP_CARD_CATH_LAB", 
        "SP_OPEN_HEART_SURG", "SP_CHEMOTHER", "SP_CHIROPRATIC", "SP_CT_SCAN", 
        "SP_DENTAL", "SP_DIETARY", "SP_EMERG_DEDICATED", "SP_EMERG_DEPT", 
        "SP_GERON_SPEC", "SP_HOME_CARE_UNIT", "SP_HOSPICE", "SP_CORONARY_CARE", 
        "SP_ICU", "SP_ICU_NEONATAL", "SP_ICU_PEDIATRIC", "SP_ICU_SURG", 
        "SP_LABORATORY_ANATOM", "SP_LABORATORY_CLINIC", "SP_LTC_UNIT", 
        "SP_MRI", "SP_NEONATAL_NURS", "SP_NEURO_SURG", "SP_NUCLEAR_MED", 
        "SP_OBSTETRICS", "SP_OCCUP_THERAPY", "SP_OR_ROOMS", "SP_OPTHALMIC_SURG", 
        "SP_OPTOMETRIC", "SP_ORGAN_BANK", "SP_ORGAN_TRANS", "SP_ORTHOPEDIC_SURG", 
        "SP_OUTPAT", "SP_OUTPAT_SURG", "SP_PEDIATRIC", "SP_POS_EMIS_TOM_SCAN", 
        "SP_PHARMACY", "SP_PHYSICAL_THERAPY", "SP_POSTOP_REC_RM", "SP_PSYCHIATRIC", 
        "SP_PSY_FORENSIC", "SP_PSY_GERIATRIC", "SP_PSY_OUTPAT", "SP_PSY_CHILD_ADOL", 
        "SP_PSY_EMERG", "SP_RADIOLOGY_DIAG", "SP_RADIOLOGY_THERAPY", 
        "SP_RECON_SURG", "SP_REHABIL_CARF", "SP_REHABIL_OUTPAT", "SP_REHABIL", 
        "SP_EXTRAC_SHOCK_WAVE", "SP_SOCIAL", "SP_SPEECH_PATH", "SP_INPAT_SURG", 
        "SP_TRANS_MEDICARE", "SP_SHOCK_TRAUMA", "SP_URGENT_CARE", "SWINGBED_IND", 
        "SWINGBED_SIZE_CD", "TYPE_FACILITY", "NON_PARTICIPATING_TYPE", 
        "NUM_ACT_THER_CONTRACT", "NUM_ACT_THER_FULL_TIME", "NUM_ACT_THER_PART_TIME", 
        "NUM_ADMN_CONTRACT", "NUM_ADMN_FULL_TIME", "NUM_ADMN_PART_TIME", 
        "NUM_T18_SNF_BEDS", "NUM_T19_SNF_BEDS", "NUM_T1819_SNF_BEDS", 
        "NUM_NURSE_AID_CONTRACT", "NUM_NURSE_AID_FULL_TIME", "NUM_NURSE_AID_PART_TIME", 
        "CHRISTIAN_SCIENCE_IND", "COMPL_BEDS_PER_ROOM", "COMPL_PATIENT_ROOM_SZ", 
        "COMPL_7_DAY_RN", "NUM_DENTIST_CONTRACT", "NUM_DENTIST_FULL_TIME", 
        "NUM_DENTIST_PART_TIME", "NUM_DIET_CONTRACT", "NUM_DIET_FULL_TIME", 
        "NUM_DIET_PART_TIME", "EXPER_RESEARCH", "NUM_FOOD_SRV_CONTRACT", 
        "NUM_FOOD_SRV_FULL_TIME", "NUM_FOOD_SRV_PART_TIME", "NUM_HOUSE_CONTRACT", 
        "NUM_HOUSE_FULL_TIME", "NUM_HOUSE_PART_TIME", "NUM_VOC_NURSE_CONTRACT", 
        "NUM_VOC_NURSE_FULL_TIME", "NUM_VOC_NURSE_PART_TIME", "LTC_CROSS_REF_PROV_NUM", 
        "NUM_MED_CONTRACT", "NUM_MED_FULL_TIME", "NUM_MED_PART_TIME", 
        "NUM_MED_AID_CONTRACT", "NUM_MED_AID_FULL_TIME", "NUM_MED_AID_PART_TIME", 
        "NUM_MEN_HLTH_CONTRACT", "NUM_MEN_HLTH_FULL_TIME", "NUM_MEN_HLTH_PART_TIME", 
        "NAME_MULT_FACL_ORG", "MULT_FACL_ORG", "NUM_AID_TRNG_CONTRACT", 
        "NUM_AID_TRNG_FULL_TIME", "NUM_AID_TRNG_PART_TIME", "NUM_NURSE_ADM_CONTRACT", 
        "NUM_NURSE_ADM_FULL_TIME", "NUM_NURSE_ADM_PART_TIME", "NUM_OCC_AID_CONTRACT", 
        "NUM_OCC_AID_FULL_TIME", "NUM_OCC_AID_PART_TIME", "NUM_OCC_ASST_CONTRACT", 
        "NUM_OCC_ASST_FULL_TIME", "NUM_OCC_ASST_PART_TIME", "NUM_OCC_THER_CONTRACT", 
        "NUM_OCC_THER_FULL_TIME", "NUM_OCC_THER_PART_TIME", "ORG_FAMILY_GRP", 
        "ORG_RESID_GRP", "NUM_OTH_CONTRACT", "NUM_OTH_FULL_TIME", "NUM_OTH_PART_TIME", 
        "NUM_OTH_ACT_CONTRACT", "NUM_OTH_ACT_FULL_TIME", "NUM_OTH_ACT_PART_TIME", 
        "NUM_OTH_PHY_CONTRACT", "NUM_OTH_PHY_FULL_TIME", "NUM_OTH_PHY_PART_TIME", 
        "NUM_OTH_SOC_CONTRACT", "NUM_OTH_SOC_FULL_TIME", "NUM_OTH_SOC_PART_TIME", 
        "NUM_PHAR_CONTRACT", "NUM_PHAR_FULL_TIME", "NUM_PHAR_PART_TIME", 
        "NUM_THER_ASST_CONTRACT", "NUM_THER_ASST_FULL_TIME", "NUM_THER_ASST_PART_TIME", 
        "NUM_THER_CONTRACT", "NUM_THER_FULL_TIME", "NUM_THER_PART_TIME", 
        "NUM_THER_AID_CONTRACT", "NUM_THER_AID_FULL_TIME", "NUM_THER_AID_PART_TIME", 
        "NUM_PHYS_EXT_CONTRACT", "NUM_PHYS_EXT_FULL_TIME", "NUM_PHYS_EXT_PART_TIME", 
        "NUM_POD_CONTRACT", "NUM_POD_FULL_TIME", "NUM_POD_PART_TIME", 
        "PROV_BASED_FACILITY", "NUM_REG_NURSE_CONTRACT", "NUM_REG_NURSE_FULL_TIME", 
        "NUM_REG_NURSE_PART_TIME", "RELATED_PROV_NUM", "RESC_SUSP_DT", 
        "NUM_RN_DON_CONTRACT", "NUM_RN_DON_FULL_TIME", "NUM_RN_DON_PART_TIME", 
        "NUM_SOCIAL_CONTRACT", "NUM_SOCIAL_FULL_TIME", "NUM_SOCIAL_PART_TIME", 
        "NUM_AIDS_BEDS", "NUM_ALZHEIMERS_BEDS", "NUM_DIAL_BEDS", "NUM_DIS_CHILD_BEDS", 
        "NUM_HEAD_TRAUMA_BEDS", "NUM_HOSPICE_BEDS", "NUM_HUNTING_DIS_BEDS", 
        "NUM_SPEC_REHAB_BEDS", "NUM_VENT_RESP_BEDS", "NUM_SPCH_PATH_CONTRACT", 
        "NUM_SPCH_PATH_FULL_TIME", "NUM_SPCH_PATH_PART_TIME", "SP_ACT_THER_OFF_RES", 
        "SP_ACT_THER_ON_NON_RES", "SP_ACT_THER_ON_RES", "SP_ADM_BLOOD_OFF_RES", 
        "SP_ADM_BLOOD_ON_NON_RES", "SP_ADM_BLOOD_ON_RES", "SP_CLIN_LAB_OFF_RES", 
        "SP_CLIN_LAB_ON_NON_RES", "SP_CLIN_LAB_ON_RES", "SP_DENTAL_OFF_RES", 
        "SP_DENTAL_ON_NON_RES", "SP_DENTAL_ON_RES", "SP_DIETARY_OFF_RES", 
        "SP_DIETARY_ON_NON_RES", "SP_DIETARY_ON_RES", "SP_HOUSE_KP_ON_NON_RES", 
        "SP_HOUSE_KP_OFF_RES", "SP_HOUSE_KP_ON_RES", "SP_MEN_HLTH_OFF_RES", 
        "SP_MEN_HLTH_ON_NON_RES", "SP_MEN_HLTH_ON_RES", "SP_NURSING_OFF_RES", 
        "SP_NURSING_ON_NON_RES", "SP_NURSING_ON_RES", "SP_OCC_THER_OFF_RES", 
        "SP_OCC_THER_ON_NON_RES", "SP_OCC_THER_ON_RES", "SP_OTH_ACT_OFF_RES", 
        "SP_OTH_ACT_ON_NON_RES", "SP_OTH_ACT_ON_RES", "SP_OTH_SOC_OFF_RES", 
        "SP_OTH_SOC_ON_NON_RES", "SP_OTH_SOC_ON_RES", "SP_PHARMACY_OFF_RES", 
        "SP_PHARMACY_ON_NON_RES", "SP_PHARMACY_ON_RES", "SP_PHYS_EXT_OFF_RES", 
        "SP_PHYS_EXT_ON_NON_RES", "SP_PHYS_EXT_ON_RES", "SP_PHYS_THER_OFF_RES", 
        "SP_PHYS_THER_ON_NON_RES", "SP_PHYS_THER_ON_RES", "SP_PHYS_OFF_RES", 
        "SP_PHYS_ON_NON_RES", "SP_PHYS_ON_RES", "SP_PODIATRY_OFF_RES", 
        "SP_PODIATRY_ON_NON_RES", "SP_PODIATRY_ON_RES", "SP_MED_SOC_OFF_RES", 
        "SP_MED_SOC_ON_NON_RES", "SP_MED_SOC_ON_RES", "SP_SPEECH_PH_OFF_RES", 
        "SP_SPEECH_PH_ON_NON_RES", "SP_SPEECH_PH_ON_RES", "SP_THER_REC_OFF_RES", 
        "SP_THER_REC_ON_NON_RES", "SP_THER_REC_ON_RES", "SP_VOC_GUID_OFF_RES", 
        "SP_VOC_GUID_ON_NON_RES", "SP_VOC_GUID_ON_RES", "SP_DIAG_XRAY_OFF_RES", 
        "SP_DIAG_XRAY_ON_NON_RES", "SP_DIAG_XRAY_ON_RES", "NUM_THER_REC_CONTRACT", 
        "NUM_THER_REC_FULL_TIME", "NUM_THER_REC_PART_TIME", "HHA_PROVIDES_DIRECT", 
        "OPERS_BRANCHES", "NUM_BRANCHES", "CHOW_IND", "HHA_QUAL_FOR_OPT", 
        "NUM_HOME_HEALTH_AIDES", "MEDICARE_CERT_HOSPICE", "HOSPICE_PROV_NUM", 
        "MEDICAID_CARE_VEND_NUM", "NUM_SOCIAL_WRKS", "NUM_SPEECH_PATH_AUDIO", 
        "SP_APPLIANCE_EQUIP", "SP_HH_AIDE_HOMEMAKER", "SP_INTERNS_RESIDENTS", 
        "SP_MEDICAL_SOCIAL", "SP_NURSING", "SP_NUTRITION_GUIDANCE", "SP_OTHER", 
        "SP_SPEECH_THERAPY", "SP_VOCAT_GUIDANCE", "SUBUNIT_IND", "OPERS_SUBUNITS", 
        "NUM_SUBUNITS", "SURETY_BOND_IND", "QUAL_OF_DIRECTOR", "NUM_AS_RADIO_TECH", 
        "NUM_BS_BA_RAD_TECH", "NUM_2YR_RADIO_TECH", "PROVIDES_OCCUP_THERAPY", 
        "NUM_PHY_THER_ARGNM", "NUM_PHYS_THERAPISTS", "NUM_SPEECH_PATH_AR", 
        "NUM_SPEECH_PATH", "SP_SPEECH_AND_PATH", "NETWORK_NUM", "NUM_PATIENT_TUE_SHIFT_4", 
        "NUM_STATIONS_HEMO", "TOT_STATIONS", "ADMIN_SUSP_DT", "NUM_ICF_MR_BEDS", 
        "NUM_DCARE_PERSNL", "LTC_AGREE_BEGIN_DT", "LTC_AGREE_END_DT", 
        "LTC_AGREE_EXT_DT", "PRIOR_ADMIN_SUSP_DT", "PRIOR_LTC_END_DT", 
        "PRIOR_LTC_EXT_DT", "PRIOR_RESC_SUSP_DT", "TOT_EMPLOYEES", "FED_PROG_SUPPORT", 
        "NUM_NURSE_PRACT", "PARENT_PROV_NUM", "NUM_PHYS", "TITL_FED_PROGR", 
        "PARTIC_OPT_SP", "SP_NURSING_2", "SP_NURSING_3", "SP_OCCUP_THERAPY_2", 
        "SP_OCCUP_THERAPY_3", "SP_ORTHOTIC_PROSTHET", "SP_ORTHOTIC_PROSTHET_2", 
        "SP_ORTHOTIC_PROSTHET_3", "SP_PHYSICAL_THERAPY_2", "SP_PHYSICAL_THERAPY_3", 
        "SP_PHYSICIAN", "SP_PHYSICIAN_2", "SP_PHYSICIAN_3", "SP_PSYCHOLOGICAL", 
        "SP_PSYCHOLOGICAL_2", "SP_PSYCHOLOGICAL_3", "SP_RESP_CARE", "SP_RESP_CARE_2", 
        "SP_RESP_CARE_3", "SP_SOCIAL_2", "SP_SOCIAL_3", "SP_SPEECH_PATH_2", 
        "SP_SPEECH_PATH_3", "DT_SERVICE_BEGAN", "FREE_STAND_IND", "HOSP_BASED_IND", 
        "NUM_OPERATING_ROOMS", "SP_CARDIOVASCULAR", "SP_FOOT", "SP_GENERAL", 
        "SP_NEUROLOGICAL", "SP_OBSTETR_GYNECOL", "SP_OPTHAMOLOGY_SURG", 
        "SP_ORAL", "SP_ORTHOPEDIC", "SP_OTOLARYRGOLOGY", "SP_PLASTIC", 
        "SP_THORACIC", "SP_UROLOGY", "SP_EKG", "SP_LABORATORY", "SP_RADIOLOGY", 
        "ACUTE_RESPITE", "NUM_STAFF_COUNSL", "NUM_VOL_COUNSL", "NUM_VOL_HHA", 
        "NUM_HOMEMAKERS", "NUM_VOL_HOMEMKR", "NUM_VOL_LPN_LVN", "NUM_MED_SOCIAL_WRKS", 
        "NUM_VOL_SOC_WORK", "NUM_VOL_PHYS", "NUM_VOL_REG_NURS", "SP_COUNSELING", 
        "SP_HOME_HEALTH_AIDE", "SP_HOMEMAKER", "SP_MEDICAL_SUPPLIES", 
        "SP_SHORT_TERM_INCARE", "NUM_OTHER_VOLS", "TOT_VOLS", "FED_FUNDED_FFHC", 
        "APPROVED_RHC_PROV_NUM", "APPROVED_MEDICARE_RHC", "ACCRED_AABB_IND", 
        "ACCRED_AOA_IND", "ACCRED_ASHI_IND", "ACCRED_CAP_IND", "ACCRED_COLA_IND", 
        "ACCRED_JCAHO_IND", "ACCRED_AABB_DT", "ACCRED_AOA_DT", "ACCRED_ASHI_DT", 
        "ACCRED_CAP_DT", "ACCRED_COLA_DT", "ACCRED_JCAHO_DT", "ACCRED_AABB_MATCH_IND", 
        "ACCRED_AOA_MATCH_IND", "ACCRED_ASHI_MATCH_IND", "ACCRED_CAP_MATCH_IND", 
        "ACCRED_COLA_MATCH_IND", "ACCRED_JCAHO_MATCH_IND", "AFFIL_PROV_NUM_1", 
        "AFFIL_PROV_NUM_2", "AFFIL_PROV_NUM_3", "AFFIL_PROV_NUM_4", "AFFIL_PROV_NUM_5", 
        "AFFIL_PROV_NUM_6", "AFFIL_PROV_NUM_7", "AFFIL_PROV_NUM_8", "APPL_ACCR_ANN_TEST_VOL", 
        "APPL_ACCRED_SCHED_CD", "APPL_RECEIVED_DT", "APPL_TOT_ANN_TEST_VOL", 
        "TYPE_APPLICATION", "CERT_TYPE_CD_1", "CERT_MAILED_DT_1", "EFF_DT_1", 
        "CLIA_MEDICARE_NUM", "FAX_NUM", "CLIA_LAB_CLASS_CD", "MULTI_SITE_IND", 
        "NON_PROFIT_IND", "TOT_NUM_NON_WAIVED_IND", "NUM_CLIN_CONSULT", 
        "NUM_DIRECTORS", "NUM_GEN_SUPER", "TOT_NUM_SITES", "NUM_AFFIL_LABS", 
        "NUM_TECH_CONSULT", "NUM_TECH_SUPER", "NUM_TEST_PERSONNEL", "TOT_NUM_WAIVED_IND", 
        "CLIA67_IND", "SHARED_LAB_XREF_NUM", "SHARED_LAB_IND", "SURV_CERT_SCHED_CD", 
        "SURV_COMPL_SCHED_CD", "SURV_TOT_ANN_TEST_VOL", "TERM_CD", "TOT_ANN_TEST_VOL_WAIVED"
      )
    )
  } else if (year == 2006) {
    return(
      c("CATEGORY_SUBTYPE_IND", "CATEGORY", "CHOW_CNT", "CHOW_DT", 
        "CITY", "COMPL_ACCEPT_PLAN_COR", "STATUS_COMPL", "SSA_COUNTY", 
        "CROSS_REF_PROV_NUM", "FMS_SURVEY_DT_1", "SURVEY_DT_1", "ELIG_CD", 
        "FACILITY_NAME", "INTER_CARRIER_NUM", "MEDICAID_VEND_NUM", "PARTCI_DT", 
        "PRIOR_CHOW_DT", "PRIOR_INTER_CARRIER_NUM", "PROV_NUM", "RECORD_TYPE", 
        "REGION", "SKELETON_IND", "STATE_ABBREV", "SSA_STATE", "STATE_REGION_CD", 
        "STREET_ADDRESS", "PHONE_NUM", "TERM_CD_1", "EXP_DT_1", "TYPE_ACTION", 
        "TYPE_CONTROL", "ZIP_CD", "WS_FIPS_STATE", "WS_FIPS_CNTY", "WS_SSA_MSA_CD", 
        "WS_SSA_MSA_SIZE_CD", "ACCRED_EFF_DT", "ACCRED_EXP_DT", "ACCRED_STAT", 
        "NUM_BEDS", "NUM_CERT_BEDS", "NUM_CERT_RN_ANEST", "CLIA_ID_NUM_A", 
        "CLIA_ID_NUM_B", "CLIA_ID_NUM_C", "CLIA_ID_NUM_D", "CLIA_ID_NUM_E", 
        "COMPL_LSC", "COMPL_SCOPE_OF_SERV", "COMPL_TECH_PERSNL", "COMPL_24_HR_RN", 
        "CURRENT_EVER_ACCRED", "CURRENT_EVER_NON_ACCRED", "CURRENT_EVER_SWINGBED", 
        "DT_VALID_SURVEY", "NUM_DIETICIANS", "FISC_YR_END_DT", "NUM_LPN_LVN", 
        "MED_SCHL_AFF", "NUM_LAB_MED_TECHS", "MEETS_1861", "NUM_NUCL_MED_TECHS", 
        "NUM_OCCUP_THERAPISTS", "NUM_OTHER_PERSNL", "PARTICIPATING_CD", 
        "NUM_PHYS_THERAPY", "NUM_PHYS_ASSIST", "PROG_PARTCI", "PSY_UNIT_BED_SZ", 
        "PSY_UNIT_EFF_DT", "PSY_UNIT_IND", "PSY_UNIT_TERM_CD", "PSY_UNIT_TERM_DT", 
        "NUM_PSYCHOL", "NUM_RADIO_TECHS", "OVERRIDE_1", "OVERRIDE_2", 
        "OVERRIDE_3", "NUM_REG_NURS", "NUM_PHARMACIST_REG", "REHAB_UNIT_BED_SZ", 
        "REHAB_UNIT_EFF_DT", "REHAB_UNIT_IND", "REHAB_UNIT_TERM_CD", 
        "REHAB_UNIT_TERM_DT", "RES_PGM_APPR_ADA", "RES_PGM_APPR_AMA", 
        "RES_PGM_APPR_AOA", "RES_PGM_APPR_OTHER", "NUM_RESID_PHYS", "NUM_INHAL_THERAPY", 
        "SP_ACUTE_REN_DIAL", "SP_ALCOH_DRUG", "SP_AMBUL_OWNED", "SP_ANESTH", 
        "SP_AUDIO", "SP_BLOOD_BANK", "SP_BURN_UNIT", "SP_CARD_CATH_LAB", 
        "SP_OPEN_HEART_SURG", "SP_CHEMOTHER", "SP_CHIROPRATIC", "SP_CT_SCAN", 
        "SP_DENTAL", "SP_DIETARY", "SP_EMERG_DEDICATED", "SP_EMERG_DEPT", 
        "SP_GERON_SPEC", "SP_HOME_CARE_UNIT", "SP_HOSPICE", "SP_CORONARY_CARE", 
        "SP_ICU", "SP_ICU_NEONATAL", "SP_ICU_PEDIATRIC", "SP_ICU_SURG", 
        "SP_LABORATORY_ANATOM", "SP_LABORATORY_CLINIC", "SP_LTC_UNIT", 
        "SP_MRI", "SP_NEONATAL_NURS", "SP_NEURO_SURG", "SP_NUCLEAR_MED", 
        "SP_OBSTETRICS", "SP_OCCUP_THERAPY", "SP_OR_ROOMS", "SP_OPTHALMIC_SURG", 
        "SP_OPTOMETRIC", "SP_ORGAN_BANK", "SP_ORGAN_TRANS", "SP_ORTHOPEDIC_SURG", 
        "SP_OUTPAT", "SP_OUTPAT_SURG", "SP_PEDIATRIC", "SP_POS_EMIS_TOM_SCAN", 
        "SP_PHARMACY", "SP_PHYSICAL_THERAPY", "SP_POSTOP_REC_RM", "SP_PSYCHIATRIC", 
        "SP_PSY_FORENSIC", "SP_PSY_GERIATRIC", "SP_PSY_OUTPAT", "SP_PSY_CHILD_ADOL", 
        "SP_PSY_EMERG", "SP_RADIOLOGY_DIAG", "SP_RADIOLOGY_THERAPY", 
        "SP_RECON_SURG", "SP_REHABIL_CARF", "SP_REHABIL_OUTPAT", "SP_REHABIL", 
        "SP_EXTRAC_SHOCK_WAVE", "SP_SOCIAL", "SP_SPEECH_PATH", "SP_INPAT_SURG", 
        "SP_TRANS_MEDICARE", "SP_SHOCK_TRAUMA", "SP_URGENT_CARE", "SWINGBED_IND", 
        "SWINGBED_SIZE_CD", "TYPE_FACILITY", "NON_PARTICIPATING_TYPE", 
        "NUM_ACT_THER_CONTRACT", "NUM_ACT_THER_FULL_TIME", "NUM_ACT_THER_PART_TIME", 
        "NUM_ADMN_CONTRACT", "NUM_ADMN_FULL_TIME", "NUM_ADMN_PART_TIME", 
        "NUM_T18_SNF_BEDS", "NUM_T19_SNF_BEDS", "NUM_T1819_SNF_BEDS", 
        "NUM_NURSE_AID_CONTRACT", "NUM_NURSE_AID_FULL_TIME", "NUM_NURSE_AID_PART_TIME", 
        "CHRISTIAN_SCIENCE_IND", "COMPL_BEDS_PER_ROOM", "COMPL_PATIENT_ROOM_SZ", 
        "COMPL_7_DAY_RN", "NUM_DENTIST_CONTRACT", "NUM_DENTIST_FULL_TIME", 
        "NUM_DENTIST_PART_TIME", "NUM_DIET_CONTRACT", "NUM_DIET_FULL_TIME", 
        "NUM_DIET_PART_TIME", "EXPER_RESEARCH", "NUM_FOOD_SRV_CONTRACT", 
        "NUM_FOOD_SRV_FULL_TIME", "NUM_FOOD_SRV_PART_TIME", "NUM_HOUSE_CONTRACT", 
        "NUM_HOUSE_FULL_TIME", "NUM_HOUSE_PART_TIME", "NUM_VOC_NURSE_CONTRACT", 
        "NUM_VOC_NURSE_FULL_TIME", "NUM_VOC_NURSE_PART_TIME", "LTC_CROSS_REF_PROV_NUM", 
        "NUM_MED_CONTRACT", "NUM_MED_FULL_TIME", "NUM_MED_PART_TIME", 
        "NUM_MED_AID_CONTRACT", "NUM_MED_AID_FULL_TIME", "NUM_MED_AID_PART_TIME", 
        "NUM_MEN_HLTH_CONTRACT", "NUM_MEN_HLTH_FULL_TIME", "NUM_MEN_HLTH_PART_TIME", 
        "NAME_MULT_FACL_ORG", "MULT_FACL_ORG", "NUM_AID_TRNG_CONTRACT", 
        "NUM_AID_TRNG_FULL_TIME", "NUM_AID_TRNG_PART_TIME", "NUM_NURSE_ADM_CONTRACT", 
        "NUM_NURSE_ADM_FULL_TIME", "NUM_NURSE_ADM_PART_TIME", "NUM_OCC_THER_FULL_TIME", 
        "NUM_OCC_THER_CONTRACT", "NUM_OCC_AID_CONTRACT", "NUM_OCC_AID_FULL_TIME", 
        "NUM_OCC_AID_PART_TIME", "NUM_OCC_ASST_CONTRACT", "NUM_OCC_ASST_FULL_TIME", 
        "NUM_OCC_ASST_PART_TIME", "NUM_OCC_THER_PART_TIME", "ORG_FAMILY_GRP", 
        "ORG_RESID_GRP", "NUM_OTH_CONTRACT", "NUM_OTH_FULL_TIME", "NUM_OTH_PART_TIME", 
        "NUM_OTH_ACT_CONTRACT", "NUM_OTH_ACT_FULL_TIME", "NUM_OTH_ACT_PART_TIME", 
        "NUM_OTH_PHY_CONTRACT", "NUM_OTH_PHY_FULL_TIME", "NUM_OTH_PHY_PART_TIME", 
        "NUM_OTH_SOC_CONTRACT", "NUM_OTH_SOC_FULL_TIME", "NUM_OTH_SOC_PART_TIME", 
        "NUM_PHAR_CONTRACT", "NUM_PHAR_FULL_TIME", "NUM_PHAR_PART_TIME", 
        "NUM_THER_ASST_CONTRACT", "NUM_THER_ASST_FULL_TIME", "NUM_THER_ASST_PART_TIME", 
        "NUM_THER_CONTRACT", "NUM_THER_FULL_TIME", "NUM_THER_PART_TIME", 
        "NUM_THER_AID_CONTRACT", "NUM_THER_AID_FULL_TIME", "NUM_THER_AID_PART_TIME", 
        "NUM_PHYS_EXT_CONTRACT", "NUM_PHYS_EXT_FULL_TIME", "NUM_PHYS_EXT_PART_TIME", 
        "NUM_POD_CONTRACT", "NUM_POD_FULL_TIME", "NUM_POD_PART_TIME", 
        "PROV_BASED_FACILITY", "NUM_REG_NURSE_CONTRACT", "NUM_REG_NURSE_FULL_TIME", 
        "NUM_REG_NURSE_PART_TIME", "RELATED_PROV_NUM", "RESC_SUSP_DT", 
        "NUM_RN_DON_CONTRACT", "NUM_RN_DON_FULL_TIME", "NUM_RN_DON_PART_TIME", 
        "NUM_SOCIAL_CONTRACT", "NUM_SOCIAL_FULL_TIME", "NUM_SOCIAL_PART_TIME", 
        "NUM_AIDS_BEDS", "NUM_ALZHEIMERS_BEDS", "NUM_DIAL_BEDS", "NUM_DIS_CHILD_BEDS", 
        "NUM_HEAD_TRAUMA_BEDS", "NUM_HOSPICE_BEDS", "NUM_HUNTING_DIS_BEDS", 
        "NUM_SPEC_REHAB_BEDS", "NUM_VENT_RESP_BEDS", "NUM_SPCH_PATH_CONTRACT", 
        "NUM_SPCH_PATH_FULL_TIME", "NUM_SPCH_PATH_PART_TIME", "SP_ACT_THER_OFF_RES", 
        "SP_ACT_THER_ON_NON_RES", "SP_ACT_THER_ON_RES", "SP_ADM_BLOOD_OFF_RES", 
        "SP_ADM_BLOOD_ON_NON_RES", "SP_ADM_BLOOD_ON_RES", "SP_CLIN_LAB_OFF_RES", 
        "SP_CLIN_LAB_ON_NON_RES", "SP_CLIN_LAB_ON_RES", "SP_DENTAL_OFF_RES", 
        "SP_DENTAL_ON_NON_RES", "SP_DENTAL_ON_RES", "SP_DIETARY_OFF_RES", 
        "SP_DIETARY_ON_NON_RES", "SP_DIETARY_ON_RES", "SP_HOUSE_KP_ON_NON_RES", 
        "SP_HOUSE_KP_OFF_RES", "SP_HOUSE_KP_ON_RES", "SP_MEN_HLTH_OFF_RES", 
        "SP_MEN_HLTH_ON_NON_RES", "SP_MEN_HLTH_ON_RES", "SP_NURSING_OFF_RES", 
        "SP_NURSING_ON_NON_RES", "SP_NURSING_ON_RES", "SP_OCC_THER_OFF_RES", 
        "SP_OCC_THER_ON_NON_RES", "SP_OCC_THER_ON_RES", "SP_OTH_ACT_OFF_RES", 
        "SP_OTH_ACT_ON_NON_RES", "SP_OTH_ACT_ON_RES", "SP_OTH_SOC_OFF_RES", 
        "SP_OTH_SOC_ON_NON_RES", "SP_OTH_SOC_ON_RES", "SP_PHARMACY_OFF_RES", 
        "SP_PHARMACY_ON_NON_RES", "SP_PHARMACY_ON_RES", "SP_PHYS_EXT_OFF_RES", 
        "SP_PHYS_EXT_ON_NON_RES", "SP_PHYS_EXT_ON_RES", "SP_PHYS_THER_OFF_RES", 
        "SP_PHYS_THER_ON_NON_RES", "SP_PHYS_THER_ON_RES", "SP_PHYS_OFF_RES", 
        "SP_PHYS_ON_NON_RES", "SP_PHYS_ON_RES", "SP_PODIATRY_OFF_RES", 
        "SP_PODIATRY_ON_NON_RES", "SP_PODIATRY_ON_RES", "SP_MED_SOC_OFF_RES", 
        "SP_MED_SOC_ON_NON_RES", "SP_MED_SOC_ON_RES", "SP_SPEECH_PH_OFF_RES", 
        "SP_SPEECH_PH_ON_NON_RES", "SP_SPEECH_PH_ON_RES", "SP_THER_REC_OFF_RES", 
        "SP_THER_REC_ON_NON_RES", "SP_THER_REC_ON_RES", "SP_VOC_GUID_OFF_RES", 
        "SP_VOC_GUID_ON_NON_RES", "SP_VOC_GUID_ON_RES", "SP_DIAG_XRAY_OFF_RES", 
        "SP_DIAG_XRAY_ON_NON_RES", "SP_DIAG_XRAY_ON_RES", "NUM_THER_REC_CONTRACT", 
        "NUM_THER_REC_FULL_TIME", "NUM_THER_REC_PART_TIME", "HHA_PROVIDES_DIRECT", 
        "OPERS_BRANCHES", "NUM_BRANCHES", "CHOW_IND", "HHA_QUAL_FOR_OPT", 
        "NUM_HOME_HEALTH_AIDES", "MEDICARE_CERT_HOSPICE", "HOSPICE_PROV_NUM", 
        "MEDICAID_CARE_VEND_NUM", "NUM_SOCIAL_WRKS", "NUM_SPEECH_PATH_AUDIO", 
        "SP_APPLIANCE_EQUIP", "SP_HH_AIDE_HOMEMAKER", "SP_INTERNS_RESIDENTS", 
        "SP_MEDICAL_SOCIAL", "SP_NURSING", "SP_NUTRITION_GUIDANCE", "SP_OTHER", 
        "SP_SPEECH_THERAPY", "SP_VOCAT_GUIDANCE", "SUBUNIT_IND", "OPERS_SUBUNITS", 
        "NUM_SUBUNITS", "SURETY_BOND_IND", "QUAL_OF_DIRECTOR", "NUM_AS_RADIO_TECH", 
        "NUM_BS_BA_RAD_TECH", "NUM_2YR_RADIO_TECH", "NUM_PHY_THER_ARGNM", 
        "NUM_PHYS_THERAPISTS", "NUM_SPEECH_PATH_AR", "NUM_SPEECH_PATH", 
        "NETWORK_NUM", "NUM_PATIENT_TUE_SHIFT_4", "NUM_STATIONS_HEMO", 
        "TOT_STATIONS", "ADMIN_SUSP_DT", "NUM_ICF_MR_BEDS", "NUM_DCARE_PERSNL", 
        "LTC_AGREE_BEGIN_DT", "LTC_AGREE_END_DT", "LTC_AGREE_EXT_DT", 
        "PRIOR_ADMIN_SUSP_DT", "PRIOR_LTC_END_DT", "PRIOR_LTC_EXT_DT", 
        "PRIOR_RESC_SUSP_DT", "TOT_EMPLOYEES", "FED_PROG_SUPPORT", "NUM_NURSE_PRACT", 
        "PARENT_PROV_NUM", "NUM_PHYS", "TITL_FED_PROGR", "PARTIC_OPT_SP", 
        "SP_NURSING_2", "SP_NURSING_3", "SP_OCCUP_THERAPY_2", "SP_OCCUP_THERAPY_3", 
        "SP_ORTHOTIC_PROSTHET", "SP_ORTHOTIC_PROSTHET_2", "SP_ORTHOTIC_PROSTHET_3", 
        "SP_PHYSICAL_THERAPY_2", "SP_PHYSICAL_THERAPY_3", "SP_PHYSICIAN", 
        "SP_PHYSICIAN_2", "SP_PHYSICIAN_3", "SP_PSYCHOLOGICAL", "SP_PSYCHOLOGICAL_2", 
        "SP_PSYCHOLOGICAL_3", "SP_RESP_CARE", "SP_RESP_CARE_2", "SP_RESP_CARE_3", 
        "SP_SOCIAL_2", "SP_SOCIAL_3", "SP_SPEECH_PATH_2", "SP_SPEECH_PATH_3", 
        "DT_SERVICE_BEGAN", "FREE_STAND_IND", "HOSP_BASED_IND", "NUM_OPERATING_ROOMS", 
        "SP_CARDIOVASCULAR", "SP_FOOT", "SP_GENERAL", "SP_NEUROLOGICAL", 
        "SP_OBSTETR_GYNECOL", "SP_OPTHAMOLOGY_SURG", "SP_ORAL", "SP_ORTHOPEDIC", 
        "SP_OTOLARYRGOLOGY", "SP_PLASTIC", "SP_THORACIC", "SP_UROLOGY", 
        "SP_EKG", "SP_LABORATORY", "SP_RADIOLOGY", "ACUTE_RESPITE", "NUM_STAFF_COUNSL", 
        "NUM_VOL_COUNSL", "NUM_VOL_HHA", "NUM_HOMEMAKERS", "NUM_VOL_HOMEMKR", 
        "NUM_VOL_LPN_LVN", "NUM_MED_SOCIAL_WRKS", "NUM_VOL_SOC_WORK", 
        "NUM_VOL_PHYS", "NUM_VOL_REG_NURS", "SP_COUNSELING", "SP_HOME_HEALTH_AIDE", 
        "SP_HOMEMAKER", "SP_MEDICAL_SUPPLIES", "SP_SHORT_TERM_INCARE", 
        "NUM_OTHER_VOLS", "TOT_VOLS", "FED_FUNDED_FFHC", "APPROVED_RHC_PROV_NUM", 
        "APPROVED_MEDICARE_RHC", "ACCRED_AABB_IND", "ACCRED_AOA_IND", 
        "ACCRED_ASHI_IND", "ACCRED_CAP_IND", "ACCRED_COLA_IND", "ACCRED_JCAHO_IND", 
        "ACCRED_AABB_DT", "ACCRED_AOA_DT", "ACCRED_ASHI_DT", "ACCRED_CAP_DT", 
        "ACCRED_COLA_DT", "ACCRED_JCAHO_DT", "ACCRED_AABB_MATCH_IND", 
        "ACCRED_AOA_MATCH_IND", "ACCRED_ASHI_MATCH_IND", "ACCRED_CAP_MATCH_IND", 
        "ACCRED_COLA_MATCH_IND", "ACCRED_JCAHO_MATCH_IND", "AFFIL_PROV_NUM_1", 
        "AFFIL_PROV_NUM_2", "AFFIL_PROV_NUM_3", "AFFIL_PROV_NUM_4", "AFFIL_PROV_NUM_5", 
        "AFFIL_PROV_NUM_6", "AFFIL_PROV_NUM_7", "AFFIL_PROV_NUM_8", "APPL_ACCR_ANN_TEST_VOL", 
        "APPL_ACCRED_SCHED_CD", "APPL_RECEIVED_DT", "APPL_TOT_ANN_TEST_VOL", 
        "TYPE_APPLICATION", "CERT_TYPE_CD_1", "CERT_MAILED_DT_1", "EFF_DT_1", 
        "CLIA_MEDICARE_NUM", "CLIA_LAB_CLASS_CD", "FAX_NUM", "CLIA_LAB_CLASS_CD_1", 
        "CLIA_LAB_CLASS_CD_10", "CLIA_LAB_CLASS_CD_2", "CLIA_LAB_CLASS_CD_3", 
        "CLIA_LAB_CLASS_CD_4", "CLIA_LAB_CLASS_CD_5", "CLIA_LAB_CLASS_CD_6", 
        "CLIA_LAB_CLASS_CD_7", "CLIA_LAB_CLASS_CD_8", "CLIA_LAB_CLASS_CD_9", 
        "MULTI_SITE_IND", "NON_PROFIT_IND", "TOT_NUM_NON_WAIVED_IND", 
        "NUM_CLIN_CONSULT", "NUM_DIRECTORS", "NUM_GEN_SUPER", "TOT_NUM_SITES", 
        "NUM_AFFIL_LABS", "NUM_TECH_CONSULT", "NUM_TECH_SUPER", "NUM_TEST_PERSONNEL", 
        "TOT_NUM_WAIVED_IND", "PEND_CLIA_LAB_CLASS_CD", "CLIA67_IND", 
        "SHARED_LAB_XREF_NUM", "SHARED_LAB_IND", "SURV_CERT_SCHED_CD", 
        "SURV_COMPL_SCHED_CD", "SURV_TOT_ANN_TEST_VOL", "TERM_CD", "TOT_ANN_TEST_VOL_WAIVED"
      )
    )
  } else if (year == 2007) {
    return(
      c("CATEGORY_SUBTYPE_IND", "CATEGORY", "CHOW_CNT", "CHOW_DT", 
        "CITY", "COMPL_ACCEPT_PLAN_COR", "STATUS_COMPL", "SSA_COUNTY", 
        "CROSS_REF_PROV_NUM", "FMS_SURVEY_DT_1", "SURVEY_DT_1", "ELIG_CD", 
        "FACILITY_NAME", "INTER_CARRIER_NUM", "MEDICAID_VEND_NUM", "PARTCI_DT", 
        "PRIOR_CHOW_DT", "PRIOR_INTER_CARRIER_NUM", "PROV_NUM", "RECORD_TYPE", 
        "REGION", "SKELETON_IND", "STATE_ABBREV", "SSA_STATE", "STATE_REGION_CD", 
        "STREET_ADDRESS", "PHONE_NUM", "TERM_CD_1", "EXP_DT_1", "TYPE_ACTION", 
        "TYPE_CONTROL", "ZIP_CD", "WS_FIPS_STATE", "WS_FIPS_CNTY", "WS_SSA_MSA_CD", 
        "WS_SSA_MSA_SIZE_CD", "ACCRED_EFF_DT", "ACCRED_EXP_DT", "ACCRED_STAT", 
        "NUM_BEDS", "NUM_CERT_BEDS", "NUM_CERT_RN_ANEST", "CLIA_ID_NUM_A", 
        "CLIA_ID_NUM_B", "CLIA_ID_NUM_C", "CLIA_ID_NUM_D", "CLIA_ID_NUM_E", 
        "COMPL_LSC", "COMPL_SCOPE_OF_SERV", "COMPL_TECH_PERSNL", "COMPL_24_HR_RN", 
        "CURRENT_EVER_ACCRED", "CURRENT_EVER_NON_ACCRED", "CURRENT_EVER_SWINGBED", 
        "DT_VALID_SURVEY", "NUM_DIETICIANS", "FISC_YR_END_DT", "NUM_LPN_LVN", 
        "MED_SCHL_AFF", "NUM_LAB_MED_TECHS", "MEETS_1861", "NUM_NUCL_MED_TECHS", 
        "NUM_OCCUP_THERAPISTS", "NUM_OTHER_PERSNL", "PARTICIPATING_CD", 
        "NUM_PHYS_THERAPY", "NUM_PHYS_ASSIST", "PROG_PARTCI", "PSY_UNIT_BED_SZ", 
        "PSY_UNIT_EFF_DT", "PSY_UNIT_IND", "PSY_UNIT_TERM_CD", "PSY_UNIT_TERM_DT", 
        "NUM_PSYCHOL", "NUM_RADIO_TECHS", "OVERRIDE_1", "OVERRIDE_2", 
        "OVERRIDE_3", "NUM_REG_NURS", "NUM_PHARMACIST_REG", "REHAB_UNIT_BED_SZ", 
        "REHAB_UNIT_EFF_DT", "REHAB_UNIT_IND", "REHAB_UNIT_TERM_CD", 
        "REHAB_UNIT_TERM_DT", "RES_PGM_APPR_ADA", "RES_PGM_APPR_AMA", 
        "RES_PGM_APPR_AOA", "RES_PGM_APPR_OTHER", "NUM_RESID_PHYS", "NUM_INHAL_THERAPY", 
        "SP_ACUTE_REN_DIAL", "SP_ALCOH_DRUG", "SP_AMBUL_OWNED", "SP_ANESTH", 
        "SP_AUDIO", "SP_BLOOD_BANK", "SP_BURN_UNIT", "SP_CARD_CATH_LAB", 
        "SP_OPEN_HEART_SURG", "SP_CHEMOTHER", "SP_CHIROPRATIC", "SP_CT_SCAN", 
        "SP_DENTAL", "SP_DIETARY", "SP_EMERG_DEDICATED", "SP_EMERG_DEPT", 
        "SP_GERON_SPEC", "SP_HOME_CARE_UNIT", "SP_HOSPICE", "SP_CORONARY_CARE", 
        "SP_ICU", "SP_ICU_NEONATAL", "SP_ICU_PEDIATRIC", "SP_ICU_SURG", 
        "SP_LABORATORY_ANATOM", "SP_LABORATORY_CLINIC", "SP_LTC_UNIT", 
        "SP_MRI", "SP_NEONATAL_NURS", "SP_NEURO_SURG", "SP_NUCLEAR_MED", 
        "SP_OBSTETRICS", "SP_OCCUP_THERAPY", "SP_OR_ROOMS", "SP_OPTHALMIC_SURG", 
        "SP_OPTOMETRIC", "SP_ORGAN_BANK", "SP_ORGAN_TRANS", "SP_ORTHOPEDIC_SURG", 
        "SP_OUTPAT", "SP_OUTPAT_SURG", "SP_PEDIATRIC", "SP_POS_EMIS_TOM_SCAN", 
        "SP_PHARMACY", "SP_PHYSICAL_THERAPY", "SP_POSTOP_REC_RM", "SP_PSYCHIATRIC", 
        "SP_PSY_FORENSIC", "SP_PSY_GERIATRIC", "SP_PSY_OUTPAT", "SP_PSY_CHILD_ADOL", 
        "SP_PSY_EMERG", "SP_RADIOLOGY_DIAG", "SP_RADIOLOGY_THERAPY", 
        "SP_RECON_SURG", "SP_REHABIL_CARF", "SP_REHABIL_OUTPAT", "SP_REHABIL", 
        "SP_EXTRAC_SHOCK_WAVE", "SP_SOCIAL", "SP_SPEECH_PATH", "SP_INPAT_SURG", 
        "SP_TRANS_MEDICARE", "SP_SHOCK_TRAUMA", "SP_URGENT_CARE", "SWINGBED_IND", 
        "SWINGBED_SIZE_CD", "TYPE_FACILITY", "NON_PARTICIPATING_TYPE", 
        "NUM_ACT_THER_CONTRACT", "NUM_ACT_THER_FULL_TIME", "NUM_ACT_THER_PART_TIME", 
        "NUM_ADMN_CONTRACT", "NUM_ADMN_FULL_TIME", "NUM_ADMN_PART_TIME", 
        "NUM_T18_SNF_BEDS", "NUM_T19_SNF_BEDS", "NUM_T1819_SNF_BEDS", 
        "NUM_NURSE_AID_CONTRACT", "NUM_NURSE_AID_FULL_TIME", "NUM_NURSE_AID_PART_TIME", 
        "CHRISTIAN_SCIENCE_IND", "COMPL_BEDS_PER_ROOM", "COMPL_PATIENT_ROOM_SZ", 
        "COMPL_7_DAY_RN", "NUM_DENTIST_CONTRACT", "NUM_DENTIST_FULL_TIME", 
        "NUM_DENTIST_PART_TIME", "NUM_DIET_CONTRACT", "NUM_DIET_FULL_TIME", 
        "NUM_DIET_PART_TIME", "EXPER_RESEARCH", "NUM_FOOD_SRV_CONTRACT", 
        "NUM_FOOD_SRV_FULL_TIME", "NUM_FOOD_SRV_PART_TIME", "NUM_HOUSE_CONTRACT", 
        "NUM_HOUSE_FULL_TIME", "NUM_HOUSE_PART_TIME", "NUM_VOC_NURSE_CONTRACT", 
        "NUM_VOC_NURSE_FULL_TIME", "NUM_VOC_NURSE_PART_TIME", "LTC_CROSS_REF_PROV_NUM", 
        "NUM_MED_CONTRACT", "NUM_MED_FULL_TIME", "NUM_MED_PART_TIME", 
        "NUM_MED_AID_CONTRACT", "NUM_MED_AID_FULL_TIME", "NUM_MED_AID_PART_TIME", 
        "NUM_MEN_HLTH_CONTRACT", "NUM_MEN_HLTH_FULL_TIME", "NUM_MEN_HLTH_PART_TIME", 
        "NAME_MULT_FACL_ORG", "MULT_FACL_ORG", "NUM_AID_TRNG_CONTRACT", 
        "NUM_AID_TRNG_FULL_TIME", "NUM_AID_TRNG_PART_TIME", "NUM_NURSE_ADM_CONTRACT", 
        "NUM_NURSE_ADM_FULL_TIME", "NUM_NURSE_ADM_PART_TIME", "NUM_OCC_THER_FULL_TIME", 
        "NUM_OCC_THER_CONTRACT", "NUM_OCC_AID_CONTRACT", "NUM_OCC_AID_FULL_TIME", 
        "NUM_OCC_AID_PART_TIME", "NUM_OCC_ASST_CONTRACT", "NUM_OCC_ASST_FULL_TIME", 
        "NUM_OCC_ASST_PART_TIME", "NUM_OCC_THER_PART_TIME", "ORG_FAMILY_GRP", 
        "ORG_RESID_GRP", "NUM_OTH_CONTRACT", "NUM_OTH_FULL_TIME", "NUM_OTH_PART_TIME", 
        "NUM_OTH_ACT_CONTRACT", "NUM_OTH_ACT_FULL_TIME", "NUM_OTH_ACT_PART_TIME", 
        "NUM_OTH_PHY_CONTRACT", "NUM_OTH_PHY_FULL_TIME", "NUM_OTH_PHY_PART_TIME", 
        "NUM_OTH_SOC_CONTRACT", "NUM_OTH_SOC_FULL_TIME", "NUM_OTH_SOC_PART_TIME", 
        "NUM_PHAR_CONTRACT", "NUM_PHAR_FULL_TIME", "NUM_PHAR_PART_TIME", 
        "NUM_THER_ASST_CONTRACT", "NUM_THER_ASST_FULL_TIME", "NUM_THER_ASST_PART_TIME", 
        "NUM_THER_CONTRACT", "NUM_THER_FULL_TIME", "NUM_THER_PART_TIME", 
        "NUM_THER_AID_CONTRACT", "NUM_THER_AID_FULL_TIME", "NUM_THER_AID_PART_TIME", 
        "NUM_PHYS_EXT_CONTRACT", "NUM_PHYS_EXT_FULL_TIME", "NUM_PHYS_EXT_PART_TIME", 
        "NUM_POD_CONTRACT", "NUM_POD_FULL_TIME", "NUM_POD_PART_TIME", 
        "PROV_BASED_FACILITY", "NUM_REG_NURSE_CONTRACT", "NUM_REG_NURSE_FULL_TIME", 
        "NUM_REG_NURSE_PART_TIME", "RELATED_PROV_NUM", "RESC_SUSP_DT", 
        "NUM_RN_DON_CONTRACT", "NUM_RN_DON_FULL_TIME", "NUM_RN_DON_PART_TIME", 
        "NUM_SOCIAL_CONTRACT", "NUM_SOCIAL_FULL_TIME", "NUM_SOCIAL_PART_TIME", 
        "NUM_AIDS_BEDS", "NUM_ALZHEIMERS_BEDS", "NUM_DIAL_BEDS", "NUM_DIS_CHILD_BEDS", 
        "NUM_HEAD_TRAUMA_BEDS", "NUM_HOSPICE_BEDS", "NUM_HUNTING_DIS_BEDS", 
        "NUM_SPEC_REHAB_BEDS", "NUM_VENT_RESP_BEDS", "NUM_SPCH_PATH_CONTRACT", 
        "NUM_SPCH_PATH_FULL_TIME", "NUM_SPCH_PATH_PART_TIME", "SP_ACT_THER_OFF_RES", 
        "SP_ACT_THER_ON_NON_RES", "SP_ACT_THER_ON_RES", "SP_ADM_BLOOD_OFF_RES", 
        "SP_ADM_BLOOD_ON_NON_RES", "SP_ADM_BLOOD_ON_RES", "SP_CLIN_LAB_OFF_RES", 
        "SP_CLIN_LAB_ON_NON_RES", "SP_CLIN_LAB_ON_RES", "SP_DENTAL_OFF_RES", 
        "SP_DENTAL_ON_NON_RES", "SP_DENTAL_ON_RES", "SP_DIETARY_OFF_RES", 
        "SP_DIETARY_ON_NON_RES", "SP_DIETARY_ON_RES", "SP_HOUSE_KP_ON_NON_RES", 
        "SP_HOUSE_KP_OFF_RES", "SP_HOUSE_KP_ON_RES", "SP_MEN_HLTH_OFF_RES", 
        "SP_MEN_HLTH_ON_NON_RES", "SP_MEN_HLTH_ON_RES", "SP_NURSING_OFF_RES", 
        "SP_NURSING_ON_NON_RES", "SP_NURSING_ON_RES", "SP_OCC_THER_OFF_RES", 
        "SP_OCC_THER_ON_NON_RES", "SP_OCC_THER_ON_RES", "SP_OTH_ACT_OFF_RES", 
        "SP_OTH_ACT_ON_NON_RES", "SP_OTH_ACT_ON_RES", "SP_OTH_SOC_OFF_RES", 
        "SP_OTH_SOC_ON_NON_RES", "SP_OTH_SOC_ON_RES", "SP_PHARMACY_OFF_RES", 
        "SP_PHARMACY_ON_NON_RES", "SP_PHARMACY_ON_RES", "SP_PHYS_EXT_OFF_RES", 
        "SP_PHYS_EXT_ON_NON_RES", "SP_PHYS_EXT_ON_RES", "SP_PHYS_THER_OFF_RES", 
        "SP_PHYS_THER_ON_NON_RES", "SP_PHYS_THER_ON_RES", "SP_PHYS_OFF_RES", 
        "SP_PHYS_ON_NON_RES", "SP_PHYS_ON_RES", "SP_PODIATRY_OFF_RES", 
        "SP_PODIATRY_ON_NON_RES", "SP_PODIATRY_ON_RES", "SP_MED_SOC_OFF_RES", 
        "SP_MED_SOC_ON_NON_RES", "SP_MED_SOC_ON_RES", "SP_SPEECH_PH_OFF_RES", 
        "SP_SPEECH_PH_ON_NON_RES", "SP_SPEECH_PH_ON_RES", "SP_THER_REC_OFF_RES", 
        "SP_THER_REC_ON_NON_RES", "SP_THER_REC_ON_RES", "SP_VOC_GUID_OFF_RES", 
        "SP_VOC_GUID_ON_NON_RES", "SP_VOC_GUID_ON_RES", "SP_DIAG_XRAY_OFF_RES", 
        "SP_DIAG_XRAY_ON_NON_RES", "SP_DIAG_XRAY_ON_RES", "NUM_THER_REC_CONTRACT", 
        "NUM_THER_REC_FULL_TIME", "NUM_THER_REC_PART_TIME", "HHA_PROVIDES_DIRECT", 
        "OPERS_BRANCHES", "NUM_BRANCHES", "CHOW_IND", "HHA_QUAL_FOR_OPT", 
        "NUM_HOME_HEALTH_AIDES", "MEDICARE_CERT_HOSPICE", "HOSPICE_PROV_NUM", 
        "MEDICAID_CARE_VEND_NUM", "NUM_SOCIAL_WRKS", "NUM_SPEECH_PATH_AUDIO", 
        "SP_APPLIANCE_EQUIP", "SP_HH_AIDE_HOMEMAKER", "SP_INTERNS_RESIDENTS", 
        "SP_MEDICAL_SOCIAL", "SP_NURSING", "SP_NUTRITION_GUIDANCE", "SP_OTHER", 
        "SP_SPEECH_THERAPY", "SP_VOCAT_GUIDANCE", "SUBUNIT_IND", "OPERS_SUBUNITS", 
        "NUM_SUBUNITS", "SURETY_BOND_IND", "QUAL_OF_DIRECTOR", "NUM_AS_RADIO_TECH", 
        "NUM_BS_BA_RAD_TECH", "NUM_2YR_RADIO_TECH", "NUM_PHY_THER_ARGNM", 
        "NUM_PHYS_THERAPISTS", "NUM_SPEECH_PATH_AR", "NUM_SPEECH_PATH", 
        "NETWORK_NUM", "NUM_PATIENT_TUE_SHIFT_4", "NUM_STATIONS_HEMO", 
        "TOT_STATIONS", "ADMIN_SUSP_DT", "NUM_ICF_MR_BEDS", "NUM_DCARE_PERSNL", 
        "LTC_AGREE_BEGIN_DT", "LTC_AGREE_END_DT", "LTC_AGREE_EXT_DT", 
        "PRIOR_ADMIN_SUSP_DT", "PRIOR_LTC_END_DT", "PRIOR_LTC_EXT_DT", 
        "PRIOR_RESC_SUSP_DT", "TOT_EMPLOYEES", "FED_PROG_SUPPORT", "NUM_NURSE_PRACT", 
        "PARENT_PROV_NUM", "NUM_PHYS", "TITL_FED_PROGR", "PARTIC_OPT_SP", 
        "SP_NURSING_2", "SP_NURSING_3", "SP_OCCUP_THERAPY_2", "SP_OCCUP_THERAPY_3", 
        "SP_ORTHOTIC_PROSTHET", "SP_ORTHOTIC_PROSTHET_2", "SP_ORTHOTIC_PROSTHET_3", 
        "SP_PHYSICAL_THERAPY_2", "SP_PHYSICAL_THERAPY_3", "SP_PHYSICIAN", 
        "SP_PHYSICIAN_2", "SP_PHYSICIAN_3", "SP_PSYCHOLOGICAL", "SP_PSYCHOLOGICAL_2", 
        "SP_PSYCHOLOGICAL_3", "SP_RESP_CARE", "SP_RESP_CARE_2", "SP_RESP_CARE_3", 
        "SP_SOCIAL_2", "SP_SOCIAL_3", "SP_SPEECH_PATH_2", "SP_SPEECH_PATH_3", 
        "DT_SERVICE_BEGAN", "FREE_STAND_IND", "HOSP_BASED_IND", "NUM_OPERATING_ROOMS", 
        "SP_CARDIOVASCULAR", "SP_FOOT", "SP_GENERAL", "SP_NEUROLOGICAL", 
        "SP_OBSTETR_GYNECOL", "SP_OPTHAMOLOGY_SURG", "SP_ORAL", "SP_ORTHOPEDIC", 
        "SP_OTOLARYRGOLOGY", "SP_PLASTIC", "SP_THORACIC", "SP_UROLOGY", 
        "SP_EKG", "SP_LABORATORY", "SP_RADIOLOGY", "ACUTE_RESPITE", "NUM_STAFF_COUNSL", 
        "NUM_VOL_COUNSL", "NUM_VOL_HHA", "NUM_HOMEMAKERS", "NUM_VOL_HOMEMKR", 
        "NUM_VOL_LPN_LVN", "NUM_MED_SOCIAL_WRKS", "NUM_VOL_SOC_WORK", 
        "NUM_VOL_PHYS", "NUM_VOL_REG_NURS", "SP_COUNSELING", "SP_HOME_HEALTH_AIDE", 
        "SP_HOMEMAKER", "SP_MEDICAL_SUPPLIES", "SP_SHORT_TERM_INCARE", 
        "NUM_OTHER_VOLS", "TOT_VOLS", "FED_FUNDED_FFHC", "APPROVED_RHC_PROV_NUM", 
        "APPROVED_MEDICARE_RHC", "ACCRED_AABB_IND", "ACCRED_AOA_IND", 
        "ACCRED_ASHI_IND", "ACCRED_CAP_IND", "ACCRED_COLA_IND", "ACCRED_JCAHO_IND", 
        "ACCRED_AABB_DT", "ACCRED_AOA_DT", "ACCRED_ASHI_DT", "ACCRED_CAP_DT", 
        "ACCRED_COLA_DT", "ACCRED_JCAHO_DT", "ACCRED_AABB_MATCH_IND", 
        "ACCRED_AOA_MATCH_IND", "ACCRED_ASHI_MATCH_IND", "ACCRED_CAP_MATCH_IND", 
        "ACCRED_COLA_MATCH_IND", "ACCRED_JCAHO_MATCH_IND", "AFFIL_PROV_NUM_1", 
        "AFFIL_PROV_NUM_2", "AFFIL_PROV_NUM_3", "AFFIL_PROV_NUM_4", "AFFIL_PROV_NUM_5", 
        "AFFIL_PROV_NUM_6", "AFFIL_PROV_NUM_7", "AFFIL_PROV_NUM_8", "APPL_ACCR_ANN_TEST_VOL", 
        "APPL_ACCRED_SCHED_CD", "APPL_RECEIVED_DT", "APPL_TOT_ANN_TEST_VOL", 
        "TYPE_APPLICATION", "CERT_TYPE_CD_1", "CERT_MAILED_DT_1", "EFF_DT_1", 
        "CLIA_MEDICARE_NUM", "CLIA_LAB_CLASS_CD", "FAX_NUM", "CLIA_LAB_CLASS_CD_1", 
        "CLIA_LAB_CLASS_CD_10", "CLIA_LAB_CLASS_CD_2", "CLIA_LAB_CLASS_CD_3", 
        "CLIA_LAB_CLASS_CD_4", "CLIA_LAB_CLASS_CD_5", "CLIA_LAB_CLASS_CD_6", 
        "CLIA_LAB_CLASS_CD_7", "CLIA_LAB_CLASS_CD_8", "CLIA_LAB_CLASS_CD_9", 
        "MULTI_SITE_IND", "NON_PROFIT_IND", "TOT_NUM_NON_WAIVED_IND", 
        "NUM_CLIN_CONSULT", "NUM_DIRECTORS", "NUM_GEN_SUPER", "TOT_NUM_SITES", 
        "NUM_AFFIL_LABS", "NUM_TECH_CONSULT", "NUM_TECH_SUPER", "NUM_TEST_PERSONNEL", 
        "TOT_NUM_WAIVED_IND", "PEND_CLIA_LAB_CLASS_CD", "CLIA67_IND", 
        "SHARED_LAB_XREF_NUM", "SHARED_LAB_IND", "SURV_CERT_SCHED_CD", 
        "SURV_COMPL_SCHED_CD", "SURV_TOT_ANN_TEST_VOL", "TERM_CD", "TOT_ANN_TEST_VOL_WAIVED"
      )
    )
  } else if (year == 2008) {
    return(
      c("CATEGORY_SUBTYPE_IND", "CATEGORY", "CHOW_CNT", "CHOW_DT", 
        "CITY", "COMPL_ACCEPT_PLAN_COR", "STATUS_COMPL", "SSA_COUNTY", 
        "CROSS_REF_PROV_NUM", "FMS_SURVEY_DT_1", "SURVEY_DT_1", "ELIG_CD", 
        "FACILITY_NAME", "INTER_CARRIER_NUM", "MEDICAID_VEND_NUM", "PARTCI_DT", 
        "PRIOR_CHOW_DT", "PRIOR_INTER_CARRIER_NUM", "PROV_NUM", "RECORD_TYPE", 
        "REGION", "SKELETON_IND", "STATE_ABBREV", "SSA_STATE", "STATE_REGION_CD", 
        "STREET_ADDRESS", "PHONE_NUM", "TERM_CD_1", "EXP_DT_1", "TYPE_ACTION", 
        "TYPE_CONTROL", "ZIP_CD", "WS_FIPS_STATE", "WS_FIPS_CNTY", "WS_SSA_MSA_CD", 
        "WS_SSA_MSA_SIZE_CD", "ACCRED_EFF_DT", "ACCRED_EXP_DT", "ACCRED_STAT", 
        "NUM_BEDS", "NUM_CERT_BEDS", "NUM_CERT_RN_ANEST", "CLIA_ID_NUM_A", 
        "CLIA_ID_NUM_B", "CLIA_ID_NUM_C", "CLIA_ID_NUM_D", "CLIA_ID_NUM_E", 
        "COMPL_LSC", "COMPL_SCOPE_OF_SERV", "COMPL_TECH_PERSNL", "COMPL_24_HR_RN", 
        "CURRENT_EVER_ACCRED", "CURRENT_EVER_NON_ACCRED", "CURRENT_EVER_SWINGBED", 
        "DT_VALID_SURVEY", "NUM_DIETICIANS", "FISC_YR_END_DT", "NUM_LPN_LVN", 
        "MED_SCHL_AFF", "NUM_LAB_MED_TECHS", "MEETS_1861", "NUM_NUCL_MED_TECHS", 
        "NUM_OCCUP_THERAPISTS", "NUM_OTHER_PERSNL", "PARTICIPATING_CD", 
        "NUM_PHYS_THERAPY", "NUM_PHYS_ASSIST", "PROG_PARTCI", "PSY_UNIT_BED_SZ", 
        "PSY_UNIT_EFF_DT", "PSY_UNIT_IND", "PSY_UNIT_TERM_CD", "PSY_UNIT_TERM_DT", 
        "NUM_PSYCHOL", "NUM_RADIO_TECHS", "OVERRIDE_1", "OVERRIDE_2", 
        "OVERRIDE_3", "NUM_REG_NURS", "NUM_PHARMACIST_REG", "REHAB_UNIT_BED_SZ", 
        "REHAB_UNIT_EFF_DT", "REHAB_UNIT_IND", "REHAB_UNIT_TERM_CD", 
        "REHAB_UNIT_TERM_DT", "RES_PGM_APPR_ADA", "RES_PGM_APPR_AMA", 
        "RES_PGM_APPR_AOA", "RES_PGM_APPR_OTHER", "NUM_RESID_PHYS", "NUM_INHAL_THERAPY", 
        "SP_ACUTE_REN_DIAL", "SP_ALCOH_DRUG", "SP_AMBUL_OWNED", "SP_ANESTH", 
        "SP_AUDIO", "SP_BLOOD_BANK", "SP_BURN_UNIT", "SP_CARD_CATH_LAB", 
        "SP_OPEN_HEART_SURG", "SP_CHEMOTHER", "SP_CHIROPRATIC", "SP_CT_SCAN", 
        "SP_DENTAL", "SP_DIETARY", "SP_EMERG_DEDICATED", "SP_EMERG_DEPT", 
        "SP_GERON_SPEC", "SP_HOME_CARE_UNIT", "SP_HOSPICE", "SP_CORONARY_CARE", 
        "SP_ICU", "SP_ICU_NEONATAL", "SP_ICU_PEDIATRIC", "SP_ICU_SURG", 
        "SP_LABORATORY_ANATOM", "SP_LABORATORY_CLINIC", "SP_LTC_UNIT", 
        "SP_MRI", "SP_NEONATAL_NURS", "SP_NEURO_SURG", "SP_NUCLEAR_MED", 
        "SP_OBSTETRICS", "SP_OCCUP_THERAPY", "SP_OR_ROOMS", "SP_OPTHALMIC_SURG", 
        "SP_OPTOMETRIC", "SP_ORGAN_BANK", "SP_ORGAN_TRANS", "SP_ORTHOPEDIC_SURG", 
        "SP_OUTPAT", "SP_OUTPAT_SURG", "SP_PEDIATRIC", "SP_POS_EMIS_TOM_SCAN", 
        "SP_PHARMACY", "SP_PHYSICAL_THERAPY", "SP_POSTOP_REC_RM", "SP_PSYCHIATRIC", 
        "SP_PSY_FORENSIC", "SP_PSY_GERIATRIC", "SP_PSY_OUTPAT", "SP_PSY_CHILD_ADOL", 
        "SP_PSY_EMERG", "SP_RADIOLOGY_DIAG", "SP_RADIOLOGY_THERAPY", 
        "SP_RECON_SURG", "SP_REHABIL_CARF", "SP_REHABIL_OUTPAT", "SP_REHABIL", 
        "SP_EXTRAC_SHOCK_WAVE", "SP_SOCIAL", "SP_SPEECH_PATH", "SP_INPAT_SURG", 
        "SP_TRANS_MEDICARE", "SP_SHOCK_TRAUMA", "SP_URGENT_CARE", "SWINGBED_IND", 
        "SWINGBED_SIZE_CD", "TYPE_FACILITY", "NON_PARTICIPATING_TYPE", 
        "NUM_ACT_THER_CONTRACT", "NUM_ACT_THER_FULL_TIME", "NUM_ACT_THER_PART_TIME", 
        "NUM_ADMN_CONTRACT", "NUM_ADMN_FULL_TIME", "NUM_ADMN_PART_TIME", 
        "NUM_T18_SNF_BEDS", "NUM_T19_SNF_BEDS", "NUM_T1819_SNF_BEDS", 
        "NUM_NURSE_AID_CONTRACT", "NUM_NURSE_AID_FULL_TIME", "NUM_NURSE_AID_PART_TIME", 
        "CHRISTIAN_SCIENCE_IND", "COMPL_BEDS_PER_ROOM", "COMPL_PATIENT_ROOM_SZ", 
        "COMPL_7_DAY_RN", "NUM_DENTIST_CONTRACT", "NUM_DENTIST_FULL_TIME", 
        "NUM_DENTIST_PART_TIME", "NUM_DIET_CONTRACT", "NUM_DIET_FULL_TIME", 
        "NUM_DIET_PART_TIME", "EXPER_RESEARCH", "NUM_FOOD_SRV_CONTRACT", 
        "NUM_FOOD_SRV_FULL_TIME", "NUM_FOOD_SRV_PART_TIME", "NUM_HOUSE_CONTRACT", 
        "NUM_HOUSE_FULL_TIME", "NUM_HOUSE_PART_TIME", "NUM_VOC_NURSE_CONTRACT", 
        "NUM_VOC_NURSE_FULL_TIME", "NUM_VOC_NURSE_PART_TIME", "LTC_CROSS_REF_PROV_NUM", 
        "NUM_MED_CONTRACT", "NUM_MED_FULL_TIME", "NUM_MED_PART_TIME", 
        "NUM_MED_AID_CONTRACT", "NUM_MED_AID_FULL_TIME", "NUM_MED_AID_PART_TIME", 
        "NUM_MEN_HLTH_CONTRACT", "NUM_MEN_HLTH_FULL_TIME", "NUM_MEN_HLTH_PART_TIME", 
        "NAME_MULT_FACL_ORG", "MULT_FACL_ORG", "NUM_AID_TRNG_CONTRACT", 
        "NUM_AID_TRNG_FULL_TIME", "NUM_AID_TRNG_PART_TIME", "NUM_NURSE_ADM_CONTRACT", 
        "NUM_NURSE_ADM_FULL_TIME", "NUM_NURSE_ADM_PART_TIME", "NUM_OCC_THER_FULL_TIME", 
        "NUM_OCC_THER_CONTRACT", "NUM_OCC_AID_CONTRACT", "NUM_OCC_AID_FULL_TIME", 
        "NUM_OCC_AID_PART_TIME", "NUM_OCC_ASST_CONTRACT", "NUM_OCC_ASST_FULL_TIME", 
        "NUM_OCC_ASST_PART_TIME", "NUM_OCC_THER_PART_TIME", "ORG_FAMILY_GRP", 
        "ORG_RESID_GRP", "NUM_OTH_CONTRACT", "NUM_OTH_FULL_TIME", "NUM_OTH_PART_TIME", 
        "NUM_OTH_ACT_CONTRACT", "NUM_OTH_ACT_FULL_TIME", "NUM_OTH_ACT_PART_TIME", 
        "NUM_OTH_PHY_CONTRACT", "NUM_OTH_PHY_FULL_TIME", "NUM_OTH_PHY_PART_TIME", 
        "NUM_OTH_SOC_CONTRACT", "NUM_OTH_SOC_FULL_TIME", "NUM_OTH_SOC_PART_TIME", 
        "NUM_PHAR_CONTRACT", "NUM_PHAR_FULL_TIME", "NUM_PHAR_PART_TIME", 
        "NUM_THER_ASST_CONTRACT", "NUM_THER_ASST_FULL_TIME", "NUM_THER_ASST_PART_TIME", 
        "NUM_THER_CONTRACT", "NUM_THER_FULL_TIME", "NUM_THER_PART_TIME", 
        "NUM_THER_AID_CONTRACT", "NUM_THER_AID_FULL_TIME", "NUM_THER_AID_PART_TIME", 
        "NUM_PHYS_EXT_CONTRACT", "NUM_PHYS_EXT_FULL_TIME", "NUM_PHYS_EXT_PART_TIME", 
        "NUM_POD_CONTRACT", "NUM_POD_FULL_TIME", "NUM_POD_PART_TIME", 
        "PROV_BASED_FACILITY", "NUM_REG_NURSE_CONTRACT", "NUM_REG_NURSE_FULL_TIME", 
        "NUM_REG_NURSE_PART_TIME", "RELATED_PROV_NUM", "NUM_RN_DON_CONTRACT", 
        "NUM_RN_DON_FULL_TIME", "NUM_RN_DON_PART_TIME", "NUM_SOCIAL_CONTRACT", 
        "NUM_SOCIAL_FULL_TIME", "NUM_SOCIAL_PART_TIME", "NUM_AIDS_BEDS", 
        "NUM_ALZHEIMERS_BEDS", "NUM_DIAL_BEDS", "NUM_DIS_CHILD_BEDS", 
        "NUM_HEAD_TRAUMA_BEDS", "NUM_HOSPICE_BEDS", "NUM_HUNTING_DIS_BEDS", 
        "NUM_SPEC_REHAB_BEDS", "NUM_VENT_RESP_BEDS", "NUM_SPCH_PATH_CONTRACT", 
        "NUM_SPCH_PATH_FULL_TIME", "NUM_SPCH_PATH_PART_TIME", "SP_ACT_THER_OFF_RES", 
        "SP_ACT_THER_ON_NON_RES", "SP_ACT_THER_ON_RES", "SP_ADM_BLOOD_OFF_RES", 
        "SP_ADM_BLOOD_ON_NON_RES", "SP_ADM_BLOOD_ON_RES", "SP_CLIN_LAB_OFF_RES", 
        "SP_CLIN_LAB_ON_NON_RES", "SP_CLIN_LAB_ON_RES", "SP_DENTAL_OFF_RES", 
        "SP_DENTAL_ON_NON_RES", "SP_DENTAL_ON_RES", "SP_DIETARY_OFF_RES", 
        "SP_DIETARY_ON_NON_RES", "SP_DIETARY_ON_RES", "SP_HOUSE_KP_ON_NON_RES", 
        "SP_HOUSE_KP_OFF_RES", "SP_HOUSE_KP_ON_RES", "SP_MEN_HLTH_OFF_RES", 
        "SP_MEN_HLTH_ON_NON_RES", "SP_MEN_HLTH_ON_RES", "SP_NURSING_OFF_RES", 
        "SP_NURSING_ON_NON_RES", "SP_NURSING_ON_RES", "SP_OCC_THER_OFF_RES", 
        "SP_OCC_THER_ON_NON_RES", "SP_OCC_THER_ON_RES", "SP_OTH_ACT_OFF_RES", 
        "SP_OTH_ACT_ON_NON_RES", "SP_OTH_ACT_ON_RES", "SP_OTH_SOC_OFF_RES", 
        "SP_OTH_SOC_ON_NON_RES", "SP_OTH_SOC_ON_RES", "SP_PHARMACY_OFF_RES", 
        "SP_PHARMACY_ON_NON_RES", "SP_PHARMACY_ON_RES", "SP_PHYS_EXT_OFF_RES", 
        "SP_PHYS_EXT_ON_NON_RES", "SP_PHYS_EXT_ON_RES", "SP_PHYS_THER_OFF_RES", 
        "SP_PHYS_THER_ON_NON_RES", "SP_PHYS_THER_ON_RES", "SP_PHYS_OFF_RES", 
        "SP_PHYS_ON_NON_RES", "SP_PHYS_ON_RES", "SP_PODIATRY_OFF_RES", 
        "SP_PODIATRY_ON_NON_RES", "SP_PODIATRY_ON_RES", "SP_MED_SOC_OFF_RES", 
        "SP_MED_SOC_ON_NON_RES", "SP_MED_SOC_ON_RES", "SP_SPEECH_PH_OFF_RES", 
        "SP_SPEECH_PH_ON_NON_RES", "SP_SPEECH_PH_ON_RES", "SP_THER_REC_OFF_RES", 
        "SP_THER_REC_ON_NON_RES", "SP_THER_REC_ON_RES", "SP_VOC_GUID_OFF_RES", 
        "SP_VOC_GUID_ON_NON_RES", "SP_VOC_GUID_ON_RES", "SP_DIAG_XRAY_OFF_RES", 
        "SP_DIAG_XRAY_ON_NON_RES", "SP_DIAG_XRAY_ON_RES", "NUM_THER_REC_CONTRACT", 
        "NUM_THER_REC_FULL_TIME", "NUM_THER_REC_PART_TIME", "HHA_PROVIDES_DIRECT", 
        "OPERS_BRANCHES", "NUM_BRANCHES", "CHOW_IND", "HHA_QUAL_FOR_OPT", 
        "NUM_HOME_HEALTH_AIDES", "MEDICARE_CERT_HOSPICE", "HOSPICE_PROV_NUM", 
        "MEDICAID_CARE_VEND_NUM", "NUM_SOCIAL_WRKS", "NUM_SPEECH_PATH_AUDIO", 
        "SP_APPLIANCE_EQUIP", "SP_HH_AIDE_HOMEMAKER", "SP_INTERNS_RESIDENTS", 
        "SP_MEDICAL_SOCIAL", "SP_NURSING", "SP_NUTRITION_GUIDANCE", "SP_OTHER", 
        "SP_SPEECH_THERAPY", "SP_VOCAT_GUIDANCE", "SUBUNIT_IND", "OPERS_SUBUNITS", 
        "NUM_SUBUNITS", "SURETY_BOND_IND", "QUAL_OF_DIRECTOR", "NUM_AS_RADIO_TECH", 
        "NUM_BS_BA_RAD_TECH", "NUM_2YR_RADIO_TECH", "NUM_PHY_THER_ARGNM", 
        "NUM_PHYS_THERAPISTS", "NUM_SPEECH_PATH_AR", "NUM_SPEECH_PATH", 
        "NETWORK_NUM", "NUM_PATIENT_TUE_SHIFT_4", "NUM_STATIONS_HEMO", 
        "TOT_STATIONS", "ADMIN_SUSP_DT", "NUM_ICF_MR_BEDS", "NUM_DCARE_PERSNL", 
        "LTC_AGREE_BEGIN_DT", "LTC_AGREE_END_DT", "LTC_AGREE_EXT_DT", 
        "PRIOR_ADMIN_SUSP_DT", "PRIOR_LTC_END_DT", "PRIOR_LTC_EXT_DT", 
        "PRIOR_RESC_SUSP_DT", "RESC_SUSP_DT", "TOT_EMPLOYEES", "FED_PROG_SUPPORT", 
        "NUM_NURSE_PRACT", "PARENT_PROV_NUM", "NUM_PHYS", "TITL_FED_PROGR", 
        "PARTIC_OPT_SP", "SP_NURSING_2", "SP_NURSING_3", "SP_OCCUP_THERAPY_2", 
        "SP_OCCUP_THERAPY_3", "SP_ORTHOTIC_PROSTHET", "SP_ORTHOTIC_PROSTHET_2", 
        "SP_ORTHOTIC_PROSTHET_3", "SP_PHYSICAL_THERAPY_2", "SP_PHYSICAL_THERAPY_3", 
        "SP_PHYSICIAN", "SP_PHYSICIAN_2", "SP_PHYSICIAN_3", "SP_PSYCHOLOGICAL", 
        "SP_PSYCHOLOGICAL_2", "SP_PSYCHOLOGICAL_3", "SP_RESP_CARE", "SP_RESP_CARE_2", 
        "SP_RESP_CARE_3", "SP_SOCIAL_2", "SP_SOCIAL_3", "SP_SPEECH_PATH_2", 
        "SP_SPEECH_PATH_3", "DT_SERVICE_BEGAN", "FREE_STAND_IND", "HOSP_BASED_IND", 
        "NUM_OPERATING_ROOMS", "SP_CARDIOVASCULAR", "SP_FOOT", "SP_GENERAL", 
        "SP_NEUROLOGICAL", "SP_OBSTETR_GYNECOL", "SP_OPTHAMOLOGY_SURG", 
        "SP_ORAL", "SP_ORTHOPEDIC", "SP_OTOLARYRGOLOGY", "SP_PLASTIC", 
        "SP_THORACIC", "SP_UROLOGY", "SP_EKG", "SP_LABORATORY", "SP_RADIOLOGY", 
        "ACUTE_RESPITE", "NUM_STAFF_COUNSL", "NUM_VOL_COUNSL", "NUM_VOL_HHA", 
        "NUM_HOMEMAKERS", "NUM_VOL_HOMEMKR", "NUM_VOL_LPN_LVN", "NUM_MED_SOCIAL_WRKS", 
        "NUM_VOL_SOC_WORK", "NUM_VOL_PHYS", "NUM_VOL_REG_NURS", "SP_COUNSELING", 
        "SP_HOME_HEALTH_AIDE", "SP_HOMEMAKER", "SP_MEDICAL_SUPPLIES", 
        "SP_SHORT_TERM_INCARE", "NUM_OTHER_VOLS", "TOT_VOLS", "FED_FUNDED_FFHC", 
        "APPROVED_RHC_PROV_NUM", "APPROVED_MEDICARE_RHC", "ACCRED_AABB_IND", 
        "ACCRED_AOA_IND", "ACCRED_ASHI_IND", "ACCRED_A2LA_IND", "ACCRED_CAP_IND", 
        "ACCRED_COLA_IND", "ACCRED_JCAHO_IND", "ACCRED_AABB_DT", "ACCRED_AOA_DT", 
        "ACCRED_ASHI_DT", "ACCRED_A2LA_DT", "ACCRED_CAP_DT", "ACCRED_COLA_DT", 
        "ACCRED_JCAHO_DT", "ACCRED_AABB_MATCH_IND", "ACCRED_AOA_MATCH_IND", 
        "ACCRED_ASHI_MATCH_IND", "ACCRED_A2LA_MATCH_IND", "ACCRED_CAP_MATCH_IND", 
        "ACCRED_COLA_MATCH_IND", "ACCRED_JCAHO_MATCH_IND", "AFFIL_PROV_NUM_1", 
        "AFFIL_PROV_NUM_2", "AFFIL_PROV_NUM_3", "AFFIL_PROV_NUM_4", "AFFIL_PROV_NUM_5", 
        "AFFIL_PROV_NUM_6", "AFFIL_PROV_NUM_7", "AFFIL_PROV_NUM_8", "APPL_ACCR_ANN_TEST_VOL", 
        "APPL_ACCRED_SCHED_CD", "APPL_RECEIVED_DT", "APPL_TOT_ANN_TEST_VOL", 
        "TYPE_APPLICATION", "CERT_TYPE_CD_1", "CERT_MAILED_DT_1", "EFF_DT_1", 
        "CLIA_MEDICARE_NUM", "CLIA_LAB_CLASS_CD", "FAX_NUM", "CLIA_LAB_CLASS_CD_1", 
        "CLIA_LAB_CLASS_CD_10", "CLIA_LAB_CLASS_CD_2", "CLIA_LAB_CLASS_CD_3", 
        "CLIA_LAB_CLASS_CD_4", "CLIA_LAB_CLASS_CD_5", "CLIA_LAB_CLASS_CD_6", 
        "CLIA_LAB_CLASS_CD_7", "CLIA_LAB_CLASS_CD_8", "CLIA_LAB_CLASS_CD_9", 
        "MULTI_SITE_IND", "NON_PROFIT_IND", "TOT_NUM_SITES", "NUM_AFFIL_LABS", 
        "PEND_CLIA_LAB_CLASS_CD", "CLIA67_IND", "SHARED_LAB_XREF_NUM", 
        "SHARED_LAB_IND", "SURV_CERT_SCHED_CD", "SURV_COMPL_SCHED_CD", 
        "SURV_TOT_ANN_TEST_VOL", "TERM_CD", "TOT_ANN_TEST_VOL_PPM", "TOT_ANN_TEST_VOL_WAIVED"
      )
    )
  } else if (year == 2009) {
    return(
      c("CATEGORY_SUBTYPE_IND", "CATEGORY", "CHOW_CNT", "CHOW_DT", 
        "CITY", "COMPL_ACCEPT_PLAN_COR", "STATUS_COMPL", "SSA_COUNTY", 
        "CROSS_REF_PROV_NUM", "FMS_SURVEY_DT_1", "SURVEY_DT_1", "ELIG_CD", 
        "FACILITY_NAME", "INTER_CARRIER_NUM", "MEDICAID_VEND_NUM", "PARTCI_DT", 
        "PRIOR_CHOW_DT", "PRIOR_INTER_CARRIER_NUM", "PROV_NUM", "RECORD_TYPE", 
        "REGION", "SKELETON_IND", "STATE_ABBREV", "SSA_STATE", "STATE_REGION_CD", 
        "STREET_ADDRESS", "PHONE_NUM", "TERM_CD_1", "EXP_DT_1", "TYPE_ACTION", 
        "TYPE_CONTROL", "ZIP_CD", "WS_FIPS_STATE", "WS_FIPS_CNTY", "WS_SSA_MSA_CD", 
        "WS_SSA_MSA_SIZE_CD", "ACCRED_EFF_DT", "ACCRED_EXP_DT", "ACCRED_STAT", 
        "NUM_BEDS", "NUM_CERT_BEDS", "NUM_CERT_RN_ANEST", "CLIA_ID_NUM_A", 
        "CLIA_ID_NUM_B", "CLIA_ID_NUM_C", "CLIA_ID_NUM_D", "CLIA_ID_NUM_E", 
        "COMPL_LSC", "COMPL_SCOPE_OF_SERV", "COMPL_TECH_PERSNL", "COMPL_24_HR_RN", 
        "CURRENT_EVER_ACCRED", "CURRENT_EVER_NON_ACCRED", "CURRENT_EVER_SWINGBED", 
        "DT_VALID_SURVEY", "NUM_DIETICIANS", "FISC_YR_END_DT", "NUM_LPN_LVN", 
        "MED_SCHL_AFF", "NUM_LAB_MED_TECHS", "MEETS_1861", "NUM_NUCL_MED_TECHS", 
        "NUM_OCCUP_THERAPISTS", "NUM_OTHER_PERSNL", "PARTICIPATING_CD", 
        "NUM_PHYS_THERAPY", "NUM_PHYS_ASSIST", "PROG_PARTCI", "PSY_UNIT_BED_SZ", 
        "PSY_UNIT_EFF_DT", "PSY_UNIT_IND", "PSY_UNIT_TERM_CD", "PSY_UNIT_TERM_DT", 
        "NUM_PSYCHOL", "NUM_RADIO_TECHS", "OVERRIDE_1", "OVERRIDE_2", 
        "OVERRIDE_3", "NUM_REG_NURS", "NUM_PHARMACIST_REG", "REHAB_UNIT_BED_SZ", 
        "REHAB_UNIT_EFF_DT", "REHAB_UNIT_IND", "REHAB_UNIT_TERM_CD", 
        "REHAB_UNIT_TERM_DT", "RES_PGM_APPR_ADA", "RES_PGM_APPR_AMA", 
        "RES_PGM_APPR_AOA", "RES_PGM_APPR_OTHER", "NUM_RESID_PHYS", "NUM_INHAL_THERAPY", 
        "SP_ACUTE_REN_DIAL", "SP_ALCOH_DRUG", "SP_AMBUL_OWNED", "SP_ANESTH", 
        "SP_AUDIO", "SP_BLOOD_BANK", "SP_BURN_UNIT", "SP_CARD_CATH_LAB", 
        "SP_OPEN_HEART_SURG", "SP_CHEMOTHER", "SP_CHIROPRATIC", "SP_CT_SCAN", 
        "SP_DENTAL", "SP_DIETARY", "SP_EMERG_DEDICATED", "SP_EMERG_DEPT", 
        "SP_GERON_SPEC", "SP_HOME_CARE_UNIT", "SP_HOSPICE", "SP_CORONARY_CARE", 
        "SP_ICU", "SP_ICU_NEONATAL", "SP_ICU_PEDIATRIC", "SP_ICU_SURG", 
        "SP_LABORATORY_ANATOM", "SP_LABORATORY_CLINIC", "SP_LTC_UNIT", 
        "SP_MRI", "SP_NEONATAL_NURS", "SP_NEURO_SURG", "SP_NUCLEAR_MED", 
        "SP_OBSTETRICS", "SP_OCCUP_THERAPY", "SP_OR_ROOMS", "SP_OPTHALMIC_SURG", 
        "SP_OPTOMETRIC", "SP_ORGAN_BANK", "SP_ORGAN_TRANS", "SP_ORTHOPEDIC_SURG", 
        "SP_OUTPAT", "SP_OUTPAT_SURG", "SP_PEDIATRIC", "SP_POS_EMIS_TOM_SCAN", 
        "SP_PHARMACY", "SP_PHYSICAL_THERAPY", "SP_POSTOP_REC_RM", "SP_PSYCHIATRIC", 
        "SP_PSY_FORENSIC", "SP_PSY_GERIATRIC", "SP_PSY_OUTPAT", "SP_PSY_CHILD_ADOL", 
        "SP_PSY_EMERG", "SP_RADIOLOGY_DIAG", "SP_RADIOLOGY_THERAPY", 
        "SP_RECON_SURG", "SP_REHABIL_CARF", "SP_REHABIL_OUTPAT", "SP_REHABIL", 
        "SP_EXTRAC_SHOCK_WAVE", "SP_SOCIAL", "SP_SPEECH_PATH", "SP_INPAT_SURG", 
        "SP_TRANS_MEDICARE", "SP_SHOCK_TRAUMA", "SP_URGENT_CARE", "SWINGBED_IND", 
        "SWINGBED_SIZE_CD", "TYPE_FACILITY", "NON_PARTICIPATING_TYPE", 
        "NUM_SPEECH_PATH_AUDIO", "NUM_NURSE_PRACT", "NUM_PHYS", "SP_RESP_CARE", 
        "NUM_MED_SOCIAL_WRKS", "FAX_NUM", "NUM_ACT_THER_CONTRACT", "NUM_ACT_THER_FULL_TIME", 
        "NUM_ACT_THER_PART_TIME", "NUM_ADMN_CONTRACT", "NUM_ADMN_FULL_TIME", 
        "NUM_ADMN_PART_TIME", "NUM_T18_SNF_BEDS", "NUM_T19_SNF_BEDS", 
        "NUM_T1819_SNF_BEDS", "NUM_NURSE_AID_CONTRACT", "NUM_NURSE_AID_FULL_TIME", 
        "NUM_NURSE_AID_PART_TIME", "CHRISTIAN_SCIENCE_IND", "COMPL_BEDS_PER_ROOM", 
        "COMPL_PATIENT_ROOM_SZ", "COMPL_7_DAY_RN", "NUM_DENTIST_CONTRACT", 
        "NUM_DENTIST_FULL_TIME", "NUM_DENTIST_PART_TIME", "NUM_DIET_CONTRACT", 
        "NUM_DIET_FULL_TIME", "NUM_DIET_PART_TIME", "EXPER_RESEARCH", 
        "NUM_FOOD_SRV_CONTRACT", "NUM_FOOD_SRV_FULL_TIME", "NUM_FOOD_SRV_PART_TIME", 
        "NUM_HOUSE_CONTRACT", "NUM_HOUSE_FULL_TIME", "NUM_HOUSE_PART_TIME", 
        "NUM_VOC_NURSE_CONTRACT", "NUM_VOC_NURSE_FULL_TIME", "NUM_VOC_NURSE_PART_TIME", 
        "LTC_CROSS_REF_PROV_NUM", "NUM_MED_CONTRACT", "NUM_MED_FULL_TIME", 
        "NUM_MED_PART_TIME", "NUM_MED_AID_CONTRACT", "NUM_MED_AID_FULL_TIME", 
        "NUM_MED_AID_PART_TIME", "NUM_MEN_HLTH_CONTRACT", "NUM_MEN_HLTH_FULL_TIME", 
        "NUM_MEN_HLTH_PART_TIME", "NAME_MULT_FACL_ORG", "MULT_FACL_ORG", 
        "NUM_AID_TRNG_CONTRACT", "NUM_AID_TRNG_FULL_TIME", "NUM_AID_TRNG_PART_TIME", 
        "NUM_NURSE_ADM_CONTRACT", "NUM_NURSE_ADM_FULL_TIME", "NUM_NURSE_ADM_PART_TIME", 
        "NUM_OCC_THER_FULL_TIME", "NUM_OCC_THER_CONTRACT", "NUM_OCC_AID_CONTRACT", 
        "NUM_OCC_AID_FULL_TIME", "NUM_OCC_AID_PART_TIME", "NUM_OCC_ASST_CONTRACT", 
        "NUM_OCC_ASST_FULL_TIME", "NUM_OCC_ASST_PART_TIME", "NUM_OCC_THER_PART_TIME", 
        "ORG_FAMILY_GRP", "ORG_RESID_GRP", "NUM_OTH_CONTRACT", "NUM_OTH_FULL_TIME", 
        "NUM_OTH_PART_TIME", "NUM_OTH_ACT_CONTRACT", "NUM_OTH_ACT_FULL_TIME", 
        "NUM_OTH_ACT_PART_TIME", "NUM_OTH_PHY_CONTRACT", "NUM_OTH_PHY_FULL_TIME", 
        "NUM_OTH_PHY_PART_TIME", "NUM_OTH_SOC_CONTRACT", "NUM_OTH_SOC_FULL_TIME", 
        "NUM_OTH_SOC_PART_TIME", "NUM_PHAR_CONTRACT", "NUM_PHAR_FULL_TIME", 
        "NUM_PHAR_PART_TIME", "NUM_THER_ASST_CONTRACT", "NUM_THER_ASST_FULL_TIME", 
        "NUM_THER_ASST_PART_TIME", "NUM_THER_CONTRACT", "NUM_THER_FULL_TIME", 
        "NUM_THER_PART_TIME", "NUM_THER_AID_CONTRACT", "NUM_THER_AID_FULL_TIME", 
        "NUM_THER_AID_PART_TIME", "NUM_PHYS_EXT_CONTRACT", "NUM_PHYS_EXT_FULL_TIME", 
        "NUM_PHYS_EXT_PART_TIME", "NUM_POD_CONTRACT", "NUM_POD_FULL_TIME", 
        "NUM_POD_PART_TIME", "PROV_BASED_FACILITY", "NUM_REG_NURSE_CONTRACT", 
        "NUM_REG_NURSE_FULL_TIME", "NUM_REG_NURSE_PART_TIME", "RELATED_PROV_NUM", 
        "NUM_RN_DON_CONTRACT", "NUM_RN_DON_FULL_TIME", "NUM_RN_DON_PART_TIME", 
        "NUM_SOCIAL_CONTRACT", "NUM_SOCIAL_FULL_TIME", "NUM_SOCIAL_PART_TIME", 
        "NUM_AIDS_BEDS", "NUM_ALZHEIMERS_BEDS", "NUM_DIAL_BEDS", "NUM_DIS_CHILD_BEDS", 
        "NUM_HEAD_TRAUMA_BEDS", "NUM_HOSPICE_BEDS", "NUM_HUNTING_DIS_BEDS", 
        "NUM_SPEC_REHAB_BEDS", "NUM_VENT_RESP_BEDS", "NUM_SPCH_PATH_CONTRACT", 
        "NUM_SPCH_PATH_FULL_TIME", "NUM_SPCH_PATH_PART_TIME", "SP_ACT_THER_OFF_RES", 
        "SP_ACT_THER_ON_NON_RES", "SP_ACT_THER_ON_RES", "SP_ADM_BLOOD_OFF_RES", 
        "SP_ADM_BLOOD_ON_NON_RES", "SP_ADM_BLOOD_ON_RES", "SP_CLIN_LAB_OFF_RES", 
        "SP_CLIN_LAB_ON_NON_RES", "SP_CLIN_LAB_ON_RES", "SP_DENTAL_OFF_RES", 
        "SP_DENTAL_ON_NON_RES", "SP_DENTAL_ON_RES", "SP_DIETARY_OFF_RES", 
        "SP_DIETARY_ON_NON_RES", "SP_DIETARY_ON_RES", "SP_HOUSE_KP_ON_NON_RES", 
        "SP_HOUSE_KP_OFF_RES", "SP_HOUSE_KP_ON_RES", "SP_MEN_HLTH_OFF_RES", 
        "SP_MEN_HLTH_ON_NON_RES", "SP_MEN_HLTH_ON_RES", "SP_NURSING_OFF_RES", 
        "SP_NURSING_ON_NON_RES", "SP_NURSING_ON_RES", "SP_OCC_THER_OFF_RES", 
        "SP_OCC_THER_ON_NON_RES", "SP_OCC_THER_ON_RES", "SP_OTH_ACT_OFF_RES", 
        "SP_OTH_ACT_ON_NON_RES", "SP_OTH_ACT_ON_RES", "SP_OTH_SOC_OFF_RES", 
        "SP_OTH_SOC_ON_NON_RES", "SP_OTH_SOC_ON_RES", "SP_PHARMACY_OFF_RES", 
        "SP_PHARMACY_ON_NON_RES", "SP_PHARMACY_ON_RES", "SP_PHYS_EXT_OFF_RES", 
        "SP_PHYS_EXT_ON_NON_RES", "SP_PHYS_EXT_ON_RES", "SP_PHYS_THER_OFF_RES", 
        "SP_PHYS_THER_ON_NON_RES", "SP_PHYS_THER_ON_RES", "SP_PHYS_OFF_RES", 
        "SP_PHYS_ON_NON_RES", "SP_PHYS_ON_RES", "SP_PODIATRY_OFF_RES", 
        "SP_PODIATRY_ON_NON_RES", "SP_PODIATRY_ON_RES", "SP_MED_SOC_OFF_RES", 
        "SP_MED_SOC_ON_NON_RES", "SP_MED_SOC_ON_RES", "SP_SPEECH_PH_OFF_RES", 
        "SP_SPEECH_PH_ON_NON_RES", "SP_SPEECH_PH_ON_RES", "SP_THER_REC_OFF_RES", 
        "SP_THER_REC_ON_NON_RES", "SP_THER_REC_ON_RES", "SP_VOC_GUID_OFF_RES", 
        "SP_VOC_GUID_ON_NON_RES", "SP_VOC_GUID_ON_RES", "SP_DIAG_XRAY_OFF_RES", 
        "SP_DIAG_XRAY_ON_NON_RES", "SP_DIAG_XRAY_ON_RES", "NUM_THER_REC_CONTRACT", 
        "NUM_THER_REC_FULL_TIME", "NUM_THER_REC_PART_TIME", "HHA_PROVIDES_DIRECT", 
        "OPERS_BRANCHES", "NUM_BRANCHES", "CHOW_IND", "HHA_QUAL_FOR_OPT", 
        "NUM_HOME_HEALTH_AIDES", "MEDICARE_CERT_HOSPICE", "HOSPICE_PROV_NUM", 
        "MEDICAID_CARE_VEND_NUM", "NUM_SOCIAL_WRKS", "SP_APPLIANCE_EQUIP", 
        "SP_HH_AIDE_HOMEMAKER", "SP_INTERNS_RESIDENTS", "SP_MEDICAL_SOCIAL", 
        "SP_NURSING", "SP_NUTRITION_GUIDANCE", "SP_OTHER", "SP_SPEECH_THERAPY", 
        "SP_VOCAT_GUIDANCE", "SUBUNIT_IND", "OPERS_SUBUNITS", "NUM_SUBUNITS", 
        "SURETY_BOND_IND", "NUM_PHYS_THERAPISTS", "SP_LABORATORY", "QUAL_OF_DIRECTOR", 
        "NUM_AS_RADIO_TECH", "NUM_BS_BA_RAD_TECH", "NUM_2YR_RADIO_TECH", 
        "NUM_PHY_THER_ARGNM", "NUM_SPEECH_PATH_AR", "NUM_SPEECH_PATH", 
        "NETWORK_NUM", "NUM_PATIENT_TUE_SHIFT_4", "NUM_STATIONS_HEMO", 
        "TOT_STATIONS", "HOSP_BASED_IND", "ADMIN_SUSP_DT", "NUM_ICF_MR_BEDS", 
        "NUM_DCARE_PERSNL", "LTC_AGREE_BEGIN_DT", "LTC_AGREE_END_DT", 
        "LTC_AGREE_EXT_DT", "PRIOR_ADMIN_SUSP_DT", "PRIOR_LTC_END_DT", 
        "PRIOR_LTC_EXT_DT", "PRIOR_RESC_SUSP_DT", "RESC_SUSP_DT", "TOT_EMPLOYEES", 
        "FED_PROG_SUPPORT", "PARENT_PROV_NUM", "TITL_FED_PROGR", "PARTIC_OPT_SP", 
        "SP_NURSING_2", "SP_NURSING_3", "SP_OCCUP_THERAPY_2", "SP_OCCUP_THERAPY_3", 
        "SP_ORTHOTIC_PROSTHET", "SP_ORTHOTIC_PROSTHET_2", "SP_ORTHOTIC_PROSTHET_3", 
        "SP_PHYSICAL_THERAPY_2", "SP_PHYSICAL_THERAPY_3", "SP_PHYSICIAN", 
        "SP_PHYSICIAN_2", "SP_PHYSICIAN_3", "SP_PSYCHOLOGICAL", "SP_PSYCHOLOGICAL_2", 
        "SP_PSYCHOLOGICAL_3", "SP_RESP_CARE_2", "SP_RESP_CARE_3", "SP_SOCIAL_2", 
        "SP_SOCIAL_3", "SP_SPEECH_PATH_2", "SP_SPEECH_PATH_3", "DT_SERVICE_BEGAN", 
        "FREE_STAND_IND", "NUM_OPERATING_ROOMS", "SP_CARDIOVASCULAR", 
        "SP_FOOT", "SP_GENERAL", "SP_NEUROLOGICAL", "SP_OBSTETR_GYNECOL", 
        "SP_OPTHAMOLOGY_SURG", "SP_ORAL", "SP_ORTHOPEDIC", "SP_OTOLARYRGOLOGY", 
        "SP_PLASTIC", "SP_THORACIC", "SP_UROLOGY", "SP_EKG", "SP_RADIOLOGY", 
        "ACUTE_RESPITE", "NUM_STAFF_COUNSL", "NUM_VOL_COUNSL", "NUM_VOL_HHA", 
        "NUM_HOMEMAKERS", "NUM_VOL_HOMEMKR", "NUM_VOL_LPN_LVN", "NUM_VOL_SOC_WORK", 
        "NUM_VOL_PHYS", "NUM_VOL_REG_NURS", "SP_COUNSELING", "SP_HOME_HEALTH_AIDE", 
        "SP_HOMEMAKER", "SP_MEDICAL_SUPPLIES", "SP_SHORT_TERM_INCARE", 
        "NUM_OTHER_VOLS", "TOT_VOLS", "FED_FUNDED_FFHC", "APPROVED_RHC_PROV_NUM", 
        "APPROVED_MEDICARE_RHC", "ACCRED_AABB_IND", "ACCRED_AOA_IND", 
        "ACCRED_ASHI_IND", "ACCRED_A2LA_IND", "ACCRED_CAP_IND", "ACCRED_COLA_IND", 
        "ACCRED_JCAHO_IND", "ACCRED_AABB_DT", "ACCRED_AOA_DT", "ACCRED_ASHI_DT", 
        "ACCRED_A2LA_DT", "ACCRED_CAP_DT", "ACCRED_COLA_DT", "ACCRED_JCAHO_DT", 
        "ACCRED_AABB_MATCH_IND", "ACCRED_AOA_MATCH_IND", "ACCRED_ASHI_MATCH_IND", 
        "ACCRED_A2LA_MATCH_IND", "ACCRED_CAP_MATCH_IND", "ACCRED_COLA_MATCH_IND", 
        "ACCRED_JCAHO_MATCH_IND", "AFFIL_PROV_NUM_1", "AFFIL_PROV_NUM_2", 
        "AFFIL_PROV_NUM_3", "AFFIL_PROV_NUM_4", "AFFIL_PROV_NUM_5", "AFFIL_PROV_NUM_6", 
        "AFFIL_PROV_NUM_7", "AFFIL_PROV_NUM_8", "APPL_ACCR_ANN_TEST_VOL", 
        "APPL_ACCRED_SCHED_CD", "APPL_RECEIVED_DT", "APPL_TOT_ANN_TEST_VOL", 
        "TYPE_APPLICATION", "CERT_TYPE_CD_1", "CERT_MAILED_DT_1", "EFF_DT_1", 
        "CLIA_MEDICARE_NUM", "CLIA_LAB_CLASS_CD", "CLIA_LAB_CLASS_CD_1", 
        "CLIA_LAB_CLASS_CD_10", "CLIA_LAB_CLASS_CD_2", "CLIA_LAB_CLASS_CD_3", 
        "CLIA_LAB_CLASS_CD_4", "CLIA_LAB_CLASS_CD_5", "CLIA_LAB_CLASS_CD_6", 
        "CLIA_LAB_CLASS_CD_7", "CLIA_LAB_CLASS_CD_8", "CLIA_LAB_CLASS_CD_9", 
        "MULTI_SITE_IND", "NON_PROFIT_IND", "TOT_NUM_SITES", "NUM_AFFIL_LABS", 
        "PEND_CLIA_LAB_CLASS_CD", "CLIA67_IND", "SHARED_LAB_XREF_NUM", 
        "SHARED_LAB_IND", "SURV_CERT_SCHED_CD", "SURV_COMPL_SCHED_CD", 
        "SURV_TOT_ANN_TEST_VOL", "TERM_CD", "TOT_ANN_TEST_VOL_PPM", "TOT_ANN_TEST_VOL_WAIVED"
      )
    )
  } else if (year == 2010) {
    return(
      c("CATEGORY_SUBTYPE_IND", "CATEGORY", "CHOW_CNT", "CHOW_DT", 
        "CITY", "COMPL_ACCEPT_PLAN_COR", "STATUS_COMPL", "SSA_COUNTY", 
        "CROSS_REF_PROV_NUM", "FMS_SURVEY_DT_1", "SURVEY_DT_1", "ELIG_CD", 
        "FACILITY_NAME", "INTER_CARRIER_NUM", "MEDICAID_VEND_NUM", "PARTCI_DT", 
        "PRIOR_CHOW_DT", "PRIOR_INTER_CARRIER_NUM", "PROV_NUM", "RECORD_TYPE", 
        "REGION", "SKELETON_IND", "STATE_ABBREV", "SSA_STATE", "STATE_REGION_CD", 
        "STREET_ADDRESS", "PHONE_NUM", "TERM_CD_1", "EXP_DT_1", "TYPE_ACTION", 
        "TYPE_CONTROL", "ZIP_CD", "WS_FIPS_STATE", "WS_FIPS_CNTY", "WS_SSA_MSA_CD", 
        "WS_SSA_MSA_SIZE_CD", "ACCRED_EFF_DT", "ACCRED_EXP_DT", "ACCRED_STAT", 
        "NUM_BEDS", "NUM_CERT_BEDS", "NUM_CERT_RN_ANEST", "CLIA_ID_NUM_A", 
        "CLIA_ID_NUM_B", "CLIA_ID_NUM_C", "CLIA_ID_NUM_D", "CLIA_ID_NUM_E", 
        "COMPL_LSC", "COMPL_SCOPE_OF_SERV", "COMPL_TECH_PERSNL", "COMPL_24_HR_RN", 
        "CURRENT_EVER_ACCRED", "CURRENT_EVER_NON_ACCRED", "CURRENT_EVER_SWINGBED", 
        "DT_VALID_SURVEY", "NUM_DIETICIANS", "FISC_YR_END_DT", "NUM_LPN_LVN", 
        "MED_SCHL_AFF", "NUM_LAB_MED_TECHS", "MEETS_1861", "NUM_NUCL_MED_TECHS", 
        "NUM_OCCUP_THERAPISTS", "NUM_OTHER_PERSNL", "PARTICIPATING_CD", 
        "NUM_PHYS_THERAPY", "NUM_PHYS_ASSIST", "PROG_PARTCI", "PSY_UNIT_BED_SZ", 
        "PSY_UNIT_EFF_DT", "PSY_UNIT_IND", "PSY_UNIT_TERM_CD", "PSY_UNIT_TERM_DT", 
        "NUM_PSYCHOL", "NUM_RADIO_TECHS", "OVERRIDE_1", "OVERRIDE_2", 
        "OVERRIDE_3", "NUM_REG_NURS", "NUM_PHARMACIST_REG", "REHAB_UNIT_BED_SZ", 
        "REHAB_UNIT_EFF_DT", "REHAB_UNIT_IND", "REHAB_UNIT_TERM_CD", 
        "REHAB_UNIT_TERM_DT", "RES_PGM_APPR_ADA", "RES_PGM_APPR_AMA", 
        "RES_PGM_APPR_AOA", "RES_PGM_APPR_OTHER", "NUM_RESID_PHYS", "NUM_INHAL_THERAPY", 
        "SP_ACUTE_REN_DIAL", "SP_ALCOH_DRUG", "SP_AMBUL_OWNED", "SP_ANESTH", 
        "SP_AUDIO", "SP_BLOOD_BANK", "SP_BURN_UNIT", "SP_CARD_CATH_LAB", 
        "SP_OPEN_HEART_SURG", "SP_CHEMOTHER", "SP_CHIROPRATIC", "SP_CT_SCAN", 
        "SP_DENTAL", "SP_DIETARY", "SP_EMERG_DEDICATED", "SP_EMERG_DEPT", 
        "SP_GERON_SPEC", "SP_HOME_CARE_UNIT", "SP_HOSPICE", "SP_CORONARY_CARE", 
        "SP_ICU", "SP_ICU_NEONATAL", "SP_ICU_PEDIATRIC", "SP_ICU_SURG", 
        "SP_LABORATORY_ANATOM", "SP_LABORATORY_CLINIC", "SP_LTC_UNIT", 
        "SP_MRI", "SP_NEONATAL_NURS", "SP_NEURO_SURG", "SP_NUCLEAR_MED", 
        "SP_OBSTETRICS", "SP_OCCUP_THERAPY", "SP_OR_ROOMS", "SP_OPTHALMIC_SURG", 
        "SP_OPTOMETRIC", "SP_ORGAN_BANK", "SP_ORGAN_TRANS", "SP_ORTHOPEDIC_SURG", 
        "SP_OUTPAT", "SP_OUTPAT_SURG", "SP_PEDIATRIC", "SP_POS_EMIS_TOM_SCAN", 
        "SP_PHARMACY", "SP_PHYSICAL_THERAPY", "SP_POSTOP_REC_RM", "SP_PSYCHIATRIC", 
        "SP_PSY_FORENSIC", "SP_PSY_GERIATRIC", "SP_PSY_OUTPAT", "SP_PSY_CHILD_ADOL", 
        "SP_PSY_EMERG", "SP_RADIOLOGY_DIAG", "SP_RADIOLOGY_THERAPY", 
        "SP_RECON_SURG", "SP_REHABIL_CARF", "SP_REHABIL_OUTPAT", "SP_REHABIL", 
        "SP_EXTRAC_SHOCK_WAVE", "SP_SOCIAL", "SP_SPEECH_PATH", "SP_INPAT_SURG", 
        "SP_TRANS_MEDICARE", "SP_SHOCK_TRAUMA", "SP_URGENT_CARE", "SWINGBED_IND", 
        "SWINGBED_SIZE_CD", "TYPE_FACILITY", "NON_PARTICIPATING_TYPE", 
        "NUM_SPEECH_PATH_AUDIO", "NUM_NURSE_PRACT", "NUM_PHYS", "SP_RESP_CARE", 
        "NUM_MED_SOCIAL_WRKS", "FAX_NUM", "NUM_ACT_THER_CONTRACT", "NUM_ACT_THER_FULL_TIME", 
        "NUM_ACT_THER_PART_TIME", "NUM_ADMN_CONTRACT", "NUM_ADMN_FULL_TIME", 
        "NUM_ADMN_PART_TIME", "NUM_T18_SNF_BEDS", "NUM_T19_SNF_BEDS", 
        "NUM_T1819_SNF_BEDS", "NUM_NURSE_AID_CONTRACT", "NUM_NURSE_AID_FULL_TIME", 
        "NUM_NURSE_AID_PART_TIME", "CHRISTIAN_SCIENCE_IND", "COMPL_BEDS_PER_ROOM", 
        "COMPL_PATIENT_ROOM_SZ", "COMPL_7_DAY_RN", "NUM_DENTIST_CONTRACT", 
        "NUM_DENTIST_FULL_TIME", "NUM_DENTIST_PART_TIME", "NUM_DIET_CONTRACT", 
        "NUM_DIET_FULL_TIME", "NUM_DIET_PART_TIME", "EXPER_RESEARCH", 
        "NUM_FOOD_SRV_CONTRACT", "NUM_FOOD_SRV_FULL_TIME", "NUM_FOOD_SRV_PART_TIME", 
        "NUM_HOUSE_CONTRACT", "NUM_HOUSE_FULL_TIME", "NUM_HOUSE_PART_TIME", 
        "NUM_VOC_NURSE_CONTRACT", "NUM_VOC_NURSE_FULL_TIME", "NUM_VOC_NURSE_PART_TIME", 
        "LTC_CROSS_REF_PROV_NUM", "NUM_MED_CONTRACT", "NUM_MED_FULL_TIME", 
        "NUM_MED_PART_TIME", "NUM_MED_AID_CONTRACT", "NUM_MED_AID_FULL_TIME", 
        "NUM_MED_AID_PART_TIME", "NUM_MEN_HLTH_CONTRACT", "NUM_MEN_HLTH_FULL_TIME", 
        "NUM_MEN_HLTH_PART_TIME", "NAME_MULT_FACL_ORG", "MULT_FACL_ORG", 
        "NUM_AID_TRNG_CONTRACT", "NUM_AID_TRNG_FULL_TIME", "NUM_AID_TRNG_PART_TIME", 
        "NUM_NURSE_ADM_CONTRACT", "NUM_NURSE_ADM_FULL_TIME", "NUM_NURSE_ADM_PART_TIME", 
        "NUM_OCC_THER_FULL_TIME", "NUM_OCC_THER_CONTRACT", "NUM_OCC_AID_CONTRACT", 
        "NUM_OCC_AID_FULL_TIME", "NUM_OCC_AID_PART_TIME", "NUM_OCC_ASST_CONTRACT", 
        "NUM_OCC_ASST_FULL_TIME", "NUM_OCC_ASST_PART_TIME", "NUM_OCC_THER_PART_TIME", 
        "ORG_FAMILY_GRP", "ORG_RESID_GRP", "NUM_OTH_CONTRACT", "NUM_OTH_FULL_TIME", 
        "NUM_OTH_PART_TIME", "NUM_OTH_ACT_CONTRACT", "NUM_OTH_ACT_FULL_TIME", 
        "NUM_OTH_ACT_PART_TIME", "NUM_OTH_PHY_CONTRACT", "NUM_OTH_PHY_FULL_TIME", 
        "NUM_OTH_PHY_PART_TIME", "NUM_OTH_SOC_CONTRACT", "NUM_OTH_SOC_FULL_TIME", 
        "NUM_OTH_SOC_PART_TIME", "NUM_PHAR_CONTRACT", "NUM_PHAR_FULL_TIME", 
        "NUM_PHAR_PART_TIME", "NUM_THER_ASST_CONTRACT", "NUM_THER_ASST_FULL_TIME", 
        "NUM_THER_ASST_PART_TIME", "NUM_THER_CONTRACT", "NUM_THER_FULL_TIME", 
        "NUM_THER_PART_TIME", "NUM_THER_AID_CONTRACT", "NUM_THER_AID_FULL_TIME", 
        "NUM_THER_AID_PART_TIME", "NUM_PHYS_EXT_CONTRACT", "NUM_PHYS_EXT_FULL_TIME", 
        "NUM_PHYS_EXT_PART_TIME", "NUM_POD_CONTRACT", "NUM_POD_FULL_TIME", 
        "NUM_POD_PART_TIME", "PROV_BASED_FACILITY", "NUM_REG_NURSE_CONTRACT", 
        "NUM_REG_NURSE_FULL_TIME", "NUM_REG_NURSE_PART_TIME", "RELATED_PROV_NUM", 
        "NUM_RN_DON_CONTRACT", "NUM_RN_DON_FULL_TIME", "NUM_RN_DON_PART_TIME", 
        "NUM_SOCIAL_CONTRACT", "NUM_SOCIAL_FULL_TIME", "NUM_SOCIAL_PART_TIME", 
        "NUM_AIDS_BEDS", "NUM_ALZHEIMERS_BEDS", "NUM_DIAL_BEDS", "NUM_DIS_CHILD_BEDS", 
        "NUM_HEAD_TRAUMA_BEDS", "NUM_HOSPICE_BEDS", "NUM_HUNTING_DIS_BEDS", 
        "NUM_SPEC_REHAB_BEDS", "NUM_VENT_RESP_BEDS", "NUM_SPCH_PATH_CONTRACT", 
        "NUM_SPCH_PATH_FULL_TIME", "NUM_SPCH_PATH_PART_TIME", "SP_ACT_THER_OFF_RES", 
        "SP_ACT_THER_ON_NON_RES", "SP_ACT_THER_ON_RES", "SP_ADM_BLOOD_OFF_RES", 
        "SP_ADM_BLOOD_ON_NON_RES", "SP_ADM_BLOOD_ON_RES", "SP_CLIN_LAB_OFF_RES", 
        "SP_CLIN_LAB_ON_NON_RES", "SP_CLIN_LAB_ON_RES", "SP_DENTAL_OFF_RES", 
        "SP_DENTAL_ON_NON_RES", "SP_DENTAL_ON_RES", "SP_DIETARY_OFF_RES", 
        "SP_DIETARY_ON_NON_RES", "SP_DIETARY_ON_RES", "SP_HOUSE_KP_ON_NON_RES", 
        "SP_HOUSE_KP_OFF_RES", "SP_HOUSE_KP_ON_RES", "SP_MEN_HLTH_OFF_RES", 
        "SP_MEN_HLTH_ON_NON_RES", "SP_MEN_HLTH_ON_RES", "SP_NURSING_OFF_RES", 
        "SP_NURSING_ON_NON_RES", "SP_NURSING_ON_RES", "SP_OCC_THER_OFF_RES", 
        "SP_OCC_THER_ON_NON_RES", "SP_OCC_THER_ON_RES", "SP_OTH_ACT_OFF_RES", 
        "SP_OTH_ACT_ON_NON_RES", "SP_OTH_ACT_ON_RES", "SP_OTH_SOC_OFF_RES", 
        "SP_OTH_SOC_ON_NON_RES", "SP_OTH_SOC_ON_RES", "SP_PHARMACY_OFF_RES", 
        "SP_PHARMACY_ON_NON_RES", "SP_PHARMACY_ON_RES", "SP_PHYS_EXT_OFF_RES", 
        "SP_PHYS_EXT_ON_NON_RES", "SP_PHYS_EXT_ON_RES", "SP_PHYS_THER_OFF_RES", 
        "SP_PHYS_THER_ON_NON_RES", "SP_PHYS_THER_ON_RES", "SP_PHYS_OFF_RES", 
        "SP_PHYS_ON_NON_RES", "SP_PHYS_ON_RES", "SP_PODIATRY_OFF_RES", 
        "SP_PODIATRY_ON_NON_RES", "SP_PODIATRY_ON_RES", "SP_MED_SOC_OFF_RES", 
        "SP_MED_SOC_ON_NON_RES", "SP_MED_SOC_ON_RES", "SP_SPEECH_PH_OFF_RES", 
        "SP_SPEECH_PH_ON_NON_RES", "SP_SPEECH_PH_ON_RES", "SP_THER_REC_OFF_RES", 
        "SP_THER_REC_ON_NON_RES", "SP_THER_REC_ON_RES", "SP_VOC_GUID_OFF_RES", 
        "SP_VOC_GUID_ON_NON_RES", "SP_VOC_GUID_ON_RES", "SP_DIAG_XRAY_OFF_RES", 
        "SP_DIAG_XRAY_ON_NON_RES", "SP_DIAG_XRAY_ON_RES", "NUM_THER_REC_CONTRACT", 
        "NUM_THER_REC_FULL_TIME", "NUM_THER_REC_PART_TIME", "HHA_PROVIDES_DIRECT", 
        "OPERS_BRANCHES", "NUM_BRANCHES", "CHOW_IND", "HHA_QUAL_FOR_OPT", 
        "NUM_HOME_HEALTH_AIDES", "MEDICARE_CERT_HOSPICE", "HOSPICE_PROV_NUM", 
        "MEDICAID_CARE_VEND_NUM", "NUM_SOCIAL_WRKS", "SP_APPLIANCE_EQUIP", 
        "SP_HH_AIDE_HOMEMAKER", "SP_INTERNS_RESIDENTS", "SP_MEDICAL_SOCIAL", 
        "SP_NURSING", "SP_NUTRITION_GUIDANCE", "SP_OTHER", "SP_SPEECH_THERAPY", 
        "SP_VOCAT_GUIDANCE", "SUBUNIT_IND", "OPERS_SUBUNITS", "NUM_SUBUNITS", 
        "SURETY_BOND_IND", "NUM_PHYS_THERAPISTS", "SP_LABORATORY", "QUAL_OF_DIRECTOR", 
        "NUM_AS_RADIO_TECH", "NUM_BS_BA_RAD_TECH", "NUM_2YR_RADIO_TECH", 
        "NUM_PHY_THER_ARGNM", "NUM_SPEECH_PATH_AR", "NUM_SPEECH_PATH", 
        "NETWORK_NUM", "NUM_PATIENT_TUE_SHIFT_4", "NUM_STATIONS_HEMO", 
        "TOT_STATIONS", "HOSP_BASED_IND", "ADMIN_SUSP_DT", "NUM_ICF_MR_BEDS", 
        "NUM_DCARE_PERSNL", "LTC_AGREE_BEGIN_DT", "LTC_AGREE_END_DT", 
        "LTC_AGREE_EXT_DT", "PRIOR_ADMIN_SUSP_DT", "PRIOR_LTC_END_DT", 
        "PRIOR_LTC_EXT_DT", "PRIOR_RESC_SUSP_DT", "RESC_SUSP_DT", "TOT_EMPLOYEES", 
        "FED_PROG_SUPPORT", "PARENT_PROV_NUM", "TITL_FED_PROGR", "PARTIC_OPT_SP", 
        "SP_NURSING_2", "SP_NURSING_3", "SP_OCCUP_THERAPY_2", "SP_OCCUP_THERAPY_3", 
        "SP_ORTHOTIC_PROSTHET", "SP_ORTHOTIC_PROSTHET_2", "SP_ORTHOTIC_PROSTHET_3", 
        "SP_PHYSICAL_THERAPY_2", "SP_PHYSICAL_THERAPY_3", "SP_PHYSICIAN", 
        "SP_PHYSICIAN_2", "SP_PHYSICIAN_3", "SP_PSYCHOLOGICAL", "SP_PSYCHOLOGICAL_2", 
        "SP_PSYCHOLOGICAL_3", "SP_RESP_CARE_2", "SP_RESP_CARE_3", "SP_SOCIAL_2", 
        "SP_SOCIAL_3", "SP_SPEECH_PATH_2", "SP_SPEECH_PATH_3", "DT_SERVICE_BEGAN", 
        "FREE_STAND_IND", "NUM_OPERATING_ROOMS", "SP_CARDIOVASCULAR", 
        "SP_FOOT", "SP_GENERAL", "SP_NEUROLOGICAL", "SP_OBSTETR_GYNECOL", 
        "SP_OPTHAMOLOGY_SURG", "SP_ORAL", "SP_ORTHOPEDIC", "SP_OTOLARYRGOLOGY", 
        "SP_PLASTIC", "SP_THORACIC", "SP_UROLOGY", "SP_EKG", "SP_RADIOLOGY", 
        "ACUTE_RESPITE", "NUM_STAFF_COUNSL", "NUM_VOL_COUNSL", "NUM_VOL_HHA", 
        "NUM_HOMEMAKERS", "NUM_VOL_HOMEMKR", "NUM_VOL_LPN_LVN", "NUM_VOL_SOC_WORK", 
        "NUM_VOL_PHYS", "NUM_VOL_REG_NURS", "SP_COUNSELING", "SP_HOME_HEALTH_AIDE", 
        "SP_HOMEMAKER", "SP_MEDICAL_SUPPLIES", "SP_SHORT_TERM_INCARE", 
        "NUM_OTHER_VOLS", "TOT_VOLS", "FED_FUNDED_FFHC", "APPROVED_RHC_PROV_NUM", 
        "APPROVED_MEDICARE_RHC", "ACCRED_AABB_IND", "ACCRED_AOA_IND", 
        "ACCRED_ASHI_IND", "ACCRED_A2LA_IND", "ACCRED_CAP_IND", "ACCRED_COLA_IND", 
        "ACCRED_JCAHO_IND", "ACCRED_AABB_DT", "ACCRED_AOA_DT", "ACCRED_ASHI_DT", 
        "ACCRED_A2LA_DT", "ACCRED_CAP_DT", "ACCRED_COLA_DT", "ACCRED_JCAHO_DT", 
        "ACCRED_AABB_MATCH_IND", "ACCRED_AOA_MATCH_IND", "ACCRED_ASHI_MATCH_IND", 
        "ACCRED_A2LA_MATCH_IND", "ACCRED_CAP_MATCH_IND", "ACCRED_COLA_MATCH_IND", 
        "ACCRED_JCAHO_MATCH_IND", "AFFIL_PROV_NUM_1", "AFFIL_PROV_NUM_2", 
        "AFFIL_PROV_NUM_3", "AFFIL_PROV_NUM_4", "AFFIL_PROV_NUM_5", "AFFIL_PROV_NUM_6", 
        "AFFIL_PROV_NUM_7", "AFFIL_PROV_NUM_8", "APPL_ACCR_ANN_TEST_VOL", 
        "APPL_ACCRED_SCHED_CD", "APPL_RECEIVED_DT", "APPL_TOT_ANN_TEST_VOL", 
        "TYPE_APPLICATION", "TYPE_APPLICATION_1", "TYPE_APPLICATION_10", 
        "TYPE_APPLICATION_2", "TYPE_APPLICATION_3", "TYPE_APPLICATION_4", 
        "TYPE_APPLICATION_5", "TYPE_APPLICATION_6", "TYPE_APPLICATION_7", 
        "TYPE_APPLICATION_8", "TYPE_APPLICATION_9", "CERT_TYPE_CD_1", 
        "CERT_MAILED_DT_1", "EFF_DT_1", "CLIA_MEDICARE_NUM", "CLIA_LAB_CLASS_CD", 
        "CLIA_LAB_CLASS_CD_1", "CLIA_LAB_CLASS_CD_10", "CLIA_LAB_CLASS_CD_2", 
        "CLIA_LAB_CLASS_CD_3", "CLIA_LAB_CLASS_CD_4", "CLIA_LAB_CLASS_CD_5", 
        "CLIA_LAB_CLASS_CD_6", "CLIA_LAB_CLASS_CD_7", "CLIA_LAB_CLASS_CD_8", 
        "CLIA_LAB_CLASS_CD_9", "MULTI_SITE_IND", "NON_PROFIT_IND", "TOT_NUM_SITES", 
        "NUM_AFFIL_LABS", "PEND_CLIA_LAB_CLASS_CD", "CLIA67_IND", "SHARED_LAB_XREF_NUM", 
        "SHARED_LAB_IND", "SURV_CERT_SCHED_CD", "SURV_COMPL_SCHED_CD", 
        "SURV_TOT_ANN_TEST_VOL", "TERM_CD", "TOT_ANN_TEST_VOL_PPM", "TOT_ANN_TEST_VOL_WAIVED"
      )
    )
  } else if (year >= 2011) {
    stop("Error: years >= 2011 have named variables in raw data")
  } else if (year <= 1999) {
    stop("Error: years < 2000 have not been pre-computed. Please run pos_names_extract() on the data and layout for the year's data.")
  }
}