% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EOF.R
\name{EOF}
\alias{EOF}
\title{Empirical Orthogonal Function}
\usage{
EOF(formula, n = 1, data = NULL, B = 0, probs = c(lower = 0.025,
  mid = 0.5, upper = 0.975), rotate = FALSE, suffix = "PC",
  fill = NULL)
}
\arguments{
\item{formula}{a formula to build the matrix that will be used in the SVD
decomposition (see Details)}

\item{n}{which singular values to return (if \code{NULL}, returns all)}

\item{data}{a data.frame}

\item{B}{number of bootstrap samples used to estimate confidence intervals.
Ignored if <= 1.}

\item{probs}{the probabilities of the lower and upper values of estimated
confidence intervals. If named, it's names will be used as column names.}

\item{rotate}{if \code{TRUE}, scores and loadings will be rotated using \link{varimax}}

\item{suffix}{character to name the principal components}

\item{fill}{value to infill implicit missing values or \code{NULL} if the
data is dense.}
}
\value{
An \code{eof} object which is just a named list of \code{data.table}s
\describe{
\item{left}{data.table with left singular vectors}
\item{right}{data.table with right singular vectors}
\item{sdev}{data.table with singular values, their explained variance,
and, optionally, quantiles estimated via bootstrap}
}

There are some methods implemented
\itemize{
\item \link{summary}
\item \link{screeplot} and the equivalent \link{autoplot}
\item \link{cut.eof}
\item \link{predict}
}
}
\description{
Computes Singular Value Decomposition (also known as Principal Components
Analysis or Empirical Orthogonal Functions).
}
\details{
Singular values can be computed over matrices so \code{formula} denotes how
to build a matrix from the data. It is a formula of the form VAR ~ LEFT | RIGHT
(see \link[Formula:Formula]{Formula::Formula}) in which VAR is the variable whose values will
populate the matrix, and LEFT represent the variables used to make the rows
and RIGHT, the columns of the matrix. Think it like "VAR \emph{as a function} of
LEFT \emph{and} RIGHT". The variable combination used in this formula \emph{must} identify
an unique value in a cell.

So, for example, \code{v ~ x + y | t} would mean that ther is one value of \code{v} for
each combination of \code{x}, \code{y} and \code{t}, and that there will be one row for
each combination of \code{x} and \code{y} and one row for each \code{t}.

In the result, the left and right vectors have dimensions of the LEFT and RIGHT
part of the \code{formula}, respectively.

It is much faster to compute only some singular vectors, so is advisable not
to set n to \code{NULL}. If the irba package is installed, EOF uses
\link[irlba:irlba]{irlba::irlba} instead of \link[base:svd]{base::svd} since it's much faster.

The bootsrapping procedure follows Fisher et.al. (2016) and returns the
standard deviation of each singular value.
}
\examples{
# The Antarctic Oscillation is computed from the
# monthly geopotential height anomalies weigthed by latitude.
library(data.table)
data(geopotential)
geopotential <- copy(geopotential)
geopotential[, gh.t.w := Anomaly(gh)*sqrt(cos(lat*pi/180)),
      by = .(lon, lat, month(date))]

eof <- EOF(gh.t.w ~ lat + lon | date, 1:5, data = geopotential,
           B = 100, probs = c(low = 0.1, hig = 0.9))

# Inspect the explained variance of each component
summary(eof)
screeplot(eof)

# Keep only the 1st.
aao <- cut(eof, 1)

# AAO field
library(ggplot2)
ggplot(aao$left, aes(lon, lat, z = gh.t.w)) +
    geom_contour(aes(color = ..level..)) +
    coord_polar()

# AAO signal
ggplot(aao$right, aes(date, gh.t.w)) +
    geom_line()

# standard deviation, \% of explained variance and
# confidence intervals.
aao$sdev

# Reconstructed fields based only on the two first
# principal components
field <- predict(eof, 1:2)

# Compare it to the real field.
ggplot(field[date == date[1]], aes(lon, lat)) +
    geom_contour_fill(aes(z = gh.t.w), data = geopotential[date == date[1]]) +
    geom_contour2(aes(z = gh.t.w, linetype = factor(-sign(stat(level))))) +
    scale_fill_divergent()


}
\references{
Fisher, A., Caffo, B., Schwartz, B., & Zipunnikov, V. (2016). Fast, Exact Bootstrap Principal Component Analysis for p > 1 million. Journal of the American Statistical Association, 111(514), 846–860. http://doi.org/10.1080/01621459.2015.1062383
}
\seealso{
Other meteorology functions: \code{\link{Derivate}},
  \code{\link{GeostrophicWind}}, \code{\link{WaveFlux}},
  \code{\link{thermodynamics}}, \code{\link{waves}}
}
\concept{meteorology functions}
