% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/imput_miss_val.R
\name{impute_missing_val}
\alias{impute_missing_val}
\title{Missing value imputation}
\usage{
impute_missing_val(
  .data,
  naxis = 1,
  algorithm = "EM-SVD",
  tol = 1e-10,
  max_iter = 1000,
  simplified = FALSE,
  verbose = TRUE
)
}
\arguments{
\item{.data}{A matrix to impute the missing entries. Frequently a two-way
table of genotype means in each environment.}

\item{naxis}{The rank of the Singular Value Approximation. Defaults to \code{1}.}

\item{algorithm}{The algorithm to impute missing values. Defaults to
\code{"EM-SVD"}. Other possible values are \code{"EM-AMMI"} and
\code{"colmeans"}. See \strong{Details} section.}

\item{tol}{The convergence tolerance for the algorithm.}

\item{max_iter}{The maximum number of steps to take. If \code{max_iter} is
achieved without convergence, the algorithm will stop with a warning.}

\item{simplified}{Valid argument when \code{algorithm = "EM-AMMI"}. IF
\code{FALSE} (default), the current effects of rows and columns change from
iteration to iteration. If \code{TRUE}, the general mean and effects of
rows and columns are computed in the first iteration only, and in next
iterations uses these values.}

\item{verbose}{Logical argument. If \code{verbose = FALSE} the code will run
silently.}
}
\value{
An object of class \code{imv} with the following values:
\itemize{
\item \strong{.data} The imputed matrix
\item \strong{pc_ss} The sum of squares representing variation explained by the
principal components
\item \strong{iter} The final number of iterations.
\item \strong{Final_RMSE} The maximum change of the estimated values for missing cells in the last step of iteration.
\item \strong{final_axis} The final number of principal component axis.
\item \strong{convergence} Logical value indicating whether the modern converged.
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

Impute the missing entries of a matrix with missing values using different
algorithms. See \strong{Details} section for more details
}
\details{
\strong{\code{EM-AMMI} algorithm}

The \code{EM-AMMI} algorithm completes a data set with missing values according to both
main and interaction effects. The algorithm works as follows (Gauch and
Zobel, 1990):
\enumerate{
\item The initial values are calculated as the grand mean increased by main
effects of rows and main effects of columns. That way, the matrix of
observations is pre-filled in.
\item The parameters of the AMMI model are estimated.
\item The adjusted means are calculated based on the AMMI model with
\code{naxis} principal components.
\item The missing cells are filled with the adjusted means.
\item The root mean square error of the predicted values (\code{RMSE_p}) is
calculated with the two lasts iteration steps. If \code{RMSE_p > tol}, the
steps 2 through 5 are repeated. Declare convergence if \code{RMSE_p < tol}.
If \code{max_iter} is achieved without convergence, the algorithm will stop
with a warning.
}

\strong{\code{EM-SVD} algorithm}

The \code{EM-SVD} algorithm impute the missing entries using a low-rank Singular
Value Decomposition approximation estimated by the Expectation-Maximization
algorithm. The algorithm works as follows (Troyanskaya et al., 2001).
\enumerate{
\item Initialize all \code{NA} values to the column means.
\item Compute the first \code{naxis} terms of the SVD of the completed matrix
\item Replace the previously missing values with their approximations from the SVD
\item The root mean square error of the predicted values (\code{RMSE_p}) is
calculated with the two lasts iteration steps. If \code{RMSE_p > tol}, the
steps 2 through 3 are repeated. Declare convergence if \code{RMSE_p < tol}.
If \code{max_iter} is achieved without convergence, the algorithm will stop
with a warning.
}

\strong{\code{colmeans} algorithm}

The \code{colmeans} algorithm simply impute the missing entires using the
column mean of the respective entire. Thus, there is no iteractive process.
}
\examples{
\donttest{
library(metan)
mat <- (1:20) \%*\% t(1:10)
mat
# 10\% of missing values at random
miss_mat <- random_na(mat, prop = 10)
miss_mat
mod <- impute_missing_val(miss_mat)
mod$.data
}
}
\references{
Gauch, H. G., & Zobel, R. W. (1990). Imputing missing yield trial data.
Theoretical and Applied Genetics, 79(6), 753-761.
\doi{10.1007/BF00224240}

Troyanskaya, O., Cantor, M., Sherlock, G., Brown, P., Hastie, T., Tibshirani,
R., . Altman, R. B. (2001). Missing value estimation methods for DNA
microarrays. Bioinformatics, 17(6), 520-525.
}
