% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/OperatorCombination.R
\name{OperatorCombination}
\alias{OperatorCombination}
\alias{MutatorCombination}
\alias{dict_mutators_combine}
\alias{RecombinatorCombination}
\alias{dict_recombinators_combine}
\title{Self-Adaptive Operator Combinations}
\description{
Combines multiple operators and makes operator-configuration parameters self-adaptive.

The \code{OperatorCombination} operators combine operators for different subspaces of individuals by wraping other \code{\link{MiesOperator}}s given during construction.
Different \code{\link{MiesOperator}}s are assigned to different components or sets of components and operate on them independently of the rest of the components
or the other operators. An operator can be assigned to a single component by giving it in \code{operators} with the name of the component, or to multiple components by
giving it in \code{operators} with the name of a \emph{group}. Groups are created by the \code{groups} argument, but several default groups that catch components by type
exist.
}
\details{
Operators can be made \emph{self-adaptive} by coupling their configuration parameter values to values in individuals. This is done by giving functions in \code{adaptions}; these
functions are executed for each individual before an operator is applied, and the result given to a named operator configuration parameter.

\code{OperatorCombination} is the base class from which \code{MutatorCombination} and \code{RecombinatorCombination} inherit. The latter two are to be used for \code{\link{Mutator}} and
\code{\link{Recombinator}} objects, respectively.

Besides groups created with the \code{groups} construction argument, there are special groups that all unnamed operators fall into based on their \code{\link[paradox:Param]{Param}}
class: \code{"ParamInt"}, \code{"ParamDbl"}, \code{"ParamFct"}, and \code{"ParamLgl"}. A component of an individual that is not named directly in \code{operators} or made part of a group
in \code{groups} is automatically in one of these special groups. There is furthermore a special catch-all group \code{"ParamAny"}, which catches all components that are
are not operated directly, not in a group, and not in another special group that is itself named directly or in a group. I.e., all components that would otherwise
have no assigned operation.

\code{RecombinatorCombination} can only combine operators where \verb{$n_indivs_in} and \verb{$n_indivs_out} can be combined. This is
currently supported either when \verb{$n_indivs_in} and \verb{$n_indivs_out} for each operator are the same (but \verb{$n_indivs_in} may be unequal \verb{$n_indivs_out} in
eacho of them); or when \verb{$n_indivs_in} is equal to \verb{$n_indivs_out} for each operator and the set of all \verb{$n_indivs_in} that occur contains \code{1} and one more integer.
\verb{$n_indivs_in} and \verb{$n_indivs_out} for the resulting \code{\link{RecombinatorCombination}} operator will be set the maximum of occuring \verb{$n_indivs_in} and \verb{$n_indivs_out},
respectively.
}
\section{Supported Operand Types}{


Supported \code{\link[paradox:Param]{Param}} classes are calculated based on the supported classes of the wrapped operators.
They are frequently just the set union of supported classes, unless inference can be drawn from type-specific groups that an operator is assigned to.
If e.g. an operator that supports \code{\link[paradox:ParamDbl]{ParamDbl}} and \code{\link[paradox:ParamInt]{ParamInt}} is assigned to group \code{"ParamInt"}, and
an operator that supports \code{\link[paradox:ParamLgl]{ParamLgl}} is assigned to component \code{"a"}, then the result will support \code{\link[paradox:ParamLgl]{ParamLgl}} and
\code{\link[paradox:ParamInt]{ParamInt}} only.
}

\section{Configuration Parameters}{


The \code{OperatorCombination} has the configuration parameters of all encapsulated \code{\link{MiesOperator}}s, minus the configuration parameters that are named in the \code{adaptions}.
Configuration parameter names are prefixed with the name of the \code{\link{MiesOperator}} in the \code{operators} list.
}

\section{Dictionary}{

This \code{\link{Mutator}} can be created with the short access form \code{\link[=mut]{mut()}}
(\code{\link[=muts]{muts()}} to get a list), or through the the \link[mlr3misc:Dictionary]{dictionary}
\code{\link{dict_mutators}} in the following way:

\if{html}{\out{<div class="sourceCode">}}\preformatted{# preferred:
mut("combine", <operators>, ...)
muts("combine", <operators>, ...)  # takes vector IDs, returns list of Mutators

# long form:
dict_mutators$get("combine", <operators>, ...)
}\if{html}{\out{</div>}}


This \code{\link{Recombinator}} can be created with the short access form \code{\link[=rec]{rec()}}
(\code{\link[=recs]{recs()}} to get a list), or through the the \link[mlr3misc:Dictionary]{dictionary}
\code{\link{dict_recombinators}} in the following way:

\if{html}{\out{<div class="sourceCode">}}\preformatted{# preferred:
rec("combine", <operators>, ...)
recs("combine", <operators>, ...)  # takes vector IDs, returns list of Recombinators

# long form:
dict_recombinators$get("combine", <operators>, ...)
}\if{html}{\out{</div>}}
}

\examples{
set.seed(1)
data = data.frame(x = 0, y = 0, a = TRUE, b = "a",
  stringsAsFactors = FALSE)  # necessary for R <= 3.6
p = ps(x = p_dbl(-1, 1), y = p_dbl(-1, 1), a = p_lgl(), b = p_fct(c("a", "b")))

# Demo operators:
m0 = mut("null")  # no mutation
msmall = mut("gauss", sdev = 0.1)  # mutates to small value around 0
mbig = mut("gauss", sdev = 100)  # likely mutates to +1 or -1
mflip = mut("unif", can_mutate_to_same = FALSE)  # flips TRUE/"a" to FALSE/"b"

# original:
data

# operators by name
op = mut("combine", operators = list(x = msmall, y = mbig, a = m0, b = mflip))
op$prime(p)
op$operate(data)

# operators by type
op = mut("combine",
  operators = list(ParamDbl = msmall, ParamLgl = m0, ParamFct = mflip)
)
op$prime(p)
op$operate(data)

# the binary ParamFct 'b' counts as 'ParamLgl' when
# 'binary_fct_as_logical' is set to 'TRUE'.
op = mut("combine",
  operators = list(ParamDbl = msmall, ParamLgl = m0),
  binary_fct_as_logical = TRUE
)
op$prime(p)
op$operate(data)

# operators by type; groups can be mixed types
op = mut("combine",
  operators = list(group1 = m0, group2 = msmall, group3 = mflip),
  groups = list(group1 = c("a", "x"), group2 = "y", group3 = "b")
)
op$prime(p)
op$operate(data)

# Special type-groups can be used inside groups.
op = mut("combine",
  operators = list(group1 = m0, b = mflip),
  groups = list(group1 = c("ParamDbl", "a"))
)
op$prime(p)
op$operate(data)

# Type-groups only capture all parameters that were not caught by name.
# The special 'ParamAny' group captures all that is left.
op = mut("combine",
  operators = list(ParamAny = m0, ParamDbl = msmall, x = mbig)
)
op$prime(p)
op$operate(data)

# Configuration parameters are named by names in the 'operators' list.
op$param_set

###
# Self-adaption:
# In this example, the 'ParamDbl''s operation is changed depending on the
# value of 'b'.
op = mut("combine",
  operators = list(ParamAny = m0, ParamLgl = mflip, ParamDbl = msmall),
  adaptions = list(ParamDbl.sdev = function(x) if (x$a) 100 else 0.1)
)
op$prime(p)

data2 = data[c(1, 1, 1, 1), ]
data2$a = c(TRUE, TRUE, FALSE, FALSE)

data2
# Note the value of x$a gets used line-wise, and that it is used *before*
# being flipped here. So the first two lines get large mutations, even though
# they have 'a' 'FALSE' after the operation.
op$operate(data2)
}
\seealso{
Other base classes: 
\code{\link{FiltorSurrogate}},
\code{\link{Filtor}},
\code{\link{MiesOperator}},
\code{\link{MutatorDiscrete}},
\code{\link{MutatorNumeric}},
\code{\link{Mutator}},
\code{\link{RecombinatorPair}},
\code{\link{Recombinator}},
\code{\link{Scalor}},
\code{\link{SelectorScalar}},
\code{\link{Selector}}

Other mutators: 
\code{\link{MutatorDiscrete}},
\code{\link{MutatorNumeric}},
\code{\link{Mutator}},
\code{\link{dict_mutators_cmpmaybe}},
\code{\link{dict_mutators_erase}},
\code{\link{dict_mutators_gauss}},
\code{\link{dict_mutators_maybe}},
\code{\link{dict_mutators_null}},
\code{\link{dict_mutators_proxy}},
\code{\link{dict_mutators_sequential}},
\code{\link{dict_mutators_unif}}

Other mutator wrappers: 
\code{\link{dict_mutators_cmpmaybe}},
\code{\link{dict_mutators_maybe}},
\code{\link{dict_mutators_proxy}},
\code{\link{dict_mutators_sequential}}

Other recombinators: 
\code{\link{RecombinatorPair}},
\code{\link{Recombinator}},
\code{\link{dict_recombinators_cmpmaybe}},
\code{\link{dict_recombinators_convex}},
\code{\link{dict_recombinators_cvxpair}},
\code{\link{dict_recombinators_maybe}},
\code{\link{dict_recombinators_null}},
\code{\link{dict_recombinators_proxy}},
\code{\link{dict_recombinators_sbx}},
\code{\link{dict_recombinators_sequential}},
\code{\link{dict_recombinators_swap}},
\code{\link{dict_recombinators_xonary}},
\code{\link{dict_recombinators_xounif}}

Other recombinator wrappers: 
\code{\link{dict_recombinators_cmpmaybe}},
\code{\link{dict_recombinators_maybe}},
\code{\link{dict_recombinators_proxy}},
\code{\link{dict_recombinators_sequential}}
}
\concept{base classes}
\concept{mutator wrappers}
\concept{mutators}
\concept{recombinator wrappers}
\concept{recombinators}
\section{Super class}{
\code{\link[miesmuschel:MiesOperator]{miesmuschel::MiesOperator}} -> \code{OperatorCombination}
}
\section{Active bindings}{
\if{html}{\out{<div class="r6-active-bindings">}}
\describe{
\item{\code{operators}}{(named \code{list} of \code{\link{MiesOperator}})\cr
List of operators to apply to components of individuals, as set during construction. Read-only.}

\item{\code{groups}}{(named \code{list} of \code{character})\cr
List of groups that operators can act on, as set during construction. Read-only.}

\item{\code{adaptions}}{(named \code{list} of \code{function})\cr
List of functions used for self-adaption of operators, as set during construction. Read-only.}

\item{\code{binary_fct_as_logical}}{(\code{logical(1)})\cr
Whether to treat binary \code{\link[paradox:ParamFct]{ParamFct}} components of \code{\link[paradox:ParamSet]{ParamSet}}s as \code{\link[paradox:ParamLgl]{ParamLgl}} with respect
to the special groups \code{"ParamLgl"} and \code{"ParamFct"}, as set during construction. Read-only.}

\item{\code{on_type_not_present}}{(\code{character(1)})\cr
Action to perform during \verb{$prime()} when an operator is assigned to a type special group but there is no component available that falls in this group.
See the construction argument. Can be changed during the object's lifetime.}

\item{\code{on_name_not_present}}{(\code{character(1)})\cr
Action to perform during \verb{$prime()} when an operator is assigned to a specifically named component, but the component is not present.
See the construction argument. Can be changed during the object's lifetime.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-OperatorCombination-new}{\code{OperatorCombination$new()}}
\item \href{#method-OperatorCombination-prime}{\code{OperatorCombination$prime()}}
\item \href{#method-OperatorCombination-clone}{\code{OperatorCombination$clone()}}
}
}
\if{html}{\out{
<details open><summary>Inherited methods</summary>
<ul>
<li><span class="pkg-link" data-pkg="miesmuschel" data-topic="MiesOperator" data-id="help"><a href='../../miesmuschel/html/MiesOperator.html#method-MiesOperator-help'><code>miesmuschel::MiesOperator$help()</code></a></span></li>
<li><span class="pkg-link" data-pkg="miesmuschel" data-topic="MiesOperator" data-id="operate"><a href='../../miesmuschel/html/MiesOperator.html#method-MiesOperator-operate'><code>miesmuschel::MiesOperator$operate()</code></a></span></li>
<li><span class="pkg-link" data-pkg="miesmuschel" data-topic="MiesOperator" data-id="print"><a href='../../miesmuschel/html/MiesOperator.html#method-MiesOperator-print'><code>miesmuschel::MiesOperator$print()</code></a></span></li>
<li><span class="pkg-link" data-pkg="miesmuschel" data-topic="MiesOperator" data-id="repr"><a href='../../miesmuschel/html/MiesOperator.html#method-MiesOperator-repr'><code>miesmuschel::MiesOperator$repr()</code></a></span></li>
</ul>
</details>
}}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-OperatorCombination-new"></a>}}
\if{latex}{\out{\hypertarget{method-OperatorCombination-new}{}}}
\subsection{Method \code{new()}}{
Initialize the \code{OperatorCombination} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{OperatorCombination$new(
  operators,
  groups = list(),
  adaptions = list(),
  binary_fct_as_logical = FALSE,
  on_type_not_present = "warn",
  on_name_not_present = "stop",
  granularity = 1,
  dict_entry = NULL,
  dict_shortaccess = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{operators}}{(named \code{list} of \code{\link{MiesOperator}})\cr
List of operators to apply to components of individuals. Names are either names of individual components, or group names which are either as defined
through \code{groups} or special groups. Individual components can only be member of either a (non-special) group or named in \code{operators}, so a name
that occurs in \code{operators} may not be a member of a group as defined in \code{groups}.\cr
The \verb{$operators} field will reflect this value.}

\item{\code{groups}}{(named \code{list} of \code{character})\cr
List of groups that operators can act on. Names of this list define new groups. The content of each list element contains the names of
components or special groups (a \code{\link[paradox:Param]{Param}} subclass name or \code{"ParamAny"}) to subsume under the group.
Individual components can only be member of either a (non-special) group or named in \code{operators}, so a name
that occurs in \code{operators} may not be a member of a group as defined in \code{groups}. The default is the empty list.\cr
The \verb{$groups} field will reflect this value.}

\item{\code{adaptions}}{(named \code{list} of \code{function})\cr
List of functions used for self-adaption of operators. The names of the list must be names of configuration parameters of wrapped operators, prefixed
with the corresponding name in the \code{operators} list. This is the same name as the configuration parameter would otherwise have if exposed by the
\code{OperatorCombination} object. The values in the list must be functions that receive a single input, the individual or individuals being operated on,
as a \code{\link[data.table:data.table]{data.table}}. It must return a value that is then assigned to the configuration parameter of the operator to which it pertains.
Note that \code{\link{MutatorCombination}} adaption functions are always called with a \code{\link[data.table:data.table]{data.table}} containing a single row, while
\code{\link{RecombinatorCombination}} adaption functions are called with \code{\link[data.table:data.table]{data.table}}s with multiple rows according to \verb{$n_indivs_in}.
In both cases, the return value must be a scalar. The default is the empty list.\cr
The \verb{$adaption} field will reflect this value.}

\item{\code{binary_fct_as_logical}}{(\code{logical(1)})\cr
Whether to treat binary \code{\link[paradox:ParamFct]{ParamFct}} components of \code{\link[paradox:ParamSet]{ParamSet}}s as \code{\link[paradox:ParamLgl]{ParamLgl}} with respect
to the special groups \code{"ParamLgl"} and \code{"ParamFct"}. This does \emph{not} perform any conversion, so a \code{\link{MiesOperator}} assigned to the \code{"ParamLgl"} special
group when \code{binary_fct_as_logical} is \code{TRUE} and there are binary \code{\link[paradox:ParamFct]{ParamFct}}s present will receive
a factorial value and must also support \code{\link[paradox:ParamFct]{ParamFct}} in this case. This is checked during \verb{$prime()}, but not during construction.
Default is \code{FALSE}.\cr
The \verb{$binary_fct_as_logical} field will reflect this value.}

\item{\code{on_type_not_present}}{(\code{character(1)})\cr
Action to perform during \verb{$prime()} when an operator is assigned to a type special group but there is no component available that falls in this group, either
because no components of the respective type are present, or because all these components are also directly named in \code{operators} or in \code{groups}.
One of \code{"quiet"} (do nothing), \code{"warn"} (give warning, default), or \code{"stop"} (generate an error).\cr
The writable \verb{$on_type_not_present} field will reflect this value.}

\item{\code{on_name_not_present}}{(\code{character(1)})\cr
Action to perform during \verb{$prime()} when an operator is assigned to a specifically named component, but the component is not present.
One of \code{"quiet"} (do nothing), \code{"warn"} (give warning), or \code{"stop"} (generate an error, default).\cr
The writable \verb{$on_name_not_present} field will reflect this value.}

\item{\code{granularity}}{(\code{integer(1)})\cr
At what granularity to query \code{adaptions} for sets of individuals. Functions in \code{adaptions} are always called once per \code{granularity} individuals
in input \code{values}, and the function argument in these calls will then have \code{granularity} number of rows. This is used internally, it is set to
1 for \code{MutatorCombination}, and to \verb{$n_indivs_in} for \code{RecombinatorCombination}.}

\item{\code{dict_entry}}{(\code{character(1)} | \code{NULL})\cr
Key of the class inside the \code{\link[mlr3misc:Dictionary]{Dictionary}} (usually one of
\code{\link{dict_mutators}}, \code{\link{dict_recombinators}}, \code{\link{dict_selectors}}), where it can
be retrieved using a \link[=mut]{short access function}. May be \code{NULL} if the operator
is not entered in a dictionary.\cr
The \verb{$dict_entry} field will reflect this value.}

\item{\code{dict_shortaccess}}{(\code{character(1)} | \code{NULL})\cr
Name of the \code{\link[mlr3misc:Dictionary]{Dictionary}} short access function in which the operator is registered.
This is used to inform the user about how to construct a given object. Should ordinarily be one of
\code{"mut"}, \code{"rec"}, \code{"sel"}.\cr
The \verb{$dict_shortaccess} field will reflect this value.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-OperatorCombination-prime"></a>}}
\if{latex}{\out{\hypertarget{method-OperatorCombination-prime}{}}}
\subsection{Method \code{prime()}}{
See \code{\link{MiesOperator}} method. Primes both this operator, as well as the wrapped operators
given to \code{operators} during construction. Priming of wrapped operators happens according
to component assignments to wrapped operators.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{OperatorCombination$prime(param_set)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{param_set}}{(\code{\link[paradox:ParamSet]{ParamSet}})\cr
Passed to \code{\link{MiesOperator}}\verb{$prime()}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\link{invisible} \code{self}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-OperatorCombination-clone"></a>}}
\if{latex}{\out{\hypertarget{method-OperatorCombination-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{OperatorCombination$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
\section{Super classes}{
\code{\link[miesmuschel:MiesOperator]{miesmuschel::MiesOperator}} -> \code{\link[miesmuschel:OperatorCombination]{miesmuschel::OperatorCombination}} -> \code{MutatorCombination}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-MutatorCombination-new}{\code{MutatorCombination$new()}}
\item \href{#method-MutatorCombination-clone}{\code{MutatorCombination$clone()}}
}
}
\if{html}{\out{
<details open><summary>Inherited methods</summary>
<ul>
<li><span class="pkg-link" data-pkg="miesmuschel" data-topic="MiesOperator" data-id="help"><a href='../../miesmuschel/html/MiesOperator.html#method-MiesOperator-help'><code>miesmuschel::MiesOperator$help()</code></a></span></li>
<li><span class="pkg-link" data-pkg="miesmuschel" data-topic="MiesOperator" data-id="operate"><a href='../../miesmuschel/html/MiesOperator.html#method-MiesOperator-operate'><code>miesmuschel::MiesOperator$operate()</code></a></span></li>
<li><span class="pkg-link" data-pkg="miesmuschel" data-topic="MiesOperator" data-id="print"><a href='../../miesmuschel/html/MiesOperator.html#method-MiesOperator-print'><code>miesmuschel::MiesOperator$print()</code></a></span></li>
<li><span class="pkg-link" data-pkg="miesmuschel" data-topic="MiesOperator" data-id="repr"><a href='../../miesmuschel/html/MiesOperator.html#method-MiesOperator-repr'><code>miesmuschel::MiesOperator$repr()</code></a></span></li>
<li><span class="pkg-link" data-pkg="miesmuschel" data-topic="OperatorCombination" data-id="prime"><a href='../../miesmuschel/html/OperatorCombination.html#method-OperatorCombination-prime'><code>miesmuschel::OperatorCombination$prime()</code></a></span></li>
</ul>
</details>
}}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-MutatorCombination-new"></a>}}
\if{latex}{\out{\hypertarget{method-MutatorCombination-new}{}}}
\subsection{Method \code{new()}}{
Initialize the \code{MutatorCombination} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{MutatorCombination$new(
  operators = list(),
  groups = list(),
  adaptions = list(),
  binary_fct_as_logical = FALSE,
  on_type_not_present = "warn",
  on_name_not_present = "stop"
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{operators}}{see above.}

\item{\code{groups}}{see above.}

\item{\code{adaptions}}{see above.}

\item{\code{binary_fct_as_logical}}{see above.}

\item{\code{on_type_not_present}}{see above.}

\item{\code{on_name_not_present}}{see above.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-MutatorCombination-clone"></a>}}
\if{latex}{\out{\hypertarget{method-MutatorCombination-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{MutatorCombination$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
\section{Super classes}{
\code{\link[miesmuschel:MiesOperator]{miesmuschel::MiesOperator}} -> \code{\link[miesmuschel:OperatorCombination]{miesmuschel::OperatorCombination}} -> \code{RecombinatorCombination}
}
\section{Active bindings}{
\if{html}{\out{<div class="r6-active-bindings">}}
\describe{
\item{\code{n_indivs_in}}{(\code{integer(1)})\cr
Number of individuals to consider at the same time. When operating, the number of input individuals must be divisible by this number.}

\item{\code{n_indivs_out}}{(\code{integer(1)})\cr
Number of individuals produced for each group of \verb{$n_indivs_in} individuals.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-RecombinatorCombination-new}{\code{RecombinatorCombination$new()}}
\item \href{#method-RecombinatorCombination-clone}{\code{RecombinatorCombination$clone()}}
}
}
\if{html}{\out{
<details open><summary>Inherited methods</summary>
<ul>
<li><span class="pkg-link" data-pkg="miesmuschel" data-topic="MiesOperator" data-id="help"><a href='../../miesmuschel/html/MiesOperator.html#method-MiesOperator-help'><code>miesmuschel::MiesOperator$help()</code></a></span></li>
<li><span class="pkg-link" data-pkg="miesmuschel" data-topic="MiesOperator" data-id="operate"><a href='../../miesmuschel/html/MiesOperator.html#method-MiesOperator-operate'><code>miesmuschel::MiesOperator$operate()</code></a></span></li>
<li><span class="pkg-link" data-pkg="miesmuschel" data-topic="MiesOperator" data-id="print"><a href='../../miesmuschel/html/MiesOperator.html#method-MiesOperator-print'><code>miesmuschel::MiesOperator$print()</code></a></span></li>
<li><span class="pkg-link" data-pkg="miesmuschel" data-topic="MiesOperator" data-id="repr"><a href='../../miesmuschel/html/MiesOperator.html#method-MiesOperator-repr'><code>miesmuschel::MiesOperator$repr()</code></a></span></li>
<li><span class="pkg-link" data-pkg="miesmuschel" data-topic="OperatorCombination" data-id="prime"><a href='../../miesmuschel/html/OperatorCombination.html#method-OperatorCombination-prime'><code>miesmuschel::OperatorCombination$prime()</code></a></span></li>
</ul>
</details>
}}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-RecombinatorCombination-new"></a>}}
\if{latex}{\out{\hypertarget{method-RecombinatorCombination-new}{}}}
\subsection{Method \code{new()}}{
Initialize the \code{RecombinatorCombination} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{RecombinatorCombination$new(
  operators = list(),
  groups = list(),
  adaptions = list(),
  binary_fct_as_logical = FALSE,
  on_type_not_present = "warn",
  on_name_not_present = "stop"
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{operators}}{see above.}

\item{\code{groups}}{see above.}

\item{\code{adaptions}}{see above.}

\item{\code{binary_fct_as_logical}}{see above.}

\item{\code{on_type_not_present}}{see above.}

\item{\code{on_name_not_present}}{see above.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-RecombinatorCombination-clone"></a>}}
\if{latex}{\out{\hypertarget{method-RecombinatorCombination-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{RecombinatorCombination$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
