% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/multilevel.cor.R
\name{multilevel.cor}
\alias{multilevel.cor}
\title{Within-Group and Between-Group Correlation Matrix}
\usage{
multilevel.cor(x, cluster, within = NULL, between = NULL, estimator = c("ML", "MLR"),
               missing = c("listwise", "fiml"), sig = FALSE, alpha = 0.05,
               print = c("all", "cor", "se", "stat", "p"), split = FALSE,
               tri = c("both", "lower", "upper"), tri.lower = TRUE,
               p.adj = c("none", "bonferroni", "holm", "hochberg", "hommel",
                         "BH", "BY", "fdr"), digits = 2, p.digits = 3,
               as.na = NULL, write = NULL, check = TRUE, output = TRUE)
}
\arguments{
\item{x}{a matrix or data frame.}

\item{cluster}{a vector representing the nested grouping structure (i.e.,
group or cluster variable).}

\item{within}{a character vector representing variables that are measured
on the within level and modeled only on the within level.
Variables not mentioned in \code{within} or \code{between}
are measured on the within level and will be modeled on both
the within and between level.}

\item{between}{a character vector representing variables that are measured
on the between level and modeled only on the between level.
Variables not mentioned in \code{within} or \code{between}
are measured on the within level and will be modeled on
both the within and between level.}

\item{estimator}{a character string indicating the estimator to be used:
\code{"ML"} (default) for maximum likelihood with
conventional standard errors and \code{"MLR"} for maximum
likelihood with Huber-White robust standard errors. Note
that by default, full information maximum likelihood (FIML)
method is used to deal with missing data when using
\code{"ML"} (\code{missing = "fiml"}), whereas incomplete
cases are removed listwise (i.e., \code{missing = "listwise"})
when using \code{"MLR"}.}

\item{missing}{a character string indicating how to deal with missing
data, i.e., \code{"listwise"} for listwise deletion or
\code{"fiml"} (default) for full information maximum
likelihood (FIML) method. Note that FIML method is only
available when \code{estimator = "ML"}. Note that it takes
longer to estimate the model when using FIML and using FIML
might cause issues in model convergence, these issues might
be resolved by switching to listwise deletion.}

\item{sig}{logical: if \code{TRUE}, statistically significant
correlation coefficients are shown in boldface on the
console.}

\item{alpha}{a numeric value between 0 and 1 indicating the significance
level at which correlation coefficients are printed
boldface when \code{sig = TRUE}.}

\item{print}{a character string or character vector indicating which
results to show on the console, i.e. \code{"all"} for all
results, \code{"cor"} for correlation coefficients,
\code{"se"} for standard errors, \code{"stat"} for z test
statistics, and \code{"p"} for \emph{p}-values.}

\item{split}{logical: if \code{TRUE}, output table is split in
within-group and between-group correlation matrix.}

\item{tri}{a character string indicating which triangular of the
matrix to show on the console when \code{split = TRUE},
i.e., \code{both} for upper and \code{upper} for the upper
triangular.}

\item{tri.lower}{logical: if \code{TRUE} (default) and \code{split = FALSE}
(default), within-group correlations are shown in the lower
triangular and between-group correlation are shown in the
upper triangular.}

\item{p.adj}{a character string indicating an adjustment method for
multiple testing based on \code{\link{p.adjust}}, i.e.,
\code{none} (default), \code{bonferroni}, \code{holm},
\code{hochberg}, \code{hommel}, \code{BH}, \code{BY}, or
\code{fdr}.}

\item{digits}{an integer value indicating the number of decimal places
to be used for displaying correlation coefficients.}

\item{p.digits}{an integer value indicating the number of decimal places
to be used for displaying \emph{p}-values.}

\item{as.na}{a numeric vector indicating user-defined missing values,
i.e. these values are converted to \code{NA} before
conducting the analysis. Note that \code{as.na()} function
is only applied to \code{x} but not to \code{cluster}.}

\item{write}{a character string for writing the results into a Excel file
naming a file with or without file extension '.xlsx', e.g.,
\code{"Results.xlsx"} or \code{"Results"}.}

\item{check}{logical: if \code{TRUE}, argument specification is checked.}

\item{output}{logical: if \code{TRUE}, output is shown on the console.}
}
\value{
Returns an object of class \code{misty.object}, which is a list with following
entries:
  \item{\code{call}}{function call}
  \item{\code{type}}{type of analysis}
  \item{\code{data}}{data frame specified in \code{x} including the group variable
                     specified in \code{cluster}}
  \item{\code{args}}{specification of function arguments}
  \item{\code{model.fit}}{fitted lavaan object (\code{model.fit})}
  \item{\code{result}}{list with result tables, i.e., \code{summary} for the
                       specification of the estimation method and missing data
                       handling in lavaan, \code{wb.cor} for the within- and
                       between-group correlations, \code{wb.se} for the standard
                       error of the within- and between-group correlations,
                       \code{wb.stat} for the test statistic of within- and between-group
                       correlations, \code{wb.p} for the significance value of
                       the within- and between-group correlations, \code{with.cor}
                       for the within-group correlations, \code{with.se} for the
                       standard error of the within-group correlations, \code{with.stat}
                       for the test statistic of within-group correlations, \code{with.p}
                       for the significance value of the within-group correlations,
                       \code{betw.cor} for the between-group correlations, \code{betw.se}
                       for the standard error of the between-group correlations,
                       \code{betw.stat} for the test statistic of between-group
                       correlations, \code{betw.p} for the significance value of
                       the between-group correlations}
}
\description{
This function is a wrapper function for computing the within-group and
between-group correlation matrix by calling the \code{sem} function in the
R package \pkg{lavaan} and provides standard errors, z test statistics, and
significance values (\emph{p}-values) for testing the hypothesis
H0: \eqn{\rho} = 0 for all pairs of variables within and between groups.
}
\details{
The specification of the within-group and between-group variables is in line
with the syntax in Mplus. That is, the \code{within} argument is used to
identify the variables in the matrix or data frame specified in \code{x} that
are measured on the individual level and modeled only on the within level.
They are specified to have no variance in the between part of the model. The
\code{between} argument is used to identify the variables in the matrix or
data frame specified in \code{x} that are measured on the cluster level and
modeled only on the between level. Variables not mentioned in the arguments
\code{within} or \code{between} are measured on the individual level and will
be modeled on both the within and between level.

The function uses maximum likelihood estimation with conventional standard
errors (\code{estimator = "ML"}) which are not robust against non-normality
and full information maximum likelihood (FIML) method (\code{missing = "fiml"})
to deal with missing data by default. FIML method cannot be used when
within-group variables have no variance within some clusters. In this cases,
the function
will switch to listwise deletion. Note that the current lavaan version 0.6-11
supports FIML method only for maximum likelihood estimation with conventional
standard errors (\code{estimator = "ML"}) in multilevel models. Maximum
likelihood estimation with Huber-White robust standard errors
(\code{estimator = "MLR"}) uses listwise deletion to deal with missing data.
When using FIML method there might be issues in model convergence, which might
be resolved by switching to listwise deletion (\code{missing = "listwise"}).

The lavaan package uses a quasi-Newton optimization method (\code{"nlminb"})
by default. If the optimizer does not converge, model estimation will switch
to the Expectation Maximization (EM) algorithm.

Statistically significant correlation coefficients can be shown in boldface
on the console when specifying \code{sig = TRUE}. However, this option is not
supported when using R Markdown, i.e., the argument \code{sig} will switch to
\code{FALSE}.

Adjustment method for multiple testing when specifying the argument \code{p.adj}
is applied to the within-group and between-group correlation matrix separately.
}
\note{
The function uses the functions \code{sem}, \code{lavInspect},
\code{lavMatrixRepresentation}, \code{lavTech}, \code{parameterEstimates},
and \code{standardizedsolution} provided in the R package \pkg{lavaan} by
Yves Rosseel (2012).
}
\examples{
\dontrun{
# Load data set "Demo.twolevel" in the lavaan package
data("Demo.twolevel", package = "lavaan")

#---------------------------
# All variables modeled on both the within and between level
multilevel.cor(Demo.twolevel[, c("y1", "y2", "y3")],
               cluster = Demo.twolevel$cluster)

# Highlight statistically significant result at alpha = 0.05
multilevel.cor(Demo.twolevel[, c("y1", "y2", "y3")], sig = TRUE,
              cluster = Demo.twolevel$cluster)

# Split output table in within-group and between-group correlation matrix.
multilevel.cor(Demo.twolevel[, c("y1", "y2", "y3")],
               cluster = Demo.twolevel$cluster, split = TRUE)

# Print correlation coefficients, standard errors, z test statistics,
# and p-values
multilevel.cor(Demo.twolevel[, c("y1", "y2", "y3")],
               cluster = Demo.twolevel$cluster, print = "all")

# Print correlation coefficients and p-values
# significance values with Bonferroni correction
multilevel.cor(Demo.twolevel[, c("y1", "y2", "y3")],
               cluster = Demo.twolevel$cluster, print = c("cor", "p"),
               p.adj = "bonferroni")

#---------------------------
# Variables "y1", "y2", and "y2" modeled on both the within and between level
# Variables "w1" and "w2" modeled on the cluster level
multilevel.cor(Demo.twolevel[, c("y1", "y2", "y3", "w1", "w2")],
               cluster = Demo.twolevel$cluster,
               between = c("w1", "w2"))

#---------------------------
# Variables "y1", "y2", and "y2" modeled only on the within level
# Variables "w1" and "w2" modeled on the cluster level
multilevel.cor(Demo.twolevel[, c("y1", "y2", "y3", "w1", "w2")],
               cluster = Demo.twolevel$cluster,
               within = c("y1", "y2", "y3"), between = c("w1", "w2"))

# Summary of the multilevel model used to compute the within-group
# and between-group correlation matrix
mod <- multilevel.cor(Demo.twolevel[, c("y1", "y2", "y3")],
                      cluster = Demo.twolevel$cluster, output = FALSE)
lavaan::summary(mod$model.fit, standardized = TRUE)

# Write Results into a Excel file
multilevel.cor(Demo.twolevel[, c("y1", "y2", "y3")],
               cluster = Demo.twolevel$cluster,
               write = "Multilevel_Correlation.xlsx")

result <- multilevel.cor(Demo.twolevel[, c("y1", "y2", "y3")],
                         cluster = Demo.twolevel$cluster, output = FALSE)
write.result(result, "Multilevel_Correlation.xlsx")
}
}
\references{
Hox, J., Moerbeek, M., & van de Schoot, R. (2018). \emph{Multilevel analysis:
Techniques and applications} (3rd. ed.). Routledge.

Snijders, T. A. B., & Bosker, R. J. (2012). \emph{Multilevel analysis: An
introduction to basic and advanced multilevel modeling} (2nd ed.). Sage
Publishers.
}
\seealso{
\code{\link{write.result}}, \code{\link{multilevel.descript}},
\code{\link{multilevel.icc}}, \code{\link{cluster.scores}}
}
\author{
Takuya Yanagida \email{takuya.yanagida@univie.ac.at}
}
