% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/result.lca.R
\name{result.lca}
\alias{result.lca}
\title{Summary Result Table and Bar Charts for Latent Class Analysis Estimated in Mplus}
\usage{
result.lca(folder = getwd(), exclude = NULL, sort.n = TRUE, sort.p = TRUE,
           plot = FALSE, group.ind = TRUE, ci = TRUE, conf.level = 0.95, adjust = TRUE,
           axis.title = 7, axis.text = 7, levels = NULL, labels = NULL,
           ylim = NULL, ylab = "Mean Value", breaks = ggplot2::waiver(),
           error.width = 0.1, legend.title = 7, legend.text = 7, legend.key.size = 0.4,
           gray = FALSE, start = 0.15, end = 0.85, dpi = 600,
           width = "n.ind", height = 4, digits = 1, p.digits = 3,
           write = NULL, check = TRUE, output = TRUE)
}
\arguments{
\item{folder}{a character vector indicating the name of the subfolders
to be excluded from the summary result table.}

\item{exclude}{a character vector indicating the name of the subfolders
excluded from the result tables.}

\item{sort.n}{logical: if \code{TRUE} (default), result table is sorted
according to the number of classes within each folder.}

\item{sort.p}{logical: if \code{TRUE} (default), class proportions are
sorted decreasing.}

\item{plot}{logical: if \code{TRUE}, bar charts with
error bars for confidence intervals are saved
in the  folder \code{_Plots} within subfolders. Note
that plots are only available for LCA with continuous
or count indicator variables.}

\item{group.ind}{logical: if \code{TRUE} (default), latent class indicators
are represented by separate bars, if \code{FALSE} latent classes
are represented by separate bars.}

\item{ci}{logical: if \code{TRUE} (default), confidence intervals
are added to the bar charts.}

\item{conf.level}{a numeric value between 0 and 1 indicating the confidence
level of the interval.}

\item{adjust}{logical: if \code{TRUE} (default), difference-adjustment
for the confidence intervals is applied.}

\item{axis.title}{a numeric value specifying the size of the axis title.}

\item{axis.text}{a numeric value specifying the size of the axis text}

\item{levels}{a character string specifying the order of the indicator
variables shown on the x-axis.}

\item{labels}{a character string specifying the labels of the indicator
variables shown on the x-axis.}

\item{ylim}{a numeric vector of length two specifying limits of the
y-axis.}

\item{ylab}{a character string specifying the label of the y-axis.}

\item{breaks}{a numeric vector specifying the points at which tick-marks
are drawn at the y-axis.}

\item{error.width}{a numeric vector specifying the width of the error bars.
By default, the width of the error bars is 0.1 plus
number of classes divided by 30.}

\item{legend.title}{a numeric value specifying the size of the legend title.}

\item{legend.text}{a numeric value specifying the size of the legend text.}

\item{legend.key.size}{a numeric value specifying the size of the legend keys.}

\item{gray}{logical: if \code{TRUE}, bar charts are drawn in gray
scale.}

\item{start}{a numeric value between 0 and 1 specifying the gray value
at the low end of the palette.}

\item{end}{a numeric value between 0 and 1 specifying the gray value
at the high end of the palette.}

\item{dpi}{a numeric value specifying the plot resolution when saving
the bar chart.}

\item{width}{a numeric value specifying the width of the plot when
saving the bar chart. By default, the width is number of
indicators plus number of classes divided by 2.}

\item{height}{a numeric value specifying the height of the plot when
saving the bar chart.}

\item{digits}{an integer value indicating the number of decimal places
to be used for displaying results. Note that the scaling
correction factor is displayed  with \code{digits} plus 1
decimal places.}

\item{p.digits}{an integer value indicating the number of decimal places
to be used for displaying \emph{p}-values, entropy value,
and class proportions.}

\item{write}{a character string for writing the results into an Excel
file naming a file with or without file extension '.xlsx',
e.g., \code{"Results.xlsx"} or \code{"Results"}.}

\item{check}{logical: if \code{TRUE}, argument specification is checked.}

\item{output}{logical: if \code{TRUE}, output is shown.}
}
\value{
Returns an object, which is a list with following entries:

\item{\code{call}}{function call}
\item{\code{type}}{type of analysis}
\item{\code{output}}{list with all Mplus outputs}
\item{\code{args}}{specification of function arguments}
\item{\code{result}}{list with result tables, i.e., \code{summary} for the
                     summary result table, \code{mean_var} for the result
                     table with means and variances for each latent class
                     separately, \code{mean} for the result table with means
                     for each latent class separately, and \code{var} for the
                     result table with variances for each latent class separately}
}
\description{
This function reads all Mplus output files from latent class analysis in
subfolders to create a summary result table and bar charts for each latent
class solution separately. By default, the function reads output files in all
subfolders of the current working directory. Optionally, bar charts for each
latent class solution can be requested by setting the argument \code{plot}
to \code{TRUE}. Note that subfolders with only one Mplus output file are
excluded.
}
\details{
The result summary table comprises following entries:
\itemize{
   \item{\code{"Folder"}}: Subfolder from which the group of Mplus outputs files
                           were summarized.
   \item{\code{"#Class"}}: Number of classes (i.e., \code{CLASSES ARE c(#Class)}).
   \item{\code{"Conv"}}: Model converged, \code{TRUE} or \code{FALSE} (i.e.,
                         \code{THE MODEL ESTIMATION TERMINATED NORMALLY}.
   \item{\code{"#Param"}}: Number of estimated parameters (i.e.,
                           \code{Number of Free Parameters}).
   \item{\code{"logLik"}}: Log-likelihood of the estimated model (i.e., \code{H0 Value}).
   \item{\code{"Scale"}}: Scaling correction factor (i.e.,
                          \code{H0 Scaling Correction Factor for}). Provided
                          only when \code{ESTIMATOR IS MLR}.
   \item{\code{"LL Rep"}}: Best log-likelihood replicated, \code{TRUE} or \code{FALSE}
                           (i.e., \code{THE BEST LOGLIKELIHOOD VALUE HAS BEEN REPLICATED}).
   \item{\code{"AIC"}}: Akaike information criterion (i.e., \code{Akaike (AIC)}).
   \item{\code{"CAIC"}}: Consistent AIC, not reported in the Mplus output, but
                         simply \code{BIC + #Param}.
   \item{\code{"BIC"}}: Bayesian information criterion (i.e., \code{Bayesian (BIC)}).
   \item{\code{"Chi-Pear"}}: Pearson chi-square test of model fit (i.e., \code{Pearson Chi-Square}),
                             only available when indicators are count or ordered categorical.
   \item{\code{"Chi-LRT"}}: Likelihood ratio chi-square test of model fit (i.e., \code{Likelihood Ratio Chi-Square}),
                            only available when indicators are count or ordered catgeorical.
   \item{\code{"SABIC"}}: Sample-size adjusted BIC (i.e., \code{Sample-Size Adjusted BIC}).
   \item{\code{"LMR-LRT"}}: Significance value (\emph{p}-value) of the Vuong-Lo-Mendell-Rubin test
                            (i.e., \code{VUONG-LO-MENDELL-RUBIN LIKELIHOOD RATIO TEST}).
                            Provided only when \code{OUTPUT: TECH11}.
   \item{\code{"A-LRT"}}: Significance value (\emph{p}-value) of the Adjusted Lo-Mendell-Rubin Test
                          (i.e., \code{LO-MENDELL-RUBIN ADJUSTED LRT TEST}).
                          Provided only when \code{OUTPUT: TECH11}.
   \item{\code{"BLRT"}}: Significance value (\emph{p}-value) of the bootstrapped
                         likelihood ratio test. Provided only when \code{OUTPUT: TECH14}.
   \item{\code{"Entropy"}}: Sample-size adjusted BIC (i.e., \code{Entropy}).
   \item{\code{"p1"}}: Class proportion of the first class based on the estimated
                       posterior probabilities (i.e., \code{FINAL CLASS COUNTS AND PROPORTIONS}).
   \item{\code{"p2"}}: Class proportion of the second class based on the estimated
                       posterior probabilities (i.e., \code{FINAL CLASS COUNTS AND PROPORTIONS}).
 }
}
\examples{
\dontrun{
# Load data set "HolzingerSwineford1939" in the lavaan package
data("HolzingerSwineford1939", package = "lavaan")

# Run LCA with k = 1 to k = 6 classes
mplus.lca(HolzingerSwineford1939, ind = c("x1", "x2", "x3", "x4"),
          run.mplus = TRUE)

# Read Mplus output files, create result table, write table, and save plots
result.lca(write = "LCA.xlsx", plot = TRUE)

#------------------------------------------
# Draw bar chart manually

library(ggplot2)

# Collect LCA results
lca.result <- result.lca()

# Result table with means
means <- lca.result$result$mean

# Extract results from variance-covariance structure A with 4 latent classes
plotdat <- means[means$folder == "A_Invariant-Theta_Diagonal-Sigma" &  means$nclass == 4, ]

# Draw bar chart
ggplot(plotdat, aes(ind, est, group = class, fill = class)) +
  geom_bar(stat = "identity", position = "dodge", color = "black",
           linewidth = 0.1) +
  geom_errorbar(aes(ymin = low, ymax = upp), width = 0.23,
                linewidth = 0.2, position = position_dodge(0.9)) +
  scale_x_discrete("") +
  scale_y_continuous("Mean Value", limits = c(0, 9),
                     breaks = seq(0, 9, by = 1)) +
  labs(fill = "Latent Class") +
  guides(fill = guide_legend(nrow = 1L)) +
  theme(axis.title = element_text(size = 11),
        axis.text = element_text(size = 11),
        legend.position = "bottom",
        legend.key.size = unit(0.5 , 'cm'),
        legend.title = element_text(size = 11),
        legend.text = element_text(size = 11),
        legend.box.spacing = unit(-9L, "pt"))

# Save bar chart
ggsave("LCA_4-Class.png", dpi = 600, width = 6, height = 4)
}
}
\references{
Masyn, K. E. (2013). Latent class analysis and finite mixture modeling. In T. D.
Little (Ed.), \emph{The Oxford handbook of quantitative methods: Statistical analysis}
(pp. 551–611). Oxford University Press.

Muthen, L. K., & Muthen, B. O. (1998-2017). \emph{Mplus User's Guide} (8th ed.).
Muthen & Muthen.
}
\seealso{
\code{\link{mplus.lca}}, \code{\link{run.mplus}}, \code{\link{read.mplus}},
\code{\link{write.mplus}}
}
\author{
Takuya Yanagida \email{takuya.yanagida@univie.ac.at}
}
