% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tune.R
\name{tune}
\alias{tune}
\title{Function for Tuning a Learner}
\usage{
tune(
  tuner,
  task,
  learner,
  resampling,
  measures = NULL,
  term_evals = NULL,
  term_time = NULL,
  terminator = NULL,
  search_space = NULL,
  store_benchmark_result = TRUE,
  store_models = FALSE,
  check_values = FALSE,
  allow_hotstart = FALSE,
  keep_hotstart_stack = FALSE,
  evaluate_default = FALSE,
  callbacks = list(),
  method
)
}
\arguments{
\item{tuner}{(\link{Tuner})\cr
Optimization algorithm.}

\item{task}{(\link[mlr3:Task]{mlr3::Task})\cr
Task to operate on.}

\item{learner}{(\link[mlr3:Learner]{mlr3::Learner})\cr
Learner to tune.}

\item{resampling}{(\link[mlr3:Resampling]{mlr3::Resampling})\cr
Resampling that is used to evaluate the performance of the hyperparameter configurations.
Uninstantiated resamplings are instantiated during construction so that all configurations are evaluated on the same data splits.
Already instantiated resamplings are kept unchanged.
Specialized \link{Tuner} change the resampling e.g. to evaluate a hyperparameter configuration on different data splits.
This field, however, always returns the resampling passed in construction.}

\item{measures}{(\link[mlr3:Measure]{mlr3::Measure} or list of \link[mlr3:Measure]{mlr3::Measure})\cr
A single measure creates a \link{TuningInstanceSingleCrit} and multiple measures a \link{TuningInstanceMultiCrit}.
If \code{NULL}, default measure is used.}

\item{term_evals}{(\code{integer(1)})\cr
Number of allowed evaluations.
Ignored if \code{terminator} is passed.}

\item{term_time}{(\code{integer(1)})\cr
Maximum allowed time in seconds.
Ignored if \code{terminator} is passed.}

\item{terminator}{(\link{Terminator})\cr
Stop criterion of the tuning process.}

\item{search_space}{(\link[paradox:ParamSet]{paradox::ParamSet})\cr
Hyperparameter search space. If \code{NULL} (default), the search space is
constructed from the \link{TuneToken} of the learner's parameter set
(learner$param_set).}

\item{store_benchmark_result}{(\code{logical(1)})\cr
If \code{TRUE} (default), store resample result of evaluated hyperparameter
configurations in archive as \link[mlr3:BenchmarkResult]{mlr3::BenchmarkResult}.}

\item{store_models}{(\code{logical(1)})\cr
If \code{TRUE}, fitted models are stored in the benchmark result
(\code{archive$benchmark_result}). If \code{store_benchmark_result = FALSE}, models
are only stored temporarily and not accessible after the tuning. This
combination is needed for measures that require a model.}

\item{check_values}{(\code{logical(1)})\cr
If \code{TRUE}, hyperparameter values are checked before evaluation and
performance scores after. If \code{FALSE} (default), values are unchecked but
computational overhead is reduced.}

\item{allow_hotstart}{(\code{logical(1)})\cr
Allow to hotstart learners with previously fitted models. See also
\link[mlr3:HotstartStack]{mlr3::HotstartStack}. The learner must support hotstarting. Sets
\code{store_models = TRUE}.}

\item{keep_hotstart_stack}{(\code{logical(1)})\cr
If \code{TRUE}, \link[mlr3:HotstartStack]{mlr3::HotstartStack} is kept in \verb{$objective$hotstart_stack}
after tuning.}

\item{evaluate_default}{(\code{logical(1)})\cr
If \code{TRUE}, learner is evaluated with hyperparameters set to their default
values at the start of the optimization.}

\item{callbacks}{(list of \link{CallbackTuning})\cr
List of callbacks.}

\item{method}{(\code{character(1)})\cr
Deprecated. Use \code{tuner} instead.}
}
\value{
\link{TuningInstanceSingleCrit} | \link{TuningInstanceMultiCrit}
}
\description{
Function to tune a \link[mlr3:Learner]{mlr3::Learner}.
The function internally creates a \link{TuningInstanceSingleCrit} or \link{TuningInstanceMultiCrit} which describe the tuning problem.
It executes the tuning with the \link{Tuner} (\code{tuner}) and returns the result with the tuning instance (\verb{$result}).
The \link{ArchiveTuning} (\verb{$archive}) stores all evaluated hyperparameter configurations and performance scores.
}
\details{
The \link[mlr3:Task]{mlr3::Task}, \link[mlr3:Learner]{mlr3::Learner}, \link[mlr3:Resampling]{mlr3::Resampling}, \link[mlr3:Measure]{mlr3::Measure} and \link{Terminator} are used to construct a \link{TuningInstanceSingleCrit}.
If multiple performance \link[=Measure]{Measures} are supplied, a \link{TuningInstanceMultiCrit} is created.
The parameter \code{term_evals} and \code{term_time} are shortcuts to create a \link{Terminator}.
If both parameters are passed, a \link{TerminatorCombo} is constructed.
For other \link[=Terminator]{Terminators}, pass one with \code{terminator}.
If no termination criterion is needed, set \code{term_evals}, \code{term_time} and \code{terminator} to \code{NULL}.
The search space is created from \link[paradox:to_tune]{paradox::TuneToken} or is supplied by \code{search_space}.
}
\section{Resources}{

There are several sections about hyperparameter optimization in the \href{https://mlr3book.mlr-org.com}{mlr3book}.
\itemize{
\item Simplify tuning with the \href{https://mlr3book.mlr-org.com/optimization.html#sec-simplified-tuning}{\code{tune()}} function.
\item Learn about \href{https://mlr3book.mlr-org.com/technical.html#sec-tuning-space}{tuning spaces}.
}

The \href{https://mlr-org.com/gallery-all-optimization.html}{gallery} features a collection of case studies and demos about optimization.
\itemize{
\item Optimize an rpart classification tree with only a \href{https://mlr-org.com/gallery/optimization/2022-11-10-hyperparameter-optimization-on-the-palmer-penguins/}{few lines of code}.
\item Tune an XGBoost model with \href{https://mlr-org.com/gallery/optimization/2022-11-04-early-stopping-with-xgboost/}{early stopping}.
\item Make us of proven \href{https://mlr-org.com/gallery/optimization/2021-07-06-introduction-to-mlr3tuningspaces/}{search space}.
\item Learn about \href{https://mlr-org.com/gallery/optimization/2023-01-16-hotstart/}{hotstarting} models.
}
}

\section{Default Measures}{

If no measure is passed, the default measure is used.
The default measure depends on the task type.\tabular{lll}{
   Task \tab Default Measure \tab Package \cr
   \code{"classif"} \tab \code{"classif.ce"} \tab \CRANpkg{mlr3} \cr
   \code{"regr"} \tab \code{"regr.mse"} \tab \CRANpkg{mlr3} \cr
   \code{"surv"} \tab \code{"surv.cindex"} \tab \CRANpkg{mlr3proba} \cr
   \code{"dens"} \tab \code{"dens.logloss"} \tab \CRANpkg{mlr3proba} \cr
   \code{"classif_st"} \tab \code{"classif.ce"} \tab \CRANpkg{mlr3spatial} \cr
   \code{"regr_st"} \tab \code{"regr.mse"} \tab \CRANpkg{mlr3spatial} \cr
   \code{"clust"} \tab \code{"clust.dunn"} \tab \CRANpkg{mlr3cluster} \cr
}
}

\section{Analysis}{

For analyzing the tuning results, it is recommended to pass the \link{ArchiveTuning} to \code{as.data.table()}.
The returned data table is joined with the benchmark result which adds the \link[mlr3:ResampleResult]{mlr3::ResampleResult} for each hyperparameter evaluation.

The archive provides various getters (e.g. \verb{$learners()}) to ease the access.
All getters extract by position (\code{i}) or unique hash (\code{uhash}).
For a complete list of all getters see the methods section.

The benchmark result (\verb{$benchmark_result}) allows to score the hyperparameter configurations again on a different measure.
Alternatively, measures can be supplied to \code{as.data.table()}.

The \CRANpkg{mlr3viz} package provides visualizations for tuning results.
}

\examples{
# Hyperparameter optimization on the Palmer Penguins data set
task = tsk("pima")

# Load learner and set search space
learner = lrn("classif.rpart",
  cp = to_tune(1e-04, 1e-1, logscale = TRUE)
)

# Run tuning
instance = tune(
  tuner = tnr("random_search", batch_size = 2),
  task = tsk("pima"),
  learner = learner,
  resampling = rsmp ("holdout"),
  measures = msr("classif.ce"),
  terminator = trm("evals", n_evals = 4)
)

# Set optimal hyperparameter configuration to learner
learner$param_set$values = instance$result_learner_param_vals

# Train the learner on the full data set
learner$train(task)

# Inspect all evaluated configurations
as.data.table(instance$archive)
}
