% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modelplot.R
\name{modelplot}
\alias{modelplot}
\title{Plot model coefficients using points or point-ranges}
\usage{
modelplot(
  models,
  conf_level = 0.95,
  coef_map = NULL,
  coef_omit = NULL,
  coef_rename = NULL,
  vcov = NULL,
  add_rows = NULL,
  facet = FALSE,
  draw = TRUE,
  background = NULL,
  ...
)
}
\arguments{
\item{models}{a model or (optionally named) list of models}

\item{conf_level}{confidence level to use for confidence intervals}

\item{coef_map}{named character vector. Values refer to the variable names
that will appear in the table. Names refer to the original term names stored
in the model object, e.g. c("hp:mpg"="hp X mpg") for an interaction term.
Coefficients that are omitted from this vector will be omitted from the
table. The table will be ordered in the same order as this vector.}

\item{coef_omit}{string regular expression. Omits all matching coefficients
from the table using `grepl(perl=TRUE)`.}

\item{coef_rename}{named character vector. Values refer to the variable names
that will appear in the table. Names refer to the original term names stored
in the model object, e.g. c("hp:mpg"="hp X mpg") for an interaction term.}

\item{vcov}{robust standard errors and other manual statistics. The `vcov` argument accepts five types of input (see the 'Details' and 'Examples' sections below):
\itemize{
  \item string, vector, or list of strings: "robust", "HC", "HC0", "HC1", "HC2", "HC3", "HC4", "HC4m", "HC5", "stata", or "classical" (alias "constant" or "iid").
  \item formula or list of formulas with the cluster variable(s) on the right-hand side (e.g., ~clusterid).
  \item function or list of functions which return variance-covariance matrices with row and column names equal to the names of your coefficient estimates (e.g., `stats::vcov`, `sandwich::vcovHC`).
  \item list of `length(models)` variance-covariance matrices with row and column names equal to the names of your coefficient estimates.
  \item a list of length(models) vectors with names equal to the names of your coefficient estimates. See 'Examples' section below. 
}}

\item{add_rows}{a data.frame (or tibble) with the same number of columns as
your main table. By default, rows are appended to the bottom of the table.
You can define a "position" attribute of integers to set the row positions.
See Examples section below.}

\item{facet}{TRUE or FALSE. When the 'models' argument includes several
model objects, TRUE draws terms in separate facets, and FALSE draws terms
side-by-side (dodged).}

\item{draw}{TRUE returns a 'ggplot2' object, FALSE returns the data.frame
used to draw the plot.}

\item{background}{A list of 'ggplot2' geoms to add to the background of the
plot.  This is especially useful to display annotations "behind" the
'geom_pointrange' that 'modelplot' draws.}

\item{...}{all other arguments are passed through to the extractor and
table-making functions. This allows users to specify additional options such
as set `broom::tidy(exponentiate=TRUE)` to exponentiate logistic regression
coefficients or `kableExtra::kbl(escape=FALSE)` to avoid escaping math
characters in `kableExtra` tables.}
}
\description{
Plot model coefficients using points or point-ranges
}
\examples{
\dontrun{

library(modelsummary)

# single model
mod <- lm(hp ~ vs + drat, mtcars)
modelplot(mod)

# omit terms with string matches or regexes
modelplot(mod, coef_omit = 'Interc')

# rename, reorder and subset with 'coef_map'
cm <- c('vs' = 'V-shape engine',
  'drat' = 'Rear axle ratio')
modelplot(mod, coef_map = cm)

# several models
models <- list()
models[['Small model']] <- lm(hp ~ vs, mtcars)
models[['Medium model']] <- lm(hp ~ vs + factor(cyl), mtcars)
models[['Large model']] <- lm(hp ~ vs + drat + factor(cyl), mtcars)
modelplot(models)

# add_rows: add an empty reference category

mod <- lm(hp ~ factor(cyl), mtcars)

add_rows = data.frame(
  term = "factory(cyl)4",
  model = "Model 1",
  estimate = NA)
attr(add_rows, "position") = 3
modelplot(mod, add_rows = add_rows) 


# customize your plots with 'ggplot2' functions
library(ggplot2)

modelplot(models) +
  scale_color_brewer(type = 'qual') +
  theme_classic()

# pass arguments to 'geom_pointrange' through the ... ellipsis
modelplot(mod, color = 'red', size = 1, fatten = .5)

# add geoms to the background, behind geom_pointrange
b <- list(geom_vline(xintercept = 0, color = 'orange'),
  annotate("rect", alpha = .1,
    xmin = -.5, xmax = .5,
    ymin = -Inf, ymax = Inf),
  geom_point(aes(y = term, x = estimate), alpha = .3,
    size = 10, color = 'red', shape = 'square'))
modelplot(mod, background = b)
}

}
