% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/datasummary_correlation.R
\name{datasummary_correlation}
\alias{datasummary_correlation}
\title{Generate a correlation table for all numeric variables in your dataset.}
\usage{
datasummary_correlation(
  data,
  output = "default",
  method = "pearson",
  fmt = 2,
  align = NULL,
  add_rows = NULL,
  add_columns = NULL,
  title = NULL,
  notes = NULL,
  escape = TRUE,
  ...
)
}
\arguments{
\item{data}{A data.frame (or tibble)}

\item{output}{filename or object type (character string)
\itemize{
\item Supported filename extensions: .docx, .html, .tex, .md, .txt, .csv, .xlsx, .png, .jpg
\item Supported object types: "default", "html", "markdown", "latex", "latex_tabular", "data.frame", "gt", "kableExtra", "huxtable", "flextable", "DT", "jupyter". The "modelsummary_list" value produces a lightweight object which can be saved and fed back to the \code{modelsummary} function.
\item The "default" output format can be set to "kableExtra", "gt", "flextable", "huxtable", "DT", or "markdown"
\itemize{
\item If the user does not choose a default value, the packages listed above are tried in sequence.
\item Session-specific configuration: \code{options("modelsummary_factory_default" = "gt")}
\item Persistent configuration: \code{modelsummary_config(output = "markdown")}
}
\item Warning: Users should not supply a file name to the \code{output} argument if they intend to customize the table with external packages. See the 'Details' section.
\item LaTeX compilation requires the \code{booktabs} and \code{siunitx} packages, but \code{siunitx} can be disabled or replaced with global options. See the 'Details' section.
}}

\item{method}{character or function
\itemize{
\item character: "pearson", "kendall", "spearman", or "pearspear"
(Pearson correlations above and Spearman correlations below the diagonal)
\item function: takes a data.frame with numeric columns and returns a
square matrix or data.frame with unique row.names and colnames
corresponding to variable names. Note that the
\code{datasummary_correlation_format} can often be useful for formatting the
output of custom correlation functions.
}}

\item{fmt}{how to format numeric values: integer, user-supplied function, or \code{modelsummary} function.
\itemize{
\item Integer: Number of decimal digits
\item User-supplied functions:
\itemize{
\item Any function which accepts a numeric vector and returns a character vector of the same length.
}
\item \code{modelsummary} functions:
\itemize{
\item \code{fmt = fmt_significant(2)}: Two significant digits (at the term-level)
\item \code{fmt = fmt_sprintf("\%.3f")}: See \code{?sprintf}
\item \code{fmt = fmt_identity()}: unformatted raw values
}
}}

\item{align}{A string with a number of characters equal to the number of columns in
the table (e.g., \code{align = "lcc"}).  Valid characters: l, c, r, d.
\itemize{
\item "l": left-aligned column
\item "c": centered column
\item "r": right-aligned column
\item "d": dot-aligned column. For LaTeX/PDF output, this option requires at least version 3.0.25 of the siunitx LaTeX package. These commands must appear in the LaTeX preamble (they are added automatically when compiling Rmarkdown documents to PDF):
\itemize{
\item \verb{\\usepackage\{booktabs\}}
\item \verb{\\usepackage\{siunitx\}}
\item \verb{\\newcolumntype\{d\}\{S[ input-open-uncertainty=, input-close-uncertainty=, parse-numbers = false, table-align-text-pre=false, table-align-text-post=false ]\}}
}
}}

\item{add_rows}{a data.frame (or tibble) with the same number of columns as
your main table. By default, rows are appended to the bottom of the table.
You can define a "position" attribute of integers to set the row positions.
See Examples section below.}

\item{add_columns}{a data.frame (or tibble) with the same number of rows as
your main table.}

\item{title}{string}

\item{notes}{list or vector of notes to append to the bottom of the table.}

\item{escape}{boolean TRUE escapes or substitutes LaTeX/HTML characters which could
prevent the file from compiling/displaying. This setting does not affect captions or notes.}

\item{...}{other parameters are passed through to the table-making
packages.}
}
\description{
The names of the variables displayed in the correlation table are the names
of the columns in the \code{data}. You can rename those columns (with or without
spaces) to produce a table of human-readable variables. See the Details and
Examples sections below, and the vignettes on the \code{modelsummary} website:
\itemize{
\item https://vincentarelbundock.github.io/modelsummary/
\item https://vincentarelbundock.github.io/modelsummary/articles/datasummary.html
}
}
\section{Global Options}{


The behavior of \code{modelsummary} can be modified by setting global options. For example:
\itemize{
\item \code{options(modelsummary_model_labels = "roman")}
}

The rest of this section describes each of the options above.
\subsection{Model labels: default column names}{

These global option changes the style of the default column headers:
\itemize{
\item \code{options(modelsummary_model_labels = "roman")}
\item \code{options(modelsummary_panel_labels = "roman")}
}

The supported styles are: "model", "panel", "arabic", "letters", "roman", "(arabic)", "(letters)", "(roman)""

The panel-specific option is only used when \code{shape="rbind"}
}

\subsection{Table-making packages}{

\code{modelsummary} supports 4 table-making packages: \code{kableExtra}, \code{gt},
\code{flextable}, \code{huxtable}, and \code{DT}. Some of these packages have overlapping
functionalities. For example, 3 of those packages can export to LaTeX. To
change the default backend used for a specific file format, you can use
the \code{options} function:

\code{options(modelsummary_factory_html = 'kableExtra')}

\code{options(modelsummary_factory_latex = 'gt')}

\code{options(modelsummary_factory_word = 'huxtable')}

\code{options(modelsummary_factory_png = 'gt')}
}

\subsection{Table themes}{

Change the look of tables in an automated and replicable way, using the \code{modelsummary} theming functionality. See the vignette: https://vincentarelbundock.github.io/modelsummary/articles/appearance.html
\itemize{
\item \code{modelsummary_theme_gt}
\item \code{modelsummary_theme_kableExtra}
\item \code{modelsummary_theme_huxtable}
\item \code{modelsummary_theme_flextable}
\item \code{modelsummary_theme_dataframe}
}
}

\subsection{Model extraction functions}{

\code{modelsummary} can use two sets of packages to extract information from
statistical models: the \code{easystats} family (\code{performance} and \code{parameters})
and \code{broom}. By default, it uses \code{easystats} first and then falls back on
\code{broom} in case of failure. You can change the order of priorities or include
goodness-of-fit extracted by \emph{both} packages by setting:

\code{options(modelsummary_get = "broom")}

\code{options(modelsummary_get = "easystats")}

\code{options(modelsummary_get = "all")}
}

\subsection{Formatting numeric entries}{

By default, LaTeX tables enclose all numeric entries in the \verb{\\num\{\}} command
from the siunitx package. To prevent this behavior, or to enclose numbers
in dollar signs (for LaTeX math mode), users can call:

\code{options(modelsummary_format_numeric_latex = "plain")}

\code{options(modelsummary_format_numeric_latex = "mathmode")}

A similar option can be used to display numerical entries using MathJax in
HTML tables:

\code{options(modelsummary_format_numeric_html = "mathjax")}
}
}

\section{Examples}{


\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(modelsummary)

# clean variable names (base R)
dat <- mtcars[, c("mpg", "hp")]
colnames(dat) <- c("Miles / Gallon", "Horse Power")
datasummary_correlation(dat)

# clean variable names (tidyverse)
library(tidyverse)
dat <- mtcars \%>\%
  select(`Miles / Gallon` = mpg,
         `Horse Power` = hp)
datasummary_correlation(dat)

# alternative methods
datasummary_correlation(dat, method = "pearspear")

# custom function
cor_fun <- function(x) cor(x, method = "kendall")
datasummary_correlation(dat, method = cor_fun)

# rename columns alphabetically and include a footnote for reference
note <- sprintf("(\%s) \%s", letters[1:ncol(dat)], colnames(dat))
note <- paste(note, collapse = "; ")

colnames(dat) <- sprintf("(\%s)", letters[1:ncol(dat)])

datasummary_correlation(dat, notes = note)

# `datasummary_correlation_format`: custom function with formatting
dat <- mtcars[, c("mpg", "hp", "disp")]

cor_fun <- function(x) \{
  out <- cor(x, method = "kendall")
  datasummary_correlation_format(
    out,
    fmt = 2,
    upper_triangle = "x",
    diagonal = ".")
\}

datasummary_correlation(dat, method = cor_fun)

# use kableExtra and psych to color significant cells
library(psych)
library(kableExtra)

dat <- mtcars[, c("vs", "hp", "gear")]

cor_fun <- function(dat) \{
  # compute correlations and format them
  correlations <- data.frame(cor(dat))
  correlations <- datasummary_correlation_format(correlations, fmt = 2)

  # calculate pvalues using the `psych` package
  pvalues <- psych::corr.test(dat)$p

  # use `kableExtra::cell_spec` to color significant cells
  for (i in 1:nrow(correlations)) \{
    for (j in 1:ncol(correlations)) \{
      if (pvalues[i, j] < 0.05 && i != j) \{
        correlations[i, j] <- cell_spec(correlations[i, j], background = "pink")
      \}
    \}
  \}
  return(correlations)
\}

# The `escape=FALSE` is important here!
datasummary_correlation(dat, method = cor_fun, escape = FALSE)
}\if{html}{\out{</div>}}
}

\references{
Arel-Bundock V (2022). “modelsummary: Data and Model Summaries in R.” \emph{Journal of Statistical Software}, \emph{103}(1), 1-23. \doi{10.18637/jss.v103.i01}.'
}
