% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/optimize_controls.R
\name{optimize_controls}
\alias{optimize_controls}
\title{Select control units that optimize covariate balance}
\usage{
optimize_controls(
  z,
  X,
  st,
  importances = NULL,
  ratio = NULL,
  q_s = NULL,
  integer = FALSE,
  solver = "Rglpk",
  seed = NULL,
  runs = 1,
  time_limit = Inf
)
}
\arguments{
\item{z}{a treatment indicator vector with \code{i}th entry equal to 0 if
unit \code{i} is a control and equal to 1 if unit \code{i} is treated.}

\item{X}{a matrix or data frame containing constraints in the columns. The number
of rows should equal the length of \code{z}. Balance is achieved when a constraint
sums to 0, such that numbers closer to 0 are better. When a constraint
does not apply to a particular unit, the entry should be \code{NA}.
This should typically be generated using \code{\link{generate_constraints}()}.}

\item{st}{a stratum vector with the \code{i}th entry equal to the
stratum of unit \code{i}. This should have the same order of units and length
as \code{z}.}

\item{importances}{a vector with length equal to the number of constraints or columns
in \code{X}. This can be generated using \code{\link{generate_constraints}()} and each nonnegative value
denotes how much to prioritize each constraint, with the default being 1
for all constraints.}

\item{ratio}{a numeric specifying the desired ratio of controls to treated in
each stratum. If \code{NULL}, \code{q_s} should be specified.}

\item{q_s}{a named vector indicating how many control units are to be selected from each stratum.
If \code{NULL}, \code{ratio} should be specified. If both are specified, \code{q_s} will take priority.
Typically, if the desired ratio is not feasible for every stratum, \code{q_s} should be generated
using \code{\link{generate_qs}()}.}

\item{integer}{a logical stating whether to use a mixed integer programming solver
instead of randomized rounding. Default is \code{FALSE}.}

\item{solver}{a character stating which solver to use to run the linear program.
Options are "Rglpk" (default) or "gurobi". You must have the 'gurobi' package
installed to use the "gurobi" option. If available, this is the recommended solver.}

\item{seed}{the seed to use when doing the randomized rounding of the linear program.
This will allow results to be reproduced if desired. The default is \code{NULL},
which will choose a random seed to use and return.}

\item{runs}{the number of times to run randomized rounding of the linear solution.
The objective values of all runs will be reported, but the detailed results
will only be reported for the run with the lowest objective value. The default is 1.}

\item{time_limit}{numeric stating maximum amount of seconds for which the
program is allowed to run before aborting. Default is \code{Inf} for no time limit.}
}
\value{
List containing:
\describe{
  \item{\code{objective}}{objective value of the randomized rounding or mixed integer
    linear program solution.}
  \item{\code{objective_wo_importances}}{objective value of the randomized rounding or mixed integer
    linear program solution not weighted by the variable importances.}
  \item{\code{eps}}{the amount of imbalance obtained in each constraint from the linear program.}
  \item{\code{importances}}{the importance of each on the balance constraints.}
  \item{\code{selected}}{the selected treated and control units.}
  \item{\code{controls}}{the linear program weight assigned to each control and
    whether it was selected by randomized rounding.}
  \item{\code{rrdetails}}{A list containing:
  \describe{
  \item{\code{seed}}{the seed used before commencing the random sampling.}
  \item{\code{raw_objective}}{objective value of the randomized rounding or mixed integer
    linear program solution before the denominator has been corrected for the number of
    units chosen with missing covariate values.}
  \item{\code{raw_objective_wo_importances}}{objective value of the randomized rounding or mixed integer
    linear program solution not weighted by the variable importances
    before the denominator has been corrected for the number of
    units chosen with missing covariate values.}
  \item{\code{raw_eps}}{the amount of imbalance obtained in each constraint from the linear program,
    before the denominators have been corrected for the number of
    units chosen with missing covariate values.}
  \item{\code{run_raw_objectives}}{the objective values for each run of randomized rounding,
  before denominators have been corrected for missingness.}
  \item{\code{run_raw_objectives_wo_importances}}{the objective values for each run of randomized rounding,
  before denominators have been corrected for missingness, not scaled by constraint importances.}
  \item{\code{run_objectives}}{the objective values for each run of randomized rounding.}
  \item{\code{run_objectives_wo_importances}}{the objective values for each run of randomized rounding,
  not scaled by constraint importances.}
  }}
  \item{\code{lpdetails}}{the full return of the function \code{\link[Rglpk]{Rglpk_solve_LP}()}
    or \code{gurobi()} plus information about the epsilons and objective values
    for the linear program solution.}
}
}
\description{
Select control units within strata that optimize covariate balance.
Uses randomized rounding of a linear program or a mixed
integer linear program.
}
\examples{

data('nh0506')

# Create strata
age_cat <- cut(nh0506$age,
               breaks = c(19, 39, 50, 85),
               labels = c('< 40 years', '40 - 50 years', '> 50 years'))
strata <- age_cat : nh0506$sex

# Balance age, race, education, poverty ratio, and bmi both across and within the levels of strata
constraints <- generate_constraints(
                 balance_formulas = list(age + race + education + povertyr + bmi ~ 1 + strata),
                 z = nh0506$z,
                 data = nh0506)

# Choose one control for every treated unit in each stratum,
# balancing the covariates as described by the constraints
results <- optimize_controls(z = nh0506$z,
                             X = constraints$X,
                             st = strata,
                             importances = constraints$importances,
                             ratio = 1)

# If you want to use a ratio that's not feasible,
# you can supply a vector of the desired number of controls per stratum, q_s,
# typically generated by creating a distance matrix between strata and using
# generate_qs():

age_dist <- matrix(data = c(0, 1, 2, 1, 0, 1, 2, 1, 0),
                   nrow = 3,
                   byrow = TRUE,
                   dimnames = list(levels(age_cat), levels(age_cat)))

sex_dist <- matrix(data = c(0, 1, 1, 0),
                   nrow = 2,
                   dimnames = list(levels(nh0506$sex), levels(nh0506$sex)))

strata_dist <- create_dist_matrix(age_dist, sex_dist)

qs <- generate_qs(z = nh0506$z,
                  st = strata,
                  ratio = 2.5,
                  max_ratio = 2.6,
                  max_extra_s = 0,
                  strata_dist = strata_dist)

results <- optimize_controls(z = nh0506$z,
                             X = constraints$X,
                             st = strata,
                             importances = constraints$importances,
                             q_s = qs)
}
