#' Random diffnet network
#'
#' Simulates a diffusion network by creating a random dynamic network and
#' adoption threshold levels.
#'
#' @export
#' @param n Integer scalar. Number of vertices.
#' @param t Integer scalar. Time length.
#' @param seed.nodes Character scalar. Type of seed nodes (early adoptes).
#' @param seed.p.adopt Numeric scalar. Proportion of early adopters.
#' @param seed.graph Character scalar. Type of algorithm used to generate the
#' seed graph (see details).
#' @param rgraph.args List. Arguments to be passed to rgraph.
#' @param rewire Logical scalar. When TRUE, network slices are generated by rewiring
#' (see \code{\link{rewire_graph}}).
#' @param rewire.args List. Arguments to be passed to \code{\link{rewire_graph}}.
#' @param threshold.dist Function. Applied via \code{\link{sapply}}, sets the
#' adoption threshold for each node.
#' @param exposure.args List. Arguments to be passed to \code{link{exposure}}.
#' @return A random \code{\link{diffnet}} class object.
#' @family simulation functions
#' @details
#'
#' Instead of randomizing whether an individual adopts the innovation or not, this
#' toy model randomizes threshold levels, seed adopters and network structure, so
#' an individual adopts the innovation in time \eqn{T} iff his exposure is above or
#' equal to his threshold. The simulation is done in the following steps:
#'
#' \enumerate{
#'  \item Using \code{seed.graph}, a baseline graph is created.
#'  \item Given the baseline graph, the set of initial adopters is defined
#'  using \code{seed.nodes}.
#'  \item Afterwards, if \code{rewire=TRUE} \eqn{t-1} slices of the network are created
#'  by iteratively rewiring the baseline graph.
#'  \item The \code{threshold.dist} function is applied to each node in the graph.
#'  \item Simulation starts at \eqn{t=2} asigning adopters in each time period
#'  accordingly to each vertex's threshold and exposure.
#' }
#'
#' \code{seed.nodes} can be \code{"marginal"}, \code{"central"} or \code{"random"},
#' So each of these values sets the initial adopters using the vertices with lowest
#' degree, with highest degree or completely randomly. The number of early adoptes
#' is set as \code{seed.p.adopt * n}. Please note that when marginal nodes are
#' set as seed it may be the case that no diffusion process is attained as the
#' chosen set of first adopters can be isolated.
#'
#' The argument \code{seed.graph} allows the user to set the algorithm used to
#' generate the first network (network in t=1). This can be either "scale-free"
#' (Barabasi-Albert model using the \code{\link{rgraph_ba}} function, the default),
#' \code{"bernoulli"} (Erdos-Renyi model using the \code{\link{rgraph_er}} function),
#' or \code{"small-world"} (Watts-Strogatz model using the \code{\link{rgraph_ws}}
#' function). The list \code{rgraph.args} passes arguments to the chosen algorithm.
#'
#' When \code{rewire=TRUE}, the networks that follow t=1 will be generated using the
#' \code{\link{rewire_graph}} function as \eqn{G(t) = G(t-1)}.
#'
#' Finally, \code{threshold.dist} sets the threshold for each vertex in the graph.
#' It is applied using \code{sapply} as follows
#'
#' \preformatted{
#' sapply(1:n, threshold.dist)
#' }
#'
#' By default sets the threshold to be random for each node in the graph.
#'
#' @examples
#' # Asimple example -----------------------------------------------------------
#' z <- rdiffnet(50,10)
#' z
#' summary(z)
#'
#' # A more complex example: Adopt if at least one neighbor has adopted --------
#' y <- rdiffnet(50, 10, threshold.dist=function(x) 1,
#'     exposure.args=list(valued=FALSE, normalized=FALSE))
#' @author Vega Yon
rdiffnet <- function(n, t,
                     seed.nodes="random", seed.p.adopt=0.05,
                     seed.graph="scale-free", rgraph.args=list(),
                     rewire=TRUE, rewire.args=list(p=.1, undirected=TRUE),
                     threshold.dist=function(x) runif(1),
                     exposure.args=list(
                       outgoing=TRUE, valued=getOption("diffnet.valued", FALSE),
                       normalized=TRUE
                       )
                     ) {

  # Step 0.0: Creating the network seed
  if (seed.graph == "scale-free") {
    if (!length(rgraph.args$m0))
      rgraph.args$t <- n-1L

    sgraph <- do.call(rgraph_ba, rgraph.args)
  } else if (seed.graph == "bernoulli") {
    rgraph.args$n <- n

    sgraph <- do.call(rgraph_er, rgraph.args)
  } else if (seed.graph == "small-world") {
    rgraph.args$n <- n
    rgraph.args$k <- 2L
    rgraph.args$p <- .1

    sgraph <- do.call(rgraph_ws, rgraph.args)
  }

  diag(sgraph) <- 0

  # Number of initial adopters
  if (n*seed.p.adopt < 1) warning("Set of initial adopters set to 1.")
  n0 <- max(1, n*seed.p.adopt)

  # Step 0.1: Setting the seed nodes
  cumadopt <- matrix(0L, ncol=t, nrow=n)
  toa   <- matrix(NA, ncol=1, nrow= n)
  if (seed.nodes %in% c("central","marginal")) {
    d <- dgr(sgraph)
    decre <- ifelse(seed.nodes == "central", TRUE, FALSE)
    d <- rownames(d[order(d, decreasing = decre),,drop=FALSE])
    d <- d[1:floor(n0)]
    d <- as.numeric(d)
    cumadopt[d, ] <- 1L
    toa[d] <- 1L
  } else if (seed.nodes == "random") {
    d <- sample.int(n, floor(n0))
    toa[d] <- 1
    cumadopt[d,] <- 1L
  } else
    stop("Invalid -seed.nodes- option. It must be either \"central\", \"marginal\", or \"random\"")

  # Step 1.0: Rewiring or not
  graph      <- vector("list", t)
  graph[[1L]] <- sgraph
  if (rewire) {
    for (i in 2L:t) {
      rewire.args$graph <- graph[[i-1L]]
      graph[[i]] <- do.call(rewire_graph, rewire.args)
    }
  } else {
    graph <- lapply(1L:t, function(x) sgraph)
  }

  names(graph) <- 1L:t

  # Step 3.0: Thresholds
  thr <- sapply(1:n, threshold.dist)

  for (i in 2:t) {

    # Computing exposure
    exposure.args[c("graph", "cumadopt")] <- list(graph[i], cumadopt[,i,drop=FALSE])
    expo <- do.call(exposure, exposure.args)

    whoadopts <- which( (expo >= thr) & is.na(toa))
    toa[whoadopts] <- i
    cumadopt[whoadopts, i:t] <- 1L
  }
  reachedt <- max(toa, na.rm=TRUE)

  if (reachedt == 1) {
    stop("No diffusion in this network (Ups!) try changing the seed or the parameters.")
  }

  as_diffnet(graph, as.integer(toa), undirected=FALSE, t0=1, t1=t,
             vertex.static.attrs = cbind(real_threshold=thr))
}

