% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cross_validate_concentration.R
\name{cross_validate_concentration}
\alias{cross_validate_concentration}
\title{Cross-validates The Concentration Parameter In A 3D Spherical Regression.}
\usage{
cross_validate_concentration(
  concentration_upper_bound = 10,
  explanatory_points,
  response_points,
  weights_generator = weight_explanatory_points,
  number_of_expansion_terms = 1,
  number_of_iterations = 1,
  allow_reflections = FALSE
)
}
\arguments{
\item{concentration_upper_bound}{A scalar numeric value representing
the upper end-point of the interval to be searched for the required minimizer.
Defaults to \code{10}.}

\item{explanatory_points}{An \emph{m}-by-3 matrix whose rows contain
the Cartesian coordinates of the explanatory points used to
calculate the regression estimators.}

\item{response_points}{An \emph{m}-by-\emph{3} matrix whose rows contain
the Cartesian coordinates of the response points corresponding
to the explanatory points.}

\item{weights_generator}{A function that, given a matrix of \emph{n}
evaluation points, returns an \emph{m}-by-\emph{n} matrix whose
\emph{j}-th column contains
the weights assigned to the explanatory points while analyzing the
\emph{j}-th evaluation point. Defaults to \code{\link{weight_explanatory_points}}.}

\item{number_of_expansion_terms}{The number of terms to be included
in the expansion of the matrix exponential applied while
approximating a local rotation matrix. Must be \code{1} or \code{2}.
Defaults to \code{1}.}

\item{number_of_iterations}{The number of
rotation fitting steps to be executed.
At each step, the points estimated during the previous step
are exploited as the current explanatory points. Defaults to \code{1}.}

\item{allow_reflections}{A logical scalar value. If set to \code{TRUE} signals
that reflections are allowed. Defaults to \code{FALSE}. It is ignored if
\code{number_of_expansion_terms} is \code{2}.}
}
\value{
A list having two components,
\code{concentration}, a scalar, numeric value representing the cross-validated
concentration for the specified 3D regression, and
\code{objective}, the value of the cross-validating objective function at argument \code{concentration}.
}
\description{
Returns a cross-validated value for the concentration parameter in
a 3D regression, relating specific explanatory points to response
ones, given a weighting scheme for the observed data set.
This function supports the method for sphere-sphere regression
proposed by Di Marzio et al. (2018).
}
\details{
Function \code{weights_generator} must be prototyped as having the
following three arguments:
\describe{
 \item{\code{evaluation_points}}{a matrix whose \emph{n} rows are the Cartesian coordinates of
given evaluation points.}
 \item{\code{explanatory_points}}{a matrix whose \emph{m} rows are the Cartesian coordinates of
given explanatory points.}
 \item{\code{concentration}}{A non negative scalar whose reciprocal value
is proportional to the bandwidth applied while estimating
a spherical regression model.}
}
It is also expected that \code{weights_generator} will return
a non \code{NULL} numerical \emph{m}-by-\emph{n} matrix whose \emph{j}-th column contains
the weights assigned to the explanatory points while analyzing the
\emph{j}-th evaluation point.
}
\examples{
library(nprotreg)

# Define a matrix of explanatory points.

number_of_explanatory_points <- 50

explanatory_points <- get_equally_spaced_points(
  number_of_explanatory_points)

# Define a matrix of response points by simulation.

local_rotation_composer <- function(point) {
  independent_components <- (1 / 2) *
    c(exp(2.0 * point[3]), - exp(2.0 * point[2]), exp(2.0 * point[1]))
}

local_error_sampler <- function(point) {
  rnorm(3)
}

response_points <- simulate_regression(explanatory_points,
                                       local_rotation_composer,
                                       local_error_sampler)

# Define an upper bound for concentration.

concentration_upper_bound <- 1

# Use default weights generator.

weights_generator <- weight_explanatory_points

# Cross-validate concentration parameter.

cv_info <- cross_validate_concentration(
  concentration_upper_bound,
  explanatory_points,
  response_points,
  weights_generator,
  number_of_expansion_terms = 1,
  number_of_iterations = 2,
  allow_reflections = FALSE
)

# Get the cross-validated concentration value.

cat("cross-validated concentration value: \n")
print(cv_info$concentration)
}
\references{
Marco Di Marzio, Agnese Panzera & Charles C. Taylor (2018)
Nonparametric rotations for sphere-sphere regression,
Journal of the American Statistical Association,
DOI: \href{https://doi.org/10.1080/01621459.2017.1421542}{10.1080/01621459.2017.1421542}
}
\seealso{
Other Regression functions: 
\code{\link{fit_regression}()},
\code{\link{get_equally_spaced_points}()},
\code{\link{get_skew_symmetric_matrix}()},
\code{\link{simulate_regression}()},
\code{\link{simulate_rigid_regression}()},
\code{\link{weight_explanatory_points}()}
}
\concept{Regression functions}
