% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ctd.R
\name{handleFlags,ctd-method}
\alias{handleFlags,ctd-method}
\title{Handle Flags in CTD Objects}
\usage{
\S4method{handleFlags}{ctd}(
  object,
  flags = NULL,
  actions = NULL,
  where = NULL,
  debug = getOption("oceDebug")
)
}
\arguments{
\item{object}{a \linkS4class{ctd} object.}

\item{flags}{A \link{list} specifying flag values upon which
actions will be taken. This can take two forms.
\itemize{
\item In the first form, the
list has named elements each containing a vector of integers. For example,
salinities flagged with values of 1 or 3:9 would be specified
by \code{flags=list(salinity=c(1,3:9))}. Several data items can be specified,
e.g. \code{flags=list(salinity=c(1,3:9), temperature=c(1,3:9))} indicates
that the actions are to take place for both salinity and temperature.
\item In the second form, \code{flags} is a list holding a single \emph{unnamed} vector, and
this means to apply the actions to \emph{all} the data entries.  For example,
\code{flags=list(c(1,3:9))} means to apply not just to salinity and temperature,
but to everything within the \code{data} slot.
}

If \code{flags} is not provided, then \code{\link[=defaultFlags]{defaultFlags()}} is called, to try to
determine a reasonable default.}

\item{actions}{an optional \link{list} that contains items with
names that match those in the \code{flags} argument.  If \code{actions}
is not supplied, the default will be to set all values identified by
\code{flags} to \code{NA}; this can also be specified by
specifying \code{actions=list("NA")}. It is also possible to specify
functions that calculate replacement values. These are provided
with \code{object} as the single argument, and must return a
replacement for the data item in question.
See \dQuote{Details} for the default that is used if \code{actions} is not supplied.}

\item{where}{an optional character value that permits the function to work with
objects that store flags in e.g. \code{object@metadata$flags$where}
instead of in \code{object@metadata$flags}, and data within
\code{object@data$where} instead of within \code{object@data}. The
default value of \code{NULL} means to look withing \code{object@metadata}
itself, and this is the default within \code{oce}.  (The purpose of \code{where}
is to make \code{oce} extensible by other packages, which may choose to store
data two levels deep in the \code{data} slot.)}

\item{debug}{An optional integer specifying the degree of debugging, with
value 0 meaning to skip debugging and 1 or higher meaning to print some
information about the arguments and the data. It is usually a good idea to set
this to 1 for initial work with a dataset, to see which flags are being
handled for each data item. If not supplied, this defaults to the value of
\code{\link{getOption}}\code{("oceDebug")}.}
}
\description{
Data-quality flags are stored in the \code{metadata}
slot of \linkS4class{oce} objects in a
\link{list} named \code{flags}.
The present function (a generic that has specialized versions
for various data classes) provides a way to
manipulate the contents of the \code{data} slot, based on
such data-quality flags. For example, a common operation is to replace
erroneous data with \code{NA}.

If \code{metadata$flags} in the first argument
is empty, then that object is returned, unaltered.
Otherwise, \code{handleFlags} analyses the data-quality flags within
the object, in context of the \code{flags} argument, and then interprets
the \code{action} argument to select an action that is to be applied
to the matched data.
}
\examples{
library(oce)
data(section)
stn <- section[["station", 100]]
# 1. Default: anything not flagged as 2 is set to NA, to focus
# solely on 'good', in the World Hydrographic Program scheme.
STN1 <- handleFlags(stn, flags=list(c(1, 3:9)))
data.frame(old=stn[["salinity"]], new=STN1[["salinity"]], salinityFlag=stn[["salinityFlag"]])

# 2. Use bottle salinity, if it is good and ctd is bad
replace <- 2 == stn[["salinityBottleFlag"]] && 2 != stn[["salinityFlag"]]
S <- ifelse(replace, stn[["salinityBottle"]], stn[["salinity"]])
STN2 <- oceSetData(stn, "salinity", S)

# 3. Use smoothed TS relationship to nudge questionable data.
f <- function(x) {
  S <- x[["salinity"]]
  T <- x[["temperature"]]
  df <- 0.5 * length(S) # smooths a bit
  sp <- smooth.spline(T, S, df=df)
  0.5 * (S + predict(sp, T)$y)
}
par(mfrow=c(1,2))
STN3 <- handleFlags(stn, flags=list(salinity=c(1,3:9)), action=list(salinity=f))
plotProfile(stn, "salinity", mar=c(3, 3, 3, 1))
p <- stn[["pressure"]]
par(mar=c(3, 3, 3, 1))
plot(STN3[["salinity"]] - stn[["salinity"]], p, ylim=rev(range(p)))

# 4. Single-variable flags (vector specification)
data(section)
# Multiple-flag scheme: one per data item
A <- section[["station", 100]]
deep <- A[["pressure"]] > 1500
flag <- ifelse(deep, 7, 2)
for (flagName in names(A[["flags"]]))
    A[[paste(flagName, "Flag", sep="")]] <- flag
Af <- handleFlags(A)
expect_equal(is.na(Af[["salinity"]]), deep)

# 5. Single-variable flags (list specification)
B <- section[["station", 100]]
B[["flags"]] <- list(flag)
Bf <- handleFlags(B)
expect_equal(is.na(Bf[["salinity"]]), deep)

}
\references{
\enumerate{
\item \url{https://www.nodc.noaa.gov/woce/woce_v3/wocedata_1/whp/exchange/exchange_format_desc.htm}
}
}
\seealso{
Other functions relating to data-quality flags: 
\code{\link{defaultFlags}()},
\code{\link{handleFlags,adp-method}},
\code{\link{handleFlags,argo-method}},
\code{\link{handleFlags,oce-method}},
\code{\link{handleFlags,section-method}},
\code{\link{handleFlags}()},
\code{\link{initializeFlagScheme,ctd-method}},
\code{\link{initializeFlagScheme,oce-method}},
\code{\link{initializeFlagScheme,section-method}},
\code{\link{initializeFlagSchemeInternal}()},
\code{\link{initializeFlagScheme}()},
\code{\link{initializeFlags,adp-method}},
\code{\link{initializeFlags,oce-method}},
\code{\link{initializeFlagsInternal}()},
\code{\link{initializeFlags}()},
\code{\link{setFlags,adp-method}},
\code{\link{setFlags,ctd-method}},
\code{\link{setFlags,oce-method}},
\code{\link{setFlags}()}

Other things related to ctd data: 
\code{\link{CTD_BCD2014666_008_1_DN.ODF.gz}},
\code{\link{[[,ctd-method}},
\code{\link{[[<-,ctd-method}},
\code{\link{as.ctd}()},
\code{\link{cnvName2oceName}()},
\code{\link{ctd-class}},
\code{\link{ctd.cnv}},
\code{\link{ctdDecimate}()},
\code{\link{ctdFindProfiles}()},
\code{\link{ctdRaw}},
\code{\link{ctdTrim}()},
\code{\link{ctd}},
\code{\link{d200321-001.ctd}},
\code{\link{d201211_0011.cnv}},
\code{\link{initialize,ctd-method}},
\code{\link{initializeFlagScheme,ctd-method}},
\code{\link{oceNames2whpNames}()},
\code{\link{oceUnits2whpUnits}()},
\code{\link{plot,ctd-method}},
\code{\link{plotProfile}()},
\code{\link{plotScan}()},
\code{\link{plotTS}()},
\code{\link{read.ctd.itp}()},
\code{\link{read.ctd.odf}()},
\code{\link{read.ctd.sbe}()},
\code{\link{read.ctd.woce.other}()},
\code{\link{read.ctd.woce}()},
\code{\link{read.ctd}()},
\code{\link{setFlags,ctd-method}},
\code{\link{subset,ctd-method}},
\code{\link{summary,ctd-method}},
\code{\link{woceNames2oceNames}()},
\code{\link{woceUnit2oceUnit}()},
\code{\link{write.ctd}()}
}
\concept{functions relating to data-quality flags}
\concept{things related to ctd data}
