% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/argo.R
\name{subset,argo-method}
\alias{subset,argo-method}
\alias{subset.argo}
\title{Subset an argo Object}
\usage{
\S4method{subset}{argo}(x, subset, ...)
}
\arguments{
\item{x}{an \linkS4class{argo} object.}

\item{subset}{An expression indicating how to subset \code{x}.}

\item{...}{optional arguments, of which only the first is examined. The
only possibility is \code{within}, a polygon enclosing data to be
retained. This must be either a list or data frame, containing items
named either \code{x} and \code{y} or \code{longitude} and
\code{latitude}; see Example 4.  If \code{within} is given,
then \code{subset} is ignored.}
}
\value{
An \linkS4class{argo} object.
}
\description{
Subset an argo object, either by selecting just the "adjusted" data
or by subsetting by pressure or other variables.
}
\details{
If \code{subset} is the string \code{"adjusted"}, then \code{subset}
replaces the station variables with their adjusted counterparts. In
the argo notation, e.g. \code{PSAL} is replaced with \code{PSAL_ADJUSTED};
in the present notation, this means that \code{salinity} in the \code{data}
slot is replaced with \code{salinityAdjusted}, and the latter is deleted.
Similar replacements are also done with the flags stored in the \code{metadata}
slot.

If \code{subset} is an expression, then the action is somewhat similar
to other \code{subset} functions, but with the restriction that
only one independent variable may be
used in in any call to the function, so that
repeated calls will be necessary to subset based on more than one
independent variable.  Subsetting may be done by anything
stored in the data, e.g. \code{time},
\code{latitude}, \code{longitude}, \code{profile}, \code{dataMode},
or \code{pressure} or by \code{profile} (a made-up variable),
\code{id} (from the \code{metadata} slot) or \code{ID} (a synonym for \code{id}).
Note that subsetting by \code{pressure}
preserves matrix shape, by setting discarded values to \code{NA}, as opposed
to dropping data (as is the case with \code{time}, for example).
}
\section{Sample of Usage}{

\preformatted{
# Example 2: restrict attention to delayed-mode profiles.
par(mfrow=c(1, 1))
plot(subset(argo, dataMode == "D"))

# Example 3: contrast adjusted and unadjusted data
par(mfrow=c(1, 2))
plotTS(argo)
plotTS(subset(argo, "adjusted"))

# Example 2. Subset by a polygon determined with locator()
par(mfrow=c(1, 2))
plot(argo, which="map")
# Can get a boundary with e.g. locator(4)
boundary <- list(x=c(-65, -40, -40, -65), y=c(65, 65, 45, 45))
argoSubset <- subset(argo, within=boundary)
plot(argoSubset, which="map")
}
}

\examples{
library(oce)
data(argo)

# Example 1: subset by time, longitude, and pressure
par(mfrow = c(2, 2))
plot(argo)
plot(subset(argo, time > mean(time)))
plot(subset(argo, longitude > mean(longitude)))
plot(subset(argoGrid(argo), pressure > 500 & pressure < 1000), which = 5)

}
\seealso{
Other things related to argo data: 
\code{\link{[[,argo-method}},
\code{\link{[[<-,argo-method}},
\code{\link{argo-class}},
\code{\link{argoGrid}()},
\code{\link{argoNames2oceNames}()},
\code{\link{argo}},
\code{\link{as.argo}()},
\code{\link{handleFlags,argo-method}},
\code{\link{plot,argo-method}},
\code{\link{read.argo.copernicus}()},
\code{\link{read.argo}()},
\code{\link{summary,argo-method}}

Other functions that subset oce objects: 
\code{\link{subset,adp-method}},
\code{\link{subset,adv-method}},
\code{\link{subset,amsr-method}},
\code{\link{subset,cm-method}},
\code{\link{subset,coastline-method}},
\code{\link{subset,ctd-method}},
\code{\link{subset,echosounder-method}},
\code{\link{subset,lobo-method}},
\code{\link{subset,met-method}},
\code{\link{subset,oce-method}},
\code{\link{subset,odf-method}},
\code{\link{subset,rsk-method}},
\code{\link{subset,sealevel-method}},
\code{\link{subset,section-method}},
\code{\link{subset,topo-method}},
\code{\link{subset,xbt-method}}
}
\author{
Dan Kelley
}
\concept{functions that subset oce objects}
\concept{things related to argo data}
