#' Get Planning Areas (All)
#'
#' @description
#' This function is a wrapper for the \href{https://www.onemap.gov.sg/docs/#planning-area-polygons}{Planning Area Polygons API}. It returns the data either in raw format or a combined sf or sp object.
#'
#' @param token User's API token. This can be retrieved using \code{\link{get_token}}
#' @param year Optional, check \href{https://www.onemap.gov.sg/docs/#planning-area-polygons}{documentation} for valid options. Invalid requests will are ignored by the API.
#' @param read Optional, which package to use to read geojson object. For "sf" objects, specify \code{read = "sf"} and for "sp" objects use \code{read = "rgdal"}. Note that if used, any missing geojson objects will be dropped (this affects the "Others" planning area returned by the API).
#'
#' @return If the parameter \code{read} is not specified, the function returns a raw JSON object with planning names and geojson string vectors. \cr \cr
#' If \code{read = "sf"}, the function returns a single "sf" dataframe with 2 columns: "name" (name of planning area) and "geometry", which contains the simple features. \cr \cr
#' If \code{read = "rgdal"}, the function returns a SpatialPolygonsDataFrame of "sp" class. The names of each planning area is recorded in the "name" column of the dataframe. \cr \cr
#' If an error occurs, the function returns NULL and a warning message is printed.
#'
#' @note
#' If \code{read} is specified, any missing geojson objects will be dropped (this affects the "Others" planning area returned by the API). The returned outputs are NOT projected. \cr \cr
#' If the user specifies \code{read = "sp"}  but does not have the \code{sp} package installed, the function will return the raw JSON and print a warning message.
#'
#' @export
#'
#' @examples
#' # returns raw JSON object
#' \dontrun{get_planning_areas(token)}
#' \dontrun{get_planning_areas(token, 2008)}
#'
#' # returns dataframe of class "sf"
#' \dontrun{get_planning_areas(token, read = "sf")}
#'
#' # returns SpatialPolygonsDataFrame ("sp" object)
#' \dontrun{get_planning_areas(token, read = "rgdal")}
#'
#' # error: output is NULL, warning message shows status code
#' \dontrun{get_planning_areas("invalid_token")}


get_planning_areas <- function(token, year = NULL, read = NULL) {

  # query API
  url <- "https://developers.onemap.sg/privateapi/popapi/getAllPlanningarea"

  query <- paste(url, "?",
                 "token=", token,
                 sep = "")
  if (!is.null(year)) {
    query <- paste(query,
                   "&year=", year,
                   sep = "")
  }

  response <- GET(query)

  # error handling
  if (http_error(response)) {
    status <- status_code(response)
    output <- NULL
    warning(paste("The request produced a", status, "error", sep = " "))
    return(output)

  }

  # return output
  output <- content(response)

  # read into requested format
  if (is.null(read)) {
    return(output)

  } else {
    output_empty <- output %>%
      map_lgl(function(x) !is.null(x$geojson))

    output <- output[output_empty]

  }

  if (read %in% c("sf", "rgdal") & requireNamespace("sf", quietly = TRUE)) {
    output <- output %>%
      map(function(x) sf::st_sf(name = x$pln_area_n, geometry = flatten(sf::st_read(x$geojson, quiet = TRUE)))) %>%
      reduce(rbind)

    if (read == "rdgal") {
      output <- sf::as_Spatial(output)
    }

  } else {
    warning(paste0("Failed to read geojson. Please ensure you have package ", read, " installed."))
  }

  return(output)

}

