#####################
## plot.tTree
#####################
plot.tTree <- function(x, y=NULL, edge.col="black", col.edge.by="prob",
                              col.pal=NULL, annot=c("dist","n.gen","prob"), sep="/", ...){
    ## if(!require(igraph)) stop("igraph is required")
    ## if(!require(adegenet)) stop("adegenet is required")
    if(!inherits(x,"tTree")) stop("x is not a tTree object")
    if(!col.edge.by %in% c("dist","n.gen","prob")) stop("unknown col.edge.by specified")

    ## get graph ##
    g <- as.igraph(x, edge.col=edge.col, col.edge.by=col.edge.by, col.pal=col.pal, annot=annot, sep=sep)

     ## make plot ##
    plot(g, layout=attr(g,"layout"), ...)

    ## return graph invisibly ##
    return(invisible(g))

} # end plot.tTree






##############
## plotChains
##############
plotChains <- function(x, what="post", type=c("series","density"), burnin=0, dens.all=TRUE,
                       col=funky(x$n.runs), lty=1, lwd=1, main=what,
                       legend=TRUE, posi="bottomleft", ...){
    ## HANDLE ARGUMENTS ##
    type <- match.arg(type)
    n.runs <- x$n.runs
    col.ori <- col
    if(!what %in% names(x$chains)) stop(paste(what,"is not a column of x$chains"))
    if(!is.null(col)) col <- rep(col, length = n.runs)
    if(!is.null(lty)) lty <- rep(lty, length = n.runs)
    if(!is.null(lwd)) lwd <- rep(lwd, length = n.runs)
    if(is.null(burnin)){
        burnin <- max(x$burnin, x$find.import.at, x$tune.end)
    }

    ## GET DATA TO PLOT ##
    dat <- cbind(x$chains$step[x$chains$run==1],data.frame(split(x$chains[,what], x$chains$run)))
    names(dat) <- c("step", paste(what, 1:n.runs,sep=""))
    if(!any(dat$step>burnin)) stop("burnin is greater than the number of steps in x")
    dat <- dat[dat$step>burnin,,drop=FALSE]

    ## MAKE PLOT ##
    if(type=="series"){
        matplot(dat$step, dat[,-1,drop=FALSE], type="l", col=col, lty=lty, xlab="MCMC iteration", ylab="value", main=main, ...)
    }

    if(type=="density"){
        ## add general density if needed ##
        temp <- lapply(dat[, -1, drop=FALSE], density)
        if(dens.all){
            temp[[n.runs+1]] <- density(unlist(dat[,-1,drop=FALSE]))
            col <- c(col, "black")
            lty <- c(lty, 1)
            lwd <- c(lwd, 3)
            n.runs <- n.runs+1
        }
        range.x <- range(sapply(temp, function(e) e$x))
        range.y <- range(sapply(temp, function(e) e$y))
        plot(1,type="n", xlim=range.x, ylim=range.y, xlab="value", ylab="density", main=main, ...)
        invisible(lapply(1:n.runs, function(i) lines(temp[[i]], col=col[i], lty=lty[i], lwd=lwd[i])))
    }

    ## ADD LEGEND ##
    if(legend){
        legend(posi, fill=col, title="Runs", leg=1:length(col.ori))
    }
    return(invisible())
} # end plotChains






##############
## transGraph
##############
transGraph <- function(x, labels=NULL, burnin=x$burnin, threshold=0.2, col.pal=NULL, curved.edges=TRUE,
                       annot=c("dist","support"), sep="/", ...){
    ## CHECKS ##
    ## if(!require(igraph)) stop("igraph is required")
    ## if(!require(adegenet)) stop("adegenet is required")

    ## HANDLE ARGUMENTS ##
    if(burnin> max(x$chains$step)) stop("burnin exceeds the number of chains in the output")
    if(is.null(col.pal)){
        col.pal <- function(n){
            return(grey(seq(1,0,length=n)))
        }
    }

    ## GET ANCESTRY DATA ##
    ances <- x$chains[x$chains$step>=burnin, grep("alpha",names(x$chains)),drop=FALSE]
    tabances <- apply(ances,2,table)
    N <- ncol(ances)


    ## GET DATA FOR GRAPH ##
    ## ancestor, descendents, list of nodes
    to.old <- rep(1:N, sapply(tabances,length))
    from.old <- as.numeric(unlist(lapply(tabances,names)))
    from.old[from.old<1] <- NA
    to.old[to.old<1] <- NA
    isNotNA <- !is.na(from.old) & !is.na(to.old)
    vnames <- sort(unique(c(from.old,to.old)))
    from <- match(from.old,vnames)
    to <- match(to.old,vnames)

    ## support for the ancestries
    support <- unlist(lapply(tabances, function(e) e/sum(e)))[isNotNA]
    edge.col <- num2col(support, col.pal=col.pal, x.min=0, x.max=1)

    ## average dates of infection
    Tinf <- x$chains[x$chains$step>=burnin, grep("Tinf",names(x$chains)),drop=FALSE]
    inf.dates <- apply(Tinf,2,mean)
    names(inf.dates) <- 1:N

    ## remove weakly supported ancestries
    dat <- data.frame(from,to,stringsAsFactors=FALSE)[isNotNA,,drop=FALSE]
    if(sum(support>=threshold)==0) warning("threshold to high - no edge left")
    dat <- dat[support>=threshold, , drop=FALSE]

    ## convert to graph
    out <- graph.data.frame(dat, directed=TRUE, vertices=data.frame(names=vnames, dates=inf.dates[as.character(vnames)]))

    ## get ancestor->descendent mutations ##
    D <- as.matrix(x$D)
    findMut <- function(i){
        if(any(is.na(c(to[i],from[i])))) return(NA)
        return(D[to[i],from[i]])
    }
    nb.mut <- sapply(1:length(to), function(i) findMut(i))
    nb.mut <- nb.mut[isNotNA]


    ## SET PARAMETERS OF THE GRAPH ##
    ## vertices
    if(is.null(labels)){
        V(out)$label <- vnames
    } else {
        V(out)$label <- labels
    }


    ## edges
    E(out)$color <- edge.col[support>=threshold]
    E(out)$support <- support[support>=threshold]
    E(out)$curved <- curved.edges
    E(out)$nb.mut <- nb.mut[support>=threshold]

    ## edge labels
    lab <- ""
    if(!is.null(annot) && length(annot)>0){
        if(any(c("dist","nb.mut","mut") %in% annot)) lab <- E(out)$nb.mut
        if(any(c("support","prob") %in% annot)) lab <- paste(lab, round(E(out)$support,2), sep=sep)
    }
    lab <- sub(paste("^",sep,sep=""),"",lab)
    E(out)$label <- lab


    ## set layout
    attr(out, "layout") <- layout.fruchterman.reingold(out, params=list(minx=V(out)$dates, maxx=V(out)$dates), rescale=FALSE)


    ## MAKE THE PLOT ##
    plot(out, layout=attr(out, "layout"), ...)


    ## RETURN OBJECT ##
    return(invisible(out))

} # end transGraph






################
## plotOutbreak
################
.entropy <- function(p){
    p <- p/sum(p, na.rm=TRUE)
    return(-sum(p*log(p), na.rm=TRUE))
}

plotOutbreak <- function(x, burnin=x$burnin, thres.hide=0.2,
                         col=NULL, col.pal=colorRampPalette(c("blue","lightgrey")),
                         arr.col.pal=NULL, cex.bubble=1, lwd.arrow=2, xlim=NULL, ...){
    ## CHECKS ##
    ## if(!require(adegenet)) stop("adegenet is not installed")

    ## GET TREE ##
    tre <- get.tTree(x,burnin=burnin)
    N <- length(tre$idx)

    ## GET NUMBER OF MUTATIONS BETWEEN SEQUENCES
    M <- as.matrix(x$D)

    ## GET ALPHA_I ##
    alpha <- x$chains[x$chains$step>burnin, grep("alpha",names(x$chains))]
    f1 <- function(vec){
        sapply(1:N, function(i) mean(vec==i,na.rm=TRUE))
    }

    ## support for ancestries
    alphadat <- apply(alpha,2,f1)

    ## define colors for the individuals
    ## default: based on entropy of ancestries support
    if(is.null(col)){
        entropy <-apply(alphadat, 2, .entropy)
        col <- num2col(entropy, col.pal=col.pal)
    }


    ## PLOT INFECTION DATES ##
    toKeep <- grep("Tinf",names(x$chains))
    Tinf <- x$chains[x$chains$step>burnin, toKeep]
    colnames(Tinf) <- colnames(as.matrix(x$D))

    ## find max date
    if(is.null(xlim)){
        min.date <- min(x$chains[,grep("Tinf",names(x$chains))])
        max.date <- max(tre$inf.curves[[1]][,1])
        xlim <- c(min.date-.5,  max.date+.5)
    }

    ## basic boxplot
    boxplot(Tinf, col=col, horizontal=TRUE, las=1, ylim=xlim, ...)

    ## add infectious periods
    lapply(1:N, function(i) points(tre$inf.curves[[i]][,1], rep(i, nrow(tre$inf.curves[[i]])),
                                   cex=sqrt(tre$inf.curves[[i]][,2])*10*cex.bubble,
                                   col=transp(col)[i], pch=19) )

    ## plot collection dates
    points(x$collec.dates, 1:N, col="black", pch=20)


    ## ADD ANCESTRIES
    if(is.null(arr.col.pal)){
        arr.col.pal <- function(n){
            return(grey(seq(1,0,length=n)))
        }
    }

    ## function to draw arrows
    drawArrow <- function(from, to){
        if(is.na(from)||from<1) return(invisible())
        ## get stuff for arrows ##
        infdays <- apply(Tinf,2,mean)
        ##x.to <- x.from <- infdays[to] # for arrows on day of infection
        x.from <- infdays[from]
        x.to <- infdays[to]
        y.from <- from
        y.to <- to
        support <- alphadat[from,to]
        ## col <- col[from]
        arr.col <- num2col(support, x.min=0, x.max=1, col.pal=arr.col.pal)
        arr.col[support<thres.hide] <- "transparent"
        ## lwd <- round(support*arrow.max)
        ## col.back <- rep("transparent",N)
        ## col.back[lwd>=2] <- "black"

        ## draw arrows ##
        arrows(x.from, y.from, x.to, y.to, col=arr.col, length=0.1, angle=20, lwd=lwd.arrow)

        ## get stuff for annotations ##
        if(support>=thres.hide){
            x.ann <- (x.from + x.to)/2
            y.ann <- 0.15+(y.from + y.to)/2
            ## nb mut
            ann <- M[from,to]
            text(x.ann,y.ann,ann)
        }
        return(invisible())
    }


    ## draw all arrows
    ances <- apply(alpha,2,function(e) table(e)/sum(e))
    names(ances) <- 1:N
    lapply(1:N, function(i) sapply(as.integer(names(ances[[i]])), function(from) drawArrow(from, i)))


    ## BUILT RESULT AND RETURN ##
    res <- list(col=col, col.pal=col.pal, entropy=entropy, arr.col.pal=arr.col.pal)
    return(invisible(res))
} # end plotOutbreak











## #############
## ## epicurves
## #############
## epicurves <- function (x, col=NULL, bg="lightgrey", line.col="white", coef=1, max.arr=5,...) {
##     if(!require(adegenet)) stop("adegenet is required")
##     if(!inherits(x,"tTree")) stop("x is not a tTree object")

##     ## GET USEFUL INFO ##
##     N <- length(x$idx)
##     timeSpan <- range(x$inf.curves[[1]][,1])
##     if(is.null(col)){
##         colPal <- colorRampPalette(c("grey30","blue","green3","orange","red","brown4","purple"))
##         col <- colPal(N)
##     }


##     ## MAKE EMPTY PLOT ##
##     plot(0,0,type="n",xlim=timeSpan+c(-1,1), ylim=c(0,N+1), xlab="Dates",ylab="Individual index",...)
##     rect(timeSpan[1]-2,-2,timeSpan[2]+2,N+2, col=bg)
##     abline(h=1:N, col="white",lwd=3)
##     abline(h=1:N, col=transp(col),lwd=2)
##     abline(v=pretty(timeSpan,4), col=line.col)

##     ## DRAW INFECTIOUSNESS CURVES ##
##     for(i in 1:N){
##         temp <- x$inf.curves[[i]][x$inf.curves[[i]][,2]> 1e-12,,drop=FALSE]
##         x.coord <- c(temp[,1], rev(temp[,1]))
##         y.coord <- c(i+temp[,2]*coef, rep(i,nrow(temp)))
##         polygon(x.coord, y.coord, col=transp(col[i]),border=col[i])
##         points(temp[,1], i+(temp[,2]*coef), type="o", pch=20,cex=0.5, col=col[i])
##     }


##     ## ADD COLLECTION DATES ##
##     points(x$collec.dates, 1:N, pch=20, cex=2, col=col)


##     ## ADD INFECTIONS DATES ##
##     points(x$inf.dates, 1:N, cex=2, col=col)


##     ## ADD INFECTIONS ##
##     arr.w <- (x$p.ances- 1/(N-1)) * max.arr
##     arr.w[arr.w<0.5] <- 0.5
##     arrows(x$inf.date,x$ances, x$inf.dates, x$idx, angle=15, col=col[x$ances], lwd=arr.w)

## } # end epicurves
