% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/generate.R
\name{generateCovItems}
\alias{generateCovItems}
\title{Generate pairwise comparison data with random correlations between items}
\usage{
generateCovItems(df, numItems, th = 0.5, scale = 1, name)
}
\arguments{
\item{df}{a data frame with pairs of vertices given in columns \code{pa1} and \code{pa2}, and item response data in other columns}

\item{numItems}{how many items to create}

\item{th}{a vector of thresholds}

\item{scale}{the scaling constant}

\item{name}{a vector of item names}
}
\description{
If you need access to the correlation matrix used to generate the
absolute latent scores then you will need to generate them yourself.
This is not difficult. See how in the example.
}
\details{
The pairwise comparison item response model has thresholds and a
scale parameter similar to the partial credit model (Masters,
1982). The model is cumbersome to describe in traditional
mathematical notation, but the R code is fairly straightforward,

\preformatted{
softmax <- function(y) exp(y) / sum(exp(y))

cmp_probs <- function(scale, pa1, pa2, thRaw) {
  th <- cumsum(thRaw)
  diff <- scale * (pa2 - pa1)
  unsummed <- c(0, c(diff + rev(th)), c(diff - th), use.names = FALSE)
  softmax(cumsum(unsummed))
}
}

The function \code{cmp_probs} takes a \code{scale} constant, the
latent scores for two objects \code{pa1} and \code{pa2}, and a
vector of thresholds \code{thRaw}. The thresholds are parameterized
as the difference from the previous threshold. For example,
thresholds c(0.5, 0.5) are not at the same location but are at
locations c(0.5, 1.0). Thresholds are symmetric. If there is one
thresholds then the model admits three possible response outcomes
(e.g. win, tie, and lose). Responses are always stored centered
with zero representing a tie. Therefore, it is necessary to add one
plus the number of thresholds to response data to index into the
vector returned by \code{cmp_probs}. For example, if our response
data (-1, 0, 1) has one threshold then we would add 2 (1 + 1
threshold) to obtain the indices (1, 2, 3).

Use \code{\link{itemModelExplorer}} to explore the item model. In
this \pkg{shiny} app, the \emph{discrimination} parameter does what is
customary in item response models. However, it is not difficult to
show that discrimination is a function of thresholds and
scale. That is, discrimination is not an independent parameter and
cannot be estimated. In pairwise comparison models, discrimination
and measurement error are confounded.
}
\examples{
library(mvtnorm)
df <- twoLevelGraph(letters[1:10], 100)
df <- generateCovItems(df, 3)

# generateCovItems essentially does the same thing as:
numItems <- 3
palist <- unique(c(df$pa1,df$pa2))
trueCor <- cov2cor(rWishart(1, numItems, diag(numItems))[,,1])
theta <- rmvnorm(length(palist), sigma=trueCor)
dimnames(theta) <- list(palist, paste0('i', 3 + 1:numItems))
df <- generateItem(df, theta)

}
\references{
Masters, G. N. (1982). A Rasch model for partial credit scoring.
Psychometrika, 47, 149–174. doi: 10.1007/BF02296272
}
\seealso{
Other item generators: \code{\link{generateFactorItems}},
  \code{\link{generateItem}}
}
\concept{item generators}
