% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_mlpmms.R
\name{fit_mlpmms}
\alias{fit_mlpmms}
\title{Step 1 of PRC-MLPMM (estimation of the linear mixed models)}
\usage{
fit_mlpmms(y.names, fixefs, ranef.time, randint.items = TRUE, long.data,
  surv.data, t.from.base, n.boots = 0, n.cores = 1, verbose = TRUE,
  seed = 123, maxiter = 100, conv = rep(0.001, 3),
  lcmm.warnings = FALSE)
}
\arguments{
\item{y.names}{a list with the names of the
response variables which the MLPMMs have to be fitted to.
Each element in the list contains all the items used to 
reconstruct a latent biological process of interest}

\item{fixefs}{a fixed effects formula for the model, where the
time variable (specified also in \code{ranef.time}) is
included as first element and within the function 
\code{contrast()}. Examples: \code{~ contrast(age)}, 
\code{~ contrast(age) + group + treatment}}

\item{ranef.time}{a character with the name of the time variable 
for which to include a shared random slope}

\item{randint.items}{logical: should item-specific random intercepts
be included in the MLCMMs? Default is \code{TRUE}. It can also be a
vector, with different values for different elements of \code{y.names}}

\item{long.data}{a data frame with the longitudinal predictors,
comprehensive of a variable called \code{id} with the subject 
ids}

\item{surv.data}{a data frame with the survival data and (if 
relevant) additional baseline covariates. \code{surv.data} should at least
contain a subject id (called \code{id}), the time to event outcome  
(\code{time}), and binary event variable (\code{event})}

\item{t.from.base}{name of the variable containing time from 
baseline in \code{long.data}}

\item{n.boots}{number of bootstrap samples to be used in the
cluster bootstrap optimism correction procedure (CBOCP). If 0, no
bootstrapping is performed}

\item{n.cores}{number of cores to use to parallelize part of
the computations. If \code{ncores = 1} (default), 
no parallelization is done. Pro tip: you can use 
\code{parallel::detectCores()} to check how many 
cores are available on your computer}

\item{verbose}{if \code{TRUE} (default and recommended value), information
on the ongoing computations is printed in the console}

\item{seed}{random seed used for the bootstrap sampling. Default 
is \code{seed = 123}}

\item{maxiter}{maximum number of iterations to use when calling
the function \code{multlcmm}. Default is 100}

\item{conv}{a vector containing the three convergence criteria
(\code{convB}, \code{convL} and \code{convG}) to use when calling
the function \code{\link{multlcmm}}. Default is c(1e-3, 1e-3, 1e-3)}

\item{lcmm.warnings}{logical. If TRUE, a warning is printed every 
time the (strict) convergence criteria of the \code{multlcmm} function
are not met. Default is \code{FALSE}}
}
\value{
A list containing the following objects:
\itemize{
\item \code{call.info}: a list containing the following function
call information: \code{call}, \code{y.names}, \code{fixefs},
\code{ranef.time}, \code{randint.items};
\item \code{mlpmm.fits.orig}: a list with the MLPMMs fitted on the
original dataset (it should comprise as many MLPMMs as the elements
of \code{y.names} are);
\item \code{df.sanitized}: a sanitized version of the supplied 
\code{long.data} dataframe, without the
longitudinal measurements that are taken after the event
or after censoring;
\item \code{n.boots}: number of bootstrap samples;
\item \code{boot.ids}: a list with the ids of bootstrapped subjects 
(when \code{n.boots > 0});
\item \code{mlpmm.fits.boot}: a list of lists, which contains the MLPMMs 
fitted on each bootstrapped datasets (when \code{n.boots > 0}).
}
}
\description{
This function performs the first step for the estimation
of the PRC-MLPMM model proposed in Signorelli et al. (2021)
}
\details{
This function is essentially a wrapper of the 
\code{\link{multlcmm}} function that has the goal of simplifying
the estimation of several MLPMMs. In general, ensuring 
convergence of the algorithm implemented in \code{multlcmm}
is sometimes difficult, and it is hard to write a function that
can automatically solve these convergence problems. \code{fit_mplmms}
returns a warning when estimation did not converge for one or 
more MLPMMs. If this happens, try to change the convergence 
criteria in \code{conv} or the relevant \code{randint.items} value.
If doing this doesn't solve the problem, it is recommended to
re-estimate the specific MLPMMs for which estimation didn't converge
directly with \code{\link{multlcmm}}, trying to manually solve
the convergence issues
}
\examples{
\donttest{
# generate example data
set.seed(123)
n.items = c(4,2,2,3,4,2)
simdata = simulate_prcmlpmm_data(n = 100, p = length(n.items),  
             p.relev = 3, n.items = n.items, 
             type = 'u+b', seed = 1)
 
# specify options for cluster bootstrap optimism correction
# procedure and for parallel computing 
do.bootstrap = FALSE
# IMPORTANT: set do.bootstrap = TRUE to compute the optimism correction!
n.boots = ifelse(do.bootstrap, 100, 0)
more.cores = FALSE
# IMPORTANT: set more.cores = TRUE to speed computations up!
if (!more.cores) n.cores = 2
if (more.cores) {
   # identify number of available cores on your machine
   n.cores = parallel::detectCores()
   if (is.na(n.cores)) n.cores = 2
}

# step 1 of PRC-MLPMM: estimate the MLPMMs
y.names = vector('list', length(n.items))
for (i in 1:length(n.items)) {
  y.names[[i]] = paste('marker', i, '_', 1:n.items[i], sep = '')
}

step1 = fit_mlpmms(y.names, fixefs = ~ contrast(age),  
                 ranef.time = age, randint.items = TRUE, 
                 long.data = simdata$long.data, 
                 surv.data = simdata$surv.data,
                 t.from.base = t.from.base,
                 n.boots = n.boots, n.cores = n.cores)

# print MLPMM summary for marker 5 (all items involved in that MLPMM):
summary(step1, 'marker5_2')
}
}
\references{
Signorelli, M. (2023). pencal: an R Package for the Dynamic 
Prediction of Survival with Many Longitudinal Predictors.
arXiv preprint: arXiv:2309.15600

Signorelli, M., Spitali, P., Al-Khalili Szigyarto, C, 
The MARK-MD Consortium, Tsonaka, R. (2021). 
Penalized regression calibration: a method for the prediction 
of survival outcomes using complex longitudinal and 
high-dimensional data. Statistics in Medicine, 40 (27), 6178-6196.
DOI: 10.1002/sim.9178
}
\seealso{
\code{\link{simulate_prcmlpmm_data}},
\code{\link{summarize_mlpmms}} (step 2),
\code{\link{fit_prcmlpmm}} (step 3),
\code{\link{performance_prc}}
}
\author{
Mirko Signorelli
}
