% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gantt.R
\docType{methods}
\name{plot,gantt-method}
\alias{plot,gantt-method}
\alias{plot.gantt}
\title{Draw a Gantt diagram}
\usage{
\S4method{plot}{gantt}(x, xlim, time.format = NULL, time.labels.by,
  time.lines.by, event.time = NULL, event.label = NULL, event.side = 3,
  col.connector = "black", col.done = gray(0.3), col.notdone = gray(0.9),
  col.eventLine = gray(0.1), col.event = par("fg"),
  cex.event = par("cex"), font.event = par("font"),
  lty.eventLine = par("lty"), lwd.eventLine = par("lwd"), bg = par("bg"),
  grid.col = "lightgray", grid.lty = "dotted", ylabels = list(col = 1, cex
  = 1, font = 1, justification = 1), arrows = NULL, main = "",
  line.main = NA, cex.main = par("cex"), mgp = c(2, 0.7, 0),
  maiAdd = rep(0, 4), axes = TRUE, debug = FALSE, ...)
}
\arguments{
\item{x}{A \code{gantt} object, i.e. one inheriting from \code{\link{gantt-class}}.}

\item{xlim}{optional range of time axis; if not provided, the range of times
in \code{x} will be used.}

\item{time.format}{format for dates on time axis; defaults to 3-letter
month.}

\item{time.labels.by}{suggested label increment on time axis, e.g.
\code{time.labels.by="2 months"} to get a two-month interval.  If not
supplied, the axis will be generated automatically.}

\item{time.lines.by}{suggested interval between vertical grid lines on the
plot, e.g. \code{time.lines.by="1 week"} for weekly.  If not supplied, the
grid will be generated automatically.}

\item{event.time}{vector of event times, e.g. conferences, whose time cannot
be altered.}

\item{event.label}{vector of character strings holding event names.}

\item{event.side}{side for event labels.}

\item{col.connector}{colour of (optional) connectors between items.}

\item{col.done}{colour of work that has been done already. This may be a
vector of colours, one for each item in the gantt table.}

\item{col.notdone}{colour of work that has not been done yet. This may be a
vector of colours, one for each item in the gantt table.}

\item{col.eventLine}{colour of event lines; may be a vector.}

\item{col.event}{colour of event labels; may be a vector.}

\item{cex.event}{expansion factor for event labels; may be a vector.}

\item{font.event}{font for event labels; may be a vector.}

\item{lty.eventLine}{line type for event lines; may be a vector.}

\item{lwd.eventLine}{line width for event lines; may be a vector.}

\item{bg}{background colour for plot.}

\item{grid.col}{colour for grid.}

\item{grid.lty}{line type for grid.}

\item{ylabels}{A \code{\link{list}} with elements \code{col} for colour,
\code{cex} for character-expansion factor, \code{font} for font, and \code{justification}
for the placement in the margin (\code{0} means left-justified, and \code{1}
means right-justified. (NOTE: left-justification works poorly in RStudio, but
properly in other systems.)
It usually makes sense for the elements in \code{ylabels} to be vectors of the same
length as the topic list. However, shorter vectors are permitted, and they lengthened by
copying the default values at the end (see Example 6).}

\item{arrows}{A vector of strings, one for each topic, indicating the nature of
the arrows that may be drawn at the ends of task bars. The individual values
may be \code{"left"}, \code{"right"}, \code{"both"} or \code{"neither"}.
Set \code{arrows=NULL}, the default, to avoid such arrows.}

\item{main}{character string to be used as chart title.}

\item{line.main}{line where title occurs. If \code{NA}, then the
title is placed in a default location; otherwise, it is \code{line.main}
lines above the top of the plot.}

\item{cex.main}{numeric, font-size factor for title.}

\item{mgp}{setting for \code{\link{par}(mgp)}, within-axis spacing. The
default value tightens axis spacing.}

\item{maiAdd}{inches to add to the auto-computed margins at the bottom,
left, top, and right margins. The values may be negative (to tighten
margins) but the sum will be truncated to remain positive.}

\item{axes}{logical, \code{TRUE} to draw the x axis. (Setting to
\code{FALSE} permits detailed axis tweaking.)}

\item{debug}{logical value, \code{TRUE} to monitor the work.}

\item{...}{extra things handed down.}
}
\description{
Plot a gantt chart that shows the time allocated to a set of tasks, optionally
also with an indication of discrete events that occur as instants in time.
}
\details{
Time is indicated along the x axis, and tasks are stacked along the y
axis, akin to progress bars. Colour-coding can be used to indicate the degree of
completion of each task. These codes can be set individually for individual
tasks. Progress bars can have arrows (on either end), suggesting tasks
with flexible start/end dates or overdue tasks.  Vertical lines may
be drawn for discreet events. See \dQuote{Examples} for a few of the
possibilities.
}
\examples{
library(plan)
data(gantt)
summary(gantt)

# 1. Simple plot
plot(gantt)

# 2. Plot with two events
event.label <- c("Proposal", "AGU")
event.time <- c("2008-01-28", "2008-12-10")
plot(gantt, event.label=event.label,event.time=event.time)

# 3. Control x axis (months, say)
plot(gantt,labels=paste("M",1:6,sep=""))

# 4. Control task colours
plot(gantt,
     col.done=c("black", "red", rep("black", 10)),
     col.notdone=c("lightgray", "pink", rep("lightgray", 10)))

# 5. Control event colours (garish, to illustrate)
plot(gantt, event.time=event.time, event.label=event.label,
     lwd.eventLine=1:2, lty.eventLine=1:2,
     col.eventLine=c("pink", "lightblue"),
     col.event=c("red", "blue"), font.event=1:2, cex.event=1:2)

# 6. Top task is in bold font and red colour
plot(gantt,ylabels=list(col="red",font=2))

# 7. Demonstrate zero-time item (which becomes a heading)
gantt[["description"]][1] <- "Preliminaries"
gantt[["end"]][1] <- gantt[["start"]][1]
plot(gantt, ylabel=list(font=2, justification=0))

# 8. Arrows at task ends
plot(gantt, arrows=c("right","left","left","right"))
}
\references{
Gantt diagrams are described on wikipedia
\url{http://en.wikipedia.org/wiki/Gantt_Chart}.
}
\seealso{
Other things related to \code{gantt} data: \code{\link{as.gantt}},
  \code{\link{gantt-class}}, \code{\link{ganttAddTask}},
  \code{\link{gantt}}, \code{\link{read.gantt}},
  \code{\link{summary,gantt-method}}
}
\author{
Dan Kelley
}
