% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/portfolioBacktest.R
\name{portfolioBacktest}
\alias{portfolioBacktest}
\title{Backtest multiple portfolios over multiple datasets of stock prices in a rolling-window basis}
\usage{
portfolioBacktest(portfolio_funs = NULL, dataset_list,
  folder_path = NULL, price_name = "adjusted", paral_portfolios = 1,
  paral_datasets = 1, show_progress_bar = FALSE, benchmark = NULL,
  shortselling = TRUE, leverage = Inf, T_rolling_window = 252,
  optimize_every = 20, rebalance_every = 1, execution = c("same day",
  "next day"), cost = list(buy = 0, sell = 0, short = 0, long_leverage =
  0), cpu_time_limit = Inf, return_portfolio = TRUE,
  return_returns = TRUE)
}
\arguments{
\item{portfolio_funs}{List of functions (can also be a single function), each of them taking as input 
a dataset containing a list of \code{xts} objects (following the format of each 
element of the argument \code{dataset_list}) properly windowed (following the
rolling-window approach) and returning the portfolio as a vector of normalized
weights. See 
\href{https://CRAN.R-project.org/package=portfolioBacktest/vignettes/PortfolioBacktest.html}{vignette}
for details.}

\item{dataset_list}{List of datasets, each containing a list of \code{xts} objects, as generated
by the function \code{\link{stockDataResample}}.}

\item{folder_path}{If \code{portfolio_funs} is not defined, this should contain the path to a folder 
containing the portfolio functions saved in files. See 
\href{https://CRAN.R-project.org/package=portfolioBacktest/vignettes/PortfolioBacktest.html}{vignette}
for details.}

\item{price_name}{Name of the \code{xts} column in each dataset that contains the prices to be used in the portfolio return 
computation (default is \code{"adjusted"}).}

\item{paral_portfolios}{Interger indicating number of portfolios to be evaluated in parallel (default is \code{1}).}

\item{paral_datasets}{Interger indicating number of datasets to be evaluated in parallel (default is \code{1}).}

\item{show_progress_bar}{Logical value indicating whether to show progress bar (default is \code{FALSE}).}

\item{benchmark}{String vector indicating the benchmark portfolios to be incorporated, currently supports:
\itemize{\item{\code{uniform} - the uniform portfolio, \eqn{w = [1/N, ..., 1/N]} with \eqn{N} be number of stocks}
         \item{\code{index} - the market index, requires an \code{xts} named `index` in the datasets.}}}

\item{shortselling}{Logical value indicating whether shortselling is allowed or not 
(default is \code{TRUE}, so no control for shorselling in the backtesting).}

\item{leverage}{Amount of leverage as in \eqn{||w||_1 <= leverage} 
(default is \code{Inf}, so no control for leverage in the backtesting).}

\item{T_rolling_window}{Length of the lookback rolling window (default is \code{252}).}

\item{optimize_every}{How often the portfolio is to be optimized (default is \code{20}).}

\item{rebalance_every}{How often the portfolio is to be rebalanced (default is \code{1}).}

\item{execution}{String that can be either \code{"same day"} (default) or \code{"next day"}. 
At the rebalancing period \code{t}, the portfolio has used information up to (and including)
period \code{t}. Same day execution means one can get into the position at that period \code{t},
whereas the next day execution means that one can only get into the position the following day.}

\item{cost}{List containing four different types of transaction costs (common for all assets) 
for buying, selling, shorting, and long leveraging. The default is 
\code{cost = list(buy = 0e-4, sell = 0e-4, short = 0e-4, long_leverage = 0e-4)}.
If some elements are not specified then they will be automatically set to zero.}

\item{cpu_time_limit}{Time limit for executing each portfolio function over a single data set 
(default is \code{Inf}, so no time limit).}

\item{return_portfolio}{Logical value indicating whether to return the portfolios (default is \code{TRUE}).
Two portfolios are returned: \code{w_designed} is the designed portfolio at each
given rebalancing period (using all the information up to and including that period,
which can be executed either on the same day or the following day)
and \code{w_bop} is the "beginning-of-period" portfolio (i.e., at each period it contains 
the weights held in the market in the previous period so that the portfolio return at 
that period is just the product of the asset returns and \code{w_bop} at that period.)}

\item{return_returns}{Logical value indicating whether to return the portfolio returns (default is \code{TRUE}).
Two series are returned: \code{return} with the portfolio returns and \code{wealth}
with the portfolio wealth (aka cumulative P&L).}
}
\value{
List with the portfolio backtest results, see 
        \href{https://CRAN.R-project.org/package=portfolioBacktest/vignettes/PortfolioBacktest.html#result-format}{vignette-result-format}
        for details. It can be accessed directly, but we highly recommend the use of the package specific functions to extract 
        any required information, namely, \code{\link{backtestSelector}}, \code{\link{backtestTable}}, 
        \code{\link{backtestBoxPlot}}, \code{\link{backtestLeaderboard}},
        \code{\link{backtestSummary}}, \code{\link{summaryTable}}, \code{\link{summaryBarPlot}}.
}
\description{
Automated backtesting of multiple portfolios over multiple 
datasets of stock prices in a rolling-window fashion. 
Each portfolio design is easily defined as a
function that takes as input a window of the stock prices and outputs the 
portfolio weights. Multiple portfolios can be easily specified as a list 
of functions or as files in a folder. Multiple datasets can be conveniently 
obtained with the function \code{\link{stockDataResample}} that resamples
the data downloaded with the function \code{\link{stockDataDownload}}.
The results can be later assessed and arranged with tables and plots.
The backtesting can be highly time-consuming depending on the number of 
portfolios and datasets can be performed with parallel computation over
multiple cores. Errors in functions are properly catched and handled so
that the execution of the overal backtesting is not stopped (error messages
are stored for debugging purposes). See 
\href{https://CRAN.R-project.org/package=portfolioBacktest/vignettes/PortfolioBacktest.html}{vignette}
for a detailed explanation.
}
\examples{
library(portfolioBacktest)
data(dataset10)  # load dataset

# define your own portfolio function
uniform_portfolio <- function(dataset) {
  N <- ncol(dataset$adjusted)
  return(rep(1/N, N))
}

# do backtest
bt <- portfolioBacktest(list("Uniform" = uniform_portfolio), dataset10)

# check your result
names(bt)
backtestSelector(bt, portfolio_name = "Uniform", measures = c("Sharpe ratio", "max drawdown"))
backtestTable(bt, measures = c("Sharpe ratio", "max drawdown"))
bt_summary <- backtestSummary(bt)
summaryTable(bt_summary)

}
\seealso{
\code{\link{stockDataDownload}}, \code{\link{stockDataResample}},
         \code{\link{backtestSelector}}, \code{\link{backtestTable}}, 
         \code{\link{backtestBoxPlot}}, \code{\link{backtestLeaderboard}},
         \code{\link{backtestSummary}}, \code{\link{summaryTable}}, \code{\link{summaryBarPlot}}.
}
\author{
Daniel P. Palomar and Rui Zhou
}
