#' Set up potions for easy data retrieval
#' 
#' Start-up function to place a list into `options` with a specified 
#' slot name.
#' @param ... One or more `==` statements giving attributes to be stored; or 
#' alternatively a `list` containing the same.
#' @param file string: optional file containing data to be stored via `options()`.
#' Valid formats are `.yml` or `.json`. 
#' @param .slot string: optional name to mandate where data is stored. Defaults 
#' to a random string generated by `stringi::stri_rand_strings`.
#' @param .pkg string: package name that `potions` is being used within. If 
#' using `potions` within a package development process, set this argument 
#' instead of `slot`.
#' @importFrom rlang abort
#' @details 
#' The default method is to use `brew` and set either `.pkg` or `.slot`, but not
#' both. Alternatively you can use `brew_package()` or `brew_interactive()`. 
#' Note that if neither `.slot` or `.pkg` are set, `potions` defaults to `.slot`
#' , unless `.pkg` information has previously been supplied (and `.slot` 
#' information has not). This might be undesirable in a package development 
#' situation.
#' 
#' If both `...` and `file` arguments are empty, this function sets up an 
#' empty `potions` object in `options("potions-pkg")`; See `potions-class` for 
#' more information on this data type. If `...` and `file` arguments 
#' are provided, they will be amalgamated. If there are identical names in both
#' lists, those in `...` are chosen.
#'  
#' If the user repeatedly calls `brew()`, later list entries overwrite early 
#' entries. Whole lists are not overwritten unless all top-level entry names 
#' match.
#' @returns This function never returns an object; it is called for its' side-
#' effect of caching data using `options()`.
#' @examples 
#' # basic usage is to pass expressions
#' brew(x == 1)
#' 
#' # lists are also permitted
#' brew(list(x = 2))
#' 
#' # as are passing lists as objects
#' my_list <- list(x = 3)
#' brew(my_list)
#' 
#' # optional clean-up
#' drain()
#' @export
brew <- function(..., file, .slot, .pkg){
  
  # determine behavior based on supplied arguments
  has_slot <- !missing(.slot)
  has_pkg <- !missing(.pkg)
  if(has_slot){
    if(has_pkg){
      abort("Both `.slot` and `.pkg` have been set; please choose one")
    }
    brew_interactive(..., file = file, .slot = .slot)
  }else{
    if(has_pkg){
      brew_package(..., file = file, .pkg = .pkg)
    }else{ # if .slot and .pkg missing, choose based on call location
      package_check <- trace_back()$namespace |> 
        check_within_pkg()
      if(package_check$within){
        brew_package(..., file = file, .pkg = package_check$pkg)
      }else{
        lookup <- check_existing_slots()
        switch(lookup$method,
               "all_empty" = {brew_interactive(..., 
                                               file = file)}, # no data; .slot is random
               ".slot" = {brew_interactive(..., 
                                           file = file, 
                                           .slot = lookup$value)}) 
      }
    }
  }
}

#' @rdname brew
#' @importFrom rlang enquos
#' @export
brew_package <- function(..., file, .pkg){
  # parse dots
  dots <- enquos(..., .ignore_empty = "all")
  data <- parse_quosures(dots)
  # check supplied data
  data <- check_potions_data(data, file)
  # check package info
  check_is_character(.pkg)
  check_length_one(.pkg)
  # create current_list
  current_list <- check_potions_storage() |>
    update_package_names(.pkg) |>
    update_package_data(provided = data, pkg = .pkg)
  # push to options
  options(list("potions-pkg" = current_list))
}

#' @rdname brew
#' @importFrom rlang enquos
#' @export
brew_interactive <- function(..., file, .slot){
  # parse dots
  dots <- enquos(..., .ignore_empty = "all")
  data <- parse_quosures(dots)
  # check supplied data
  data <- check_potions_data(data, file) 
  .slot <- check_slot_name(.slot)
  current_list <- check_potions_storage() |> 
                  update_default_name(.slot = .slot) |>
                  update_slot_data(data, .slot)
  options(list("potions-pkg" = current_list))
}