% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/processanimateR.R
\name{animate_process}
\alias{animate_process}
\title{Animate cases on a process map}
\usage{
animate_process(eventlog, processmap = process_map(eventlog, render = F,
  ...), animation_mode = c("absolute", "relative", "off"),
  animation_duration = 60, animation_jitter = 0,
  animation_timeline = TRUE, animation_legend = NULL,
  token_size = NULL, token_size_scale = c("identity", "linear", "sqrt",
  "log", "quantize", "ordinal", "time"), token_size_scale_domain = NULL,
  token_size_scale_range = NULL, token_color = NULL,
  token_color_scale = c("identity", "linear", "sqrt", "log", "quantize",
  "ordinal", "time"), token_color_scale_domain = NULL,
  token_color_scale_range = NULL, token_image = NULL,
  token_opacity = NULL, token_shape = c("circle", "rect", "image"),
  token_attributes = NULL,
  token_callback_onclick = c("function(svg_root, svg_element, case_id) {",
  "}"),
  activity_callback_onclick = c("function(svg_root, svg_element, activity_id) {",
  "}"), elementId = NULL, preRenderHook = NULL, width = NULL,
  height = NULL, ...)
}
\arguments{
\item{eventlog}{The event log object that should be animated}

\item{processmap}{The process map created with processmapR that the event log should be animated on,
if not provided a standard process map will be generated by using processmapR::process_map.}

\item{animation_mode}{Whether to animate the cases according to their actual time of occurence ("absolute") or to start all cases at once ("relative").}

\item{animation_duration}{The overall duration of the animation, all times are scaled according to this overall duration.}

\item{animation_jitter}{The magnitude of a random coordinate translation, known as jitter in scatterplots, which is added to each token. Adding jitter can help to disambiguate tokens traveling on top of each other.}

\item{animation_timeline}{Whether to render a timeline slider in supported browsers (Recent versions of Chrome and Firefox only).}

\item{animation_legend}{Whether to show a legend for the 'size' or the 'color' scale.}

\item{token_size}{The event attribute (character) or alternatively a data frame with three columns (case, time, size) matching the case identifier of the supplied event log.
The token size is scaled accordingly during the animation (default size is 4). You may use \code{\link{add_token_size}} to add a suitable attribute to the event log.}

\item{token_size_scale}{Which D3 scale function to use for generating the size scale.}

\item{token_size_scale_domain}{The domain of the D3 scale function. Can be left NULL in which case it will be automatically determined.}

\item{token_size_scale_range}{The range of the D3 scale function. Should be a vector of two or more numerical values.}

\item{token_color}{The event attribute (character) or alternatively a data frame with three columns (case, time, color) matching the case identifier of the supplied event log.
The token color is change accordingly during the animation (default color is orange). You may use \code{\link{add_token_color}} to add a suitable attribute to the event log.}

\item{token_color_scale}{Which D3 scale function to use for generating the color scale.}

\item{token_color_scale_domain}{The domain of the D3 scale function. Can be left NULL in which case it will be automatically determined.}

\item{token_color_scale_range}{The range of the D3 scale function. Should be a vector of two or more numerical values.}

\item{token_image}{The event attribute (character) or alternatively a data frame with three columns (case, time, image) matching the case identifier of the supplied event log.
The token image is change accordingly during the animation (by default a SVG shape is used).}

\item{token_opacity}{The event attribute (character) or alternatively a data frame with three columns (case, time, transparency) matching the case identifier of the supplied event log.
The token fill-opacity is change accordingly during the animation (by default the token is drawn with 0.9 opacity).}

\item{token_shape}{The (fixed) SVG shape to be used to draw tokens. Can be either 'circle' (default), 'rect' or 'image'. In the latter case the image URL needs to be specified as parameter 'token_image'.}

\item{token_attributes}{A list of additional (fixed) SVG attributes to be added to each token.}

\item{token_callback_onclick}{A javascript function that is called when a token is clicked. The function is parsed by \code{\link{JS}} and received three parameters: 'svg_root', 'svg_element', and 'case_id'.}

\item{activity_callback_onclick}{A javascript function that is called when an activity is clicked. The function is parsed by \code{\link{JS}} and received three parameters: 'svg_root', 'svg_element', and 'activity_id'.}

\item{elementId}{passed through to \code{\link{createWidget}}. A custom elementId is useful to capture the selection events via input$elementId_tokens and input$elementId_activities when used in Shiny.}

\item{preRenderHook}{passed through to \code{\link{createWidget}}.}

\item{width, height}{Fixed size for widget (in css units). The default is NULL, which results in intelligent automatic sizing based on the widget's container.}

\item{...}{Options passed on to \code{\link{process_map}}.}
}
\description{
A function for creating a SVG animation of an event log on a process map created by processmapR.
}
\examples{
# Create small example event log
library(bupaR)
example_log <- bupaR::eventlog(
                 eventlog = data.frame(
                   case = c(rep(1,6),rep(2,6),rep(3,8)),
                   act = c(rep(c("A","A","B","B","C","C"),2),
                           rep(c("A", "B", "D", "C"), each = 2)),
                   act_id = rep(c(1:10),each=2),
                   lifecycle = rep(c("start", "complete"),10),
                   res = rep(sample(LETTERS[1:3], 10, replace = TRUE), each = 2),
                   time = as.POSIXct(c(0,5,10,15,20,25,
                                       5,10,15,20,25,30,
                                       10,20,30,40,50,60,70,80) * 600, origin = "2018/10/03")),
                 case_id = "case",
                 activity_id = "act",
                 timestamp = "time",
                 activity_instance_id = "act_id",
                 lifecycle_id = "lifecycle",
                 resource_id = "res")

# Animate the process with default options (absolute time and 60s duration)
animate_process(example_log)

\donttest{
# Animate the process with default options (relative time and with jitter)
animate_process(example_log, animation_mode = "relative", animation_jitter = 10)

# Change default token sizes
animate_process(example_log, token_size = 2)

# Change default token color
animate_process(example_log, token_color = "red")

# Change default token opacity
animate_process(example_log, token_opacity = 0.5)

# Change default token image (GIFs work too)
animate_process(example_log, token_shape = "image", token_size = 10,
                token_image = "https://upload.wikimedia.org/wikipedia/en/5/5f/Pacman.gif")

# Change token color based on a factor attribute
animate_process(example_log,
                animation_legend = "color", token_color = "res",
                token_color_scale = "ordinal",
                token_color_scale_range = RColorBrewer::brewer.pal(8, "Paired"))

# This requires example data
library(eventdataR)
data(traffic_fines)
# Change token color based on a numeric attribute, here the nonsensical 'time' of an event
animate_process(edeaR::filter_trace_frequency(bupaR::sample_n(traffic_fines,1000),percentage=0.95),
               animation_legend = "color", token_color = "amount",
               token_color_scale = "linear", token_color_scale_range = c("yellow","red"),
               animation_mode = "relative")
}

}
\seealso{
processmapR:process_map
}
\author{
Felix Mannhardt <felix.mannhardt@sintef.no> (SINTEF Digital)
}
