\name{ratiocalc}
\alias{ratiocalc}

\title{Calculation of ratios/propagated errors/t-statistics for all permutations/combinations of qPCR runs with or without reference data}

\description{For multiple qPCR data from type 'modlist' or 'pcrbatch', this function calculates ratios for all permutations/combinations of qPCR runs.
 This can be single qPCR data or data containing replicates. If qPCR runs for reference data are given, the target ratios are normalized
 against the ratios from the reference data. Propagated, Monte Carlo simulated and unpropagated errors (s.d. of evaluated expressions) are calculated for all ratios.
 Statistical significance for the ratios is calculated as t-statistics either from the crossing points or from efficiency/crossing points. 
}

\usage{ratiocalc(data, group = NULL, ratio = c("ind", "first"), 
	  which.eff = c("sig", "sli", "exp"), iter = c("combs", "perms"), 
	  rep.all = TRUE, ttest = c("cp", "Ecp"), ...)  
}

\arguments{
  \item{data}{multiple qPCR data generated by \code{\link{modlist}} or \code{\link{pcrbatch}}.}
  \item{group}{a numeric vector or factor defining the replicates (if any) as well as target and reference data. See 'Details'}.
  \item{ratio}{the efficiency value used for calculating the ratios. See 'Details'.}
  \item{which.eff}{in case of 'pcrbatch' data, efficiency from which method is to be used. Defaults to sigmoidal fit.}
  \item{iter}{type of iteration. Either all combinations (default) or permutations.}
  \item{rep.all}{logical. Should repeats be allowed for the iterations (i.e. 1-1)?}
  \item{ttest}{The parameters on which the t-test is to be conducted. Either crossing points or efficiency/crossing points. See 'Details'.}
  \item{...}{other parameters to be passed, i.e. to \code{\link{propagate}} or \code{\link{t.test}}.}
}

\details{
The \code{group} variable must be defined for the different target and reference runs. In general, target PCRs are defined by (replicate)
 numbers < 100, while reference PCRs are >= 100. Runs are matched by x vs. x + 100. If no grouping vector is defined, PCR runs are treated as single. 
 If reference PCRs are given, their number must match the number of target PCRs. Both target and reference data/grouping can be mixed but should be defined in ascending order,
 i.e. NOT c(1,1,2,2,102,102,101,101).\cr

Examples:\cr
No replicates: NULL (as is).\cr
Three runs with four replicates each: c(1, 1, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3) or gl(3, 4).\cr
Six single runs with reference data: c(1,2,3,4,5,6,101,102,103,104,105,106).\cr
Three runs with two replicates each and reference data:  c(1,101,1,101,2,102,2,102,3,103,3,103).\cr
Same as above but in different order: c(1,1,2,2,3,3,101,101,102,102,103,103).\cr

Ratios and propagated errors are calculated for all pairwise permutations/combinations of (replicated) runs and normalized against reference data, if these are supplied.
 Different values for the efficiency can be applied within the function such that the calculated ratios are based on individual efficiencies or efficiencies
 that are held constant for all runs. In detail this means for the different values of \code{ratio}:\cr

"ind":
\deqn{\frac{E_1^{cp1}}{E_2^{cp2}}/\frac{E_3^{cp3}}{E_4^{cp4}}}

"first":
\deqn{\frac{E_1^{cp1}}{E_1^{cp2}}/\frac{E_3^{cp3}}{E_3^{cp4}}}

numeric (E):
\deqn{\frac{E^{cp1}}{E^{cp2}}/\frac{E^{cp3}}{E^{cp4}}} 

with E1 (cp1), E2 (cp2), E3 (cp3), E4 (cp4) being the efficiencies (threshold cycles/crossing points) of target1, target2, reference1 and reference2 PCRs, respectively.   
The propagated errors are calculated by \code{\link{propagate}} and often seem quite high. This largely depends on the error of the base (i.e. efficiency)
 of the exponential function. The error usually decreases when setting \code{cov = TRUE} in the \code{...} part of the function. It can be debated anyhow,
 if the variables 'efficiency' and 'threshold cycles' have a covariance structure. As the efficiency is deduced at the second derivative maximum of the sigmoidal
 curve, variance in the second should show an effect on the first, such that the use of a var-covar matrix might be feasible. It is also commonly encountered
 that the propagated error is much higher when using reference data, as the number of partial derivatives functions increases. 
 The t-test can either be conducted on the crossing points (cp) or on \eqn{E^{cp}}, using the efficiency (E) as defined above. If reference data is supplied,
 the t-test is done using the delta-ct's (or \eqn{E^{\Delta ct's}}) from target/reference and/or control/reference. If \code{p.value = -1}, an error occurred in the t.test.
}

\value{
  A list with the following components:
  \item{ratios}{a dataframe containing the ratios, different errors (see \code{\link{propagate}}), 
		number of observations (\code{n}) and p-values from the t-test for each
		permutation/combination of runs/replicates (\code{p.value}).}
  \item{propList}{the complete list output from \code{\link{propagate}}.} 
  \item{exp.ratio}{the evaluated expression for calculating the ratios.}
  \item{exp.htest}{the evaluated expression for calculating the t-statistics.}  
}

\author{
  Andrej-Nikolai Spiess
}

\note{
When using a grouping regime, the column names in the \code{ratio} slot are always the names of the first run of each group.\cr
Also, when using \code{ratio = "first"}, combinations of the same two runs will not necessarily give reciprocal values, because
 different efficiencies are used for the calculation of the ratios. 
If confusion exists about the different error types, see \code{\link{propagate}} or \url{www.dr-spiess.de/qpcR/errors.pdf}.
}

\references{
Livak KJ \emph{et al.} (2001) Analysis of relative gene expression data using real-time quantitative PCR and the 2(-Delta Delta C(T)) method. \emph{Methods}, \bold{25}: 402-428.\cr
Tichopad A \emph{et al.} (2003) Standardized determination of real-time PCR efficiency from a single reaction set-up. \emph{Nucleic Acids Res}, \bold{31}: e122.\cr
Liu W & Saint DA (2002) Validation of a quantitative method for real time PCR kinetics. \emph{Biochem Biophys Res Commun}, \bold{294}: 347-53.\cr
Pfaffl M \emph{et al.} (2002) Relative expression software tool (REST) for group-wise comparison and statistical analysis of relative expression results in real-time PCR. \emph{Nucl Acids Res}, \bold{30}: e36. 
}


\examples{
## using 'modlist'
DAT <- modlist(reps, 2:9, fct = l5())
GROUP <- c(1, 1, 2, 2, 101, 101, 102, 102)
res <-  ratiocalc(DAT, group = GROUP)
print(res$ratio)

## using 'pcrbatch' and combinations
DAT2 <- pcrbatch(reps, 2:9, fct = l5())
GROUP <- c(1, 1, 2, 2, 101, 101, 102, 102)
res <-  ratiocalc(DAT2, group = GROUP, iter = "combs")
print(res$ratio)

## using only the efficiency estimate of the
## first curve
res <-  ratiocalc(DAT2, group = GROUP, ratio = "first")
print(res$ratio)

## using constant value for the efficiency
res <-  ratiocalc(DAT2, group = GROUP, ratio = "first")
print(res$ratio)

## strong differences in calculated error and
## simulated error indicate non-normality of
## propagated error
res <-  ratiocalc(DAT, group = GROUP, do.sim = TRUE)
print(res$ratio)

## Does error propagation in qPCR quantitation make sense?
## In ratio calculations based on (E1^cp1)/(E2^cp2),
## only 2\% error in each of the variables result in
## over 50\% propagated error!
x <- NULL
y <- NULL
for (i in seq(0, 0.1, by = 0.01)) {
      E1 <- c(1.7, 1.7 * i)
      cp1 <- c(15, 15 * i)
      E2 <- c(1.7, 1.7 * i)
      cp2 <- c(18, 18 * i)
      DF <- cbind(E1, cp1, E2, cp2)
      res <- propagate(expression((E1^cp1)/(E2^cp2)), DF, type = "stat")
      x <- c(x, i * 100)
      y <- c(y, (res$errProp/res$evalExpr) * 100)
}
plot(x, y, xlim = c(0, 10), lwd = 2, xlab = "c.v. [\%]", ylab = "c.v. (prop) [\%]")

}



\keyword{nonlinear}

