% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/wavelet.R
\name{wavelet}
\alias{wavelet}
\alias{wavelet_kernels}
\alias{morlet_wavelet}
\title{'Morlet' wavelet transform (Discrete)}
\usage{
wavelet_kernels(freqs, srate, wave_num)

morlet_wavelet(
  data,
  freqs,
  srate,
  wave_num,
  precision = c("float", "double"),
  trend = c("constant", "linear", "none"),
  ...
)
}
\arguments{
\item{freqs}{frequency in which \code{data} will be projected on}

\item{srate}{sample rate, number of time points per second}

\item{wave_num}{desired number of cycles in wavelet kernels to
balance the precision in time and amplitude (control the
smoothness); positive integers are strongly suggested}

\item{data}{numerical vector such as analog voltage signals}

\item{precision}{the precision of computation; choices are
\code{'float'} (default) and \code{'double'}.}

\item{trend}{choices are \code{'constant'}: center the signal at zero;
\code{'linear'}: remove the linear trend; \code{'none'} do nothing}

\item{...}{further passed to \code{\link{detrend}};}
}
\value{
\code{wavelet_kernels} returns wavelet kernels to be
used for wavelet function; \code{morlet_wavelet} returns a file-based array
if \code{precision} is \code{'float'}, or a list of real and imaginary
arrays if \code{precision} is \code{'double'}
}
\description{
Transform analog voltage signals with 'Morlet'
wavelets: complex wavelet kernels with \eqn{\pi/2} phase
differences.
}
\examples{


# generate sine waves
time <- seq(0, 3, by = 0.01)
x <- sin(time * 20*pi) + exp(-time^2) * cos(time * 10*pi)

plot(time, x, type = 'l')

# freq from 1 - 15 Hz; wavelet using float precision
freq <- seq(1, 15, 0.2)
coef <- morlet_wavelet(x, freq, 100, c(2,3))

# to get coefficients in complex number from 1-10 time points
coef[1:10, ]

# power
power <- Mod(coef[])^2

# Power peaks at 5Hz and 10Hz at early stages
# After 1.0 second, 5Hz component fade away
image(power, x = time, y = freq, ylab = "frequency")

# wavelet using double precision
coef2 <- morlet_wavelet(x, freq, 100, c(2,3), precision = "double")
power2 <- (coef2$real[])^2 + (coef2$imag[])^2

image(power2, x = time, y = freq, ylab = "frequency")

# The maximum relative change of power with different precisions
max(abs(power/power2 - 1))

# display kernels
freq <- seq(1, 15, 1)
kern <- wavelet_kernels(freq, 100, c(2,3))
print(kern)

plot(kern)

}
