% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/profile.R
\name{step_profile}
\alias{step_profile}
\title{Create a Profiling Version of a Data Set}
\usage{
step_profile(
  recipe,
  ...,
  profile = NULL,
  pct = 0.5,
  index = 1,
  grid = list(pctl = TRUE, len = 100),
  columns = NULL,
  role = NA,
  trained = FALSE,
  skip = FALSE,
  id = rand_id("profile")
)
}
\arguments{
\item{recipe}{A recipe object. The step will be added to the
sequence of operations for this recipe.}

\item{...}{One or more selector functions to choose variables
for this step. See \code{\link[=selections]{selections()}} for more details.}

\item{profile}{A call to \code{\link[dplyr:vars]{dplyr::vars()}}) to specify which
variable will be profiled (see \code{\link[=selections]{selections()}}). If a column is
included in both lists to be fixed and to be profiled, an error
is thrown.}

\item{pct}{A value between 0 and 1 that is the percentile to
fix continuous variables. This is applied to all continuous
variables captured by the selectors. For date variables, either
the minimum, median, or maximum used based on their distance to
\code{pct}.}

\item{index}{The level that qualitative variables will be
fixed. If the variables are character (not factors), this will
be the index of the sorted unique values. This is applied to all
qualitative variables captured by the selectors.}

\item{grid}{A named list with elements \code{pctl} (a logical) and
\code{len} (an integer). If \code{pctl = TRUE}, then \code{len} denotes how
many percentiles to use to create the profiling grid. This
creates a grid between 0 and 1 and the profile is determined by
the percentiles of the data. For example, if \code{pctl = TRUE} and
\code{len = 3}, the profile would contain the minimum, median, and
maximum values. If \code{pctl = FALSE}, it defines how many grid
points between the minimum and maximum values should be created.
This parameter is ignored for qualitative variables (since all
of their possible levels are profiled). In the case of date
variables, \code{pctl = FALSE} will always be used since there is no
quantile method for dates.}

\item{columns}{A character string that contains the names of
columns that should be fixed and their values. These values are
not determined until \code{\link[=prep.recipe]{prep.recipe()}} is called.}

\item{role}{Not used by this step since no new variables are
created.}

\item{trained}{A logical to indicate if the quantities for
preprocessing have been estimated.}

\item{skip}{A logical. Should the step be skipped when the
recipe is baked by \code{\link[=bake.recipe]{bake.recipe()}}? While all operations are baked
when \code{\link[=prep.recipe]{prep.recipe()}} is run, some operations may not be able to be
conducted on new data (e.g. processing the outcome variable(s)).
Care should be taken when using \code{skip = TRUE} as it may affect
the computations for subsequent operations.}

\item{id}{A character string that is unique to this step to identify it.}
}
\value{
An updated version of \code{recipe} with the new step added to the
sequence of any existing operations.
}
\description{
\code{step_profile} creates a \emph{specification} of a recipe step that
will fix the levels of all variables but one and will create a
sequence of values for the remaining variable. This step can be
helpful when creating partial regression plots for additive
models.
}
\details{
This step is atypical in that, when baked, the
\code{new_data} argument is ignored; the resulting data set is
based on the fixed and profiled variable's information.

When you \code{\link[=tidy]{tidy()}} this step, a tibble with columns \code{terms} (which
is the columns that will be affected) and \code{type} (fixed or
profiled) is returned.
}
\examples{
library(modeldata)
data(okc)

# Setup a grid across date but keep the other values fixed
recipe(~ diet + height + date, data = okc) \%>\%
  step_profile(-date, profile = vars(date)) \%>\%
  prep(training = okc) \%>\%
  juice


##########

# An *additive* model; not for use when there are interactions or
# other functional relationships between predictors

lin_mod <- lm(mpg ~ poly(disp, 2) + cyl + hp, data = mtcars)

# Show the difference in the two grid creation methods

disp_pctl <- recipe(~ disp + cyl + hp, data = mtcars) \%>\%
  step_profile(-disp, profile = vars(disp)) \%>\%
  prep(training = mtcars)

disp_grid <- recipe(~ disp + cyl + hp, data = mtcars) \%>\%
  step_profile(
    -disp,
    profile = vars(disp),
    grid = list(pctl = FALSE, len = 100)
  ) \%>\%
  prep(training = mtcars)

grid_data <- bake(disp_grid, new_data = NULL)
grid_data <- grid_data \%>\%
  mutate(pred = predict(lin_mod, grid_data),
         method = "grid")

pctl_data <- bake(disp_pctl, new_data = NULL)
pctl_data <- pctl_data \%>\%
  mutate(pred = predict(lin_mod, pctl_data),
         method = "percentile")

plot_data <- bind_rows(grid_data, pctl_data)

library(ggplot2)

ggplot(plot_data, aes(x = disp, y = pred)) +
  geom_point(alpha = .5, cex = 1) +
  facet_wrap(~ method)
}
