#' @name exportArms
#'
#' @title Export the Arms for a Project
#' @description This function allows you to export the Arms for a project
#'   Note: this only works for longitudinal projects
#'
#' @param rcon A REDCap connection object as generated by \code{redcapConnection}.
#' @param arms \code{character} identifying the arm numbers to retrieve. 
#'   May also be \code{integerish}.
#' @param ... Arguments to be passed to other methods.
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}}
#' @param config \code{list} Additional configuration parameters to pass to 
#'   \code{\link[httr]{POST}}. These are appended to any parameters in 
#'   \code{rcon$config}.
#' @param api_param \code{list} Additional API parameters to pass into the
#'   body of the API call. This provides users to execute calls with options
#'   that may not otherwise be supported by \code{redcapAPI}.
#' 
#' @details
#' It is not sufficient to make the project a longitudinal project. The
#' project must satisfy one of two conditions: 1) have at least two arms and
#' one event defined in each arm; or 2) have one arm and at least two events defined. If 
#' neither of these conditions are satisfied, the API will return a message
#' such as \code{ERROR: You cannot export arms for classic projects}, an 
#' error message that isn't as descriptive of the nature of the problem as 
#' we might like.
#' 
#' If the project information reports that the project is not longitudinal, 
#' a data frame with 0 rows is returned without calling the API. 
#' 
#' @section REDCap API Documentation:
#' This function allows you to export the Arms for a project
#' 
#' NOTE: this only works for longitudinal projects.
#' 
#' @section REDCap Version:
#' 5.8.2+ 
#'
#' @return 
#' Returns a data frame with two columns
#' 
#' \itemize{
#'   \item{\code{arm_num} }{The arm number}
#'   \item{\code{name} }{The arm's descriptive name}
#' }
#'
#' @author Benjamin Nutter
#'
#' @references
#' Please refer to your institution's API documentation.
#'
#' Additional details on API parameters are found on the package wiki at
#' \url{https://github.com/vubiostat/redcapAPI/wiki/REDCap-API-Parameters}
#' 
#' @export
#'

exportArms <- function(rcon, ...){
  UseMethod("exportArms")
}

#' @rdname exportArms
#' @export

exportArms.redcapApiConnection <- function(rcon, 
                                           arms = character(0), 
                                           ...,
                                           error_handling = getOption("redcap_error_handling"), 
                                           config = list(), 
                                           api_param = list())
{
  if (is.numeric(arms)) arms <- as.character(arms)
  
  # Argument Validation ---------------------------------------------
  coll <- checkmate::makeAssertCollection()

  checkmate::assert_class(x = rcon,
                          classes = "redcapApiConnection",
                          add = coll)
  
  checkmate::assert_character(x = arms,
                              add = coll)
  
  error_handling <- checkmate::matchArg(x = error_handling, 
                                        choices = c("null", "error"),
                                        .var.name = "error_handling",
                                        add = coll)
  
  checkmate::assert_list(x = config, 
                         names = "named", 
                         add = coll)
  
  checkmate::assert_list(x = api_param, 
                         names = "named", 
                         add = coll)
  
  checkmate::reportAssertions(coll)
  
  if(rcon$projectInformation()$is_longitudinal == 0){
    return(REDCAP_ARMS_STRUCTURE) # defined in redcapDataStructure.R
  }
  
  # Build the body list ---------------------------------------------
  body <- c(list(token = rcon$token, 
                 content = 'arm', 
                 format = 'csv', 
                 returnFormat = 'csv'), 
            vectorToApiBodyList(arms, 
                                parameter_name = "arms"))
  
  body <- body[lengths(body) > 0]
  
  
  # API Call --------------------------------------------------------
  response <- makeApiCall(rcon, 
                          body = c(body, api_param), 
                          config = config)
  
  if (response$status_code != 200){
    redcap_error(response, 
                 error_handling = error_handling)
  }
  
  utils::read.csv(text = as.character(response), 
                  stringsAsFactors = FALSE,
                  na.strings = "")
}
