#
#   This work was created by the National Center for Ecological Analysis and Synthesis.
#
#     Copyright 2015 Regents of the University of California
#
#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.
#
#' @title Create, query and write RDF graphs.
#' @description The R package \emph{redland} provides methods to create, query and write information stored in the
#' Resource Description Framework (RDF). This package is implemented as R scripts that provide an
#' R interface (aka "wrapper") to the Redland RDF C libraries. Documentation for the redland R package classes
#' and functions are available from the standard R help facility, for example,  \code{'help("Node-class")'}, \code{'?getNodeType'}, etc.
#'
#' An overview of the redland R package is available with the R command: \code{'vignette("redland_overview")'}.
#'
#' The Redland C library functions are described at \url{http://librdf.org/docs/api/index.html}.
#'
#' An introduction to RDF can be found at \url{http://www.w3.org/TR/rdf-primer/}.
#'
#' @details
#' The redland R package classes and the corresponding Redland C library types are shown in the following table:
#' \tabular{llll}{
#' \strong{Concept}\tab \strong{Redland C type}\tab \strong{redland R class}\tab \strong{Purpose}\cr
#' Resource / Literal\tab librdf_node\tab \linkS4class{Node}\tab RDF Model & Syntax nodes\cr
#' Statement / Triple\tab librdf_statement\tab \linkS4class{Statement}\tab RDF Model & Syntax arcs (statements, triples)\cr
#' Model\tab librdf_model\tab \linkS4class{Model}\tab Set of Statements usually held in one Storage.\cr
#' Node\tab librdf_node\tab \linkS4class{Node}\tab The subject, predicate or object of a Statement\cr
#' Storage\tab librdf_storage\tab \linkS4class{Storage}\tab Storage for Models either persistent or in-memory.\cr
#' Parser\tab librdf_parser\tab \linkS4class{Parser}\tab Syntax parsers delivering Stream of Statements or writing to a Model\cr
#' Query\tab librdf_query\tab \linkS4class{Query}\tab Querying of an Model delivering a QueryResults\cr
#' QueryResults\tab librdf_query_results\tab \linkS4class{QueryResults}\tab Results of applying an Query to a Model giving either variable bindings with Node values or Stream of Statements\cr
#' Serializer\tab librdf_serializer\tab \linkS4class{Serializer}\tab Serializes a Model into a syntax such as RDF/XML\cr
#' World\tab librdf_world\tab \linkS4class{World}\tab RDF wrapper class handling Redland startup/shutdown\cr
#' }
#'
#' @note
#' In order to communicate with the Redland RDF C libraries, the redland R package uses an interface layer that is created
#' with the software package \emph{Simplified Wrapper and Interface Generator} (\href{https://github.com/swig/swig}{SWIG}).
#' The relationship between the redland R package and the Redland C libraries is:
#'
#' User script -> redland R package -> SWIG R interface -> Redland C libraries -> RDF data
#'
#' It is recommended that the redland package R classes be used to interact with RDF, as these higher level classes take care of many of the the
#' details of communicating with the Redland C libraries. However, all of the lower level R interface functions generated by SWIG are made available
#' by the redland package. These interface functions usually have names beginning with \code{'librdf_'}, \code{'rasqal_'} or \code{'raptor_'} and
#' are usually the same name as the underlying C library function. Documentation
#' for the R SWIG interface functions can be found via R help i.e. \code{'?librdf_iterator'}.
#'
#' @examples
#' # This example creates the necessary R objects to hold an RDF model and reads 
#' # in a file that contains RDF/XML statements. This model is then queried for 
#' # and the query results inspected.
#' world <- new("World")
#' storage <- new("Storage", world, "hashes", name="", options="hash-type='memory'")
#' model <- new("Model", world, storage, options="")
#' filePath <- system.file("extdata/example.rdf", package="redland")
#' parser <- new("Parser", world)
#' parseFileIntoModel(parser, world, filePath, model)
#' queryString <- paste("PREFIX dc: <http://purl.org/dc/elements/1.1/> ",
#'                      "SELECT ?a ?c WHERE { ?a dc:description ?c . }", sep="")
#' query <- new("Query", world, queryString, base_uri=NULL, 
#'              query_language="sparql", query_uri=NULL)
#' results <- getResults(query, model, "rdfxml")
#' 
#' # When the query object is no longer needed, the resources it had allocated can be freed.
#' freeQuery(query)
#' rm(query)
#' @docType package
#' @name redland
#' @author Matthew B. Jones (NCEAS) and Peter Slaughter (NCEAS)
NULL
