% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_MO2.R
\encoding{UTF-8}
\name{calc_MO2}
\alias{calc_MO2}
\title{Calculate metabolic rate}
\usage{
calc_MO2(
  duration,
  o2,
  o2_unit = "percent_a.s.",
  bin_width,
  vol,
  temp = 25,
  sal = 35,
  atm_pres = 1013.25,
  time,
  pH,
  good_data = TRUE
)
}
\arguments{
\item{duration}{numeric vector of the timepoint for each observation (minutes).}

\item{o2}{numeric vector of O2 observations.}

\item{o2_unit}{a string describing the unit used to measure \code{o2}. Default is "percent_a.s." Options are from \code{\link{conv_o2}}.}

\item{bin_width}{numeric or data frame. \itemize{
\item{OPTION 1: }{OPTION 1: A single number defining how long of a period should be binned for each MO2 determination (minutes). If MO2 is to be calculated from one observation to the next (rather than binned observations), set \code{bin_width} to 0. To calculate a single MO2 value from all observations, set \code{bin_width} to \code{Inf}.}
\item{OPTION 2: }{A data frame with two numeric columns: "o2" and "width" generated by \code{\link{make_bins}}. Useful for Pcrit calculations or another application where bins of different widths are desired at different PO2s. For each row, set the "width" value to the bin duration (minutes) desired for observations <= the value in the "o2" column.}
}}

\item{vol}{volume of the respirometer (liter).}

\item{temp}{temperature (°C). Default is 25 °C.}

\item{sal}{salinity (psu). Default is 35 psu.}

\item{atm_pres}{atmospheric pressure (mbar). Default is 1013.25 mbar.}

\item{time}{(optional). Numeric vector of timestamp observations.}

\item{pH}{(optional). Numeric vector of pH observations.}

\item{good_data}{logical vector of whether O2 observations are "good" measurements and should be included in analysis. Linear regressions will not be fit over bins that include "bad" data. Bins will be split at bad data points. Default is that all observations are \code{TRUE}.}
}
\value{
A data frame is returned:
\describe{
\item{DUR_MEAN}{Mean duration of the bin (minutes).}
\item{DUR_RANGE}{Range of duration timepoints in the bin.}
\item{TIME_MEAN}{Exists only if the parameter \code{time} has values. Mean timestamp of the bin.}
\item{TIME_RANGE}{Exists only if the parameter \code{time} has values. Range of timestamps in the bin.}
\item{TEMP_MEAN}{Mean temperature of the bin.}
\item{PH_MEAN}{Exists only if the parameter \code{pH} has values. Mean pH of the bin. Averaged using \code{mean_pH()}.}
\item{O2_MEAN}{Mean O2 value of the bin in the unit chosen by \code{o2_unit}).}
\item{O2_RANGE}{Range of O2 values in the bin.}
\item{MO2}{Metabolic rate (umol O2 / hour).}
\item{R2}{Coefficient of determination for the linear regression fit to calculate MO2.}
\item{N}{Number of observations in the bin.}
}
}
\description{
Calculates metabolic rate (MO2) given O2 measurements over time. Oxygen measurements are split into bins and MO2s are calculated from each bin (unless \code{bin_width} is set to \code{0}). The \code{bin_width} parameter defines the width of the bins in timed intervals (e.g. 15 minutes). Linear regressions are fit through each bin and the calculated MO2 is returned as the slope of the change in O2 over time.
}
\note{
Whole-animal MO2 is returned. If mass-specific MO2 is desired, the output from \code{calc_MO2} can be divided by the animal's mass.
No matter what unit of oxygen partial pressure or concentration measurement you put into the function as \code{o2_unit}, the output in the MO2 column is always expressed in umol O2 / hour. This is because there is a vast variety of units for which people prefer to report dissolved oxygen levels, but most physiologists are at least unified in reporting metabolic rate as umol O2 per hour. If you prefer to report MO2 as mg O2 per hour, for example, you can always do something like:
\code{conv_resp_unit(df$MO2, from = 'umol_O2 / hr', 'mg_O2 / hr')}
If only beginning and ending O2 observations are known, consider using \code{\link{closed}}. Both functions will work fine, but \code{\link{closed}} is simpler.
}
\examples{
# get O2 data
file <- system.file('extdata', 'witrox_file.txt', package = 'respirometry')
o2_data <- na.omit(import_witrox(file, split_channels = TRUE)$CH_4)

# calculate MO2
(mo2_5_min <- calc_MO2(duration = o2_data$DURATION, o2 = o2_data$O2,
bin_width = 5, vol = 10, temp = o2_data$TEMP, sal = o2_data$SAL))

# what if measurements from the 10 to 12 minute marks can't be trusted?
bad_data = o2_data$DURATION >= 10 & o2_data$DURATION <= 12
(mo2_5_min <- calc_MO2(duration = o2_data$DURATION, o2 = o2_data$O2,
bin_width = 5, vol = 10, temp = o2_data$TEMP, sal = o2_data$SAL, good_data = !bad_data))

# easily make a Pcrit plot
plot(mo2_5_min$O2_MEAN, mo2_5_min$MO2)

# I want to express MO2 in mg per min instead.
(mo2_5_min$MO2 <- conv_resp_unit(value = mo2_5_min$MO2, from = 'umol_O2 / hr', to = 'mg_O2 / min'))

# just endpoint measurement:
calc_MO2(duration = o2_data$DURATION, o2 = o2_data$O2,
bin_width = Inf, vol = 10, temp = o2_data$TEMP, sal = o2_data$SAL)

# In my trial, observations above 77\% air saturation were really noisy, but much less noisy at
# lower O2 values. I want to adjust my bin width based on the PO2 to obtain the best balance of
# resolution and precision throughout the whole trial. Below 77\% a.s., use 4 minute bins. Above
# 77\% a.s. use 10 minute bins.
bins = data.frame(o2 = c(77, 100), width = c(4, 10))
calc_MO2(duration = o2_data$DURATION, o2 = o2_data$O2,
bin_width = bins, vol = 10, temp = o2_data$TEMP, sal = o2_data$SAL)

}
\seealso{
\code{\link{make_bins}}, \code{\link{calc_b}}, \code{\link{closed}}, \code{\link{scale_MO2}}, \code{\link{conv_resp_unit}}
}
\author{
Matthew A. Birk, \email{matthewabirk@gmail.com}
}
