#' @title Generate param files to prepare for trial simulations
#' @export
#' @seealso [get_facts_file_example()], [run_engine()], [run_engine_contin()]
#' @description Generate the preparatory files required for simulation.
#' @details For advanced control over trial simulations, you must first call
#'   [run_flfll()] and then call one of the engine functions
#'   such as [run_engine_contin()].
#'   [run_flfll()] generates the preparatory `*.param` files that the
#'   `run_engine_*()` functions understand.
#'   You will pass these `*.param` files or their parent
#'   directory to `param_files` argument of [run_engine_contin()] etc.
#' @return Character, the value of `output_path`.
#'   `output_path` is the directory path to the files generated by
#'   `run_flfll()`.
#' @inheritParams get_facts_version
#' @param output_path Character, directory path to the files to generate.
#' @param log_path Character, path to the log file generated by FLFLL.
#' @param n_burn Number of burn-in iterations for the MCMC.
#' @param n_mcmc Number of MCMC iterations used in inference.
#' @param n_weeks_files Number of `weeks*.csv` files to save in `output_path`.
#' @param n_patients_files Number of `patients*.csv` files to save in
#'   `output_path`.
#' @param n_mcmc_files Number of `mcmc*.csv` files to save in
#'   `output_path`.
#' @param n_mcmc_thin Number of thinning iterations for the MCMC.
#' @param flfll_seed Positive integer, random number generator seed for FLFLL.
#'   This seed is only used for stochastic preprocessing steps for generating
#'   the `*.param` files. It is not the random number generator seed
#'   for the actual trial simulations. To set the trial simulation seed,
#'   use the `seed` argument of [run_facts()], [run_engine()], or
#'   one of the specific `run_engine*()` functions.
#' @param flfll_offset Integer, offset for the random number generator.
#' @param verbose Logical, whether to print progress information to the
#'   R console.
#' @examples
#' # Can only run if system dependencies are configured:
#' if (file.exists(Sys.getenv("RFACTS_PATHS"))) {
#' facts_file <- get_facts_file_example("contin.facts") # example FACTS file
#' out <- run_flfll(facts_file, verbose = FALSE) # Generate param files.
#' # Run the simulations.
#' run_engine(
#'   facts_file,
#'   param_files = out,
#'   n_sims = 1,
#'   verbose = FALSE
#' )
#' read_patients(out)
#' }
run_flfll <- function(
  facts_file,
  output_path = tempfile(),
  log_path = output_path,
  n_burn = NULL,
  n_mcmc = NULL,
  n_weeks_files = 1e4,
  n_patients_files = 1e4,
  n_mcmc_files = 0,
  n_mcmc_thin = NULL,
  flfll_seed = NULL,
  flfll_offset = NULL,
  verbose = FALSE
) {
  assert_files_exist(facts_file)
  tmp_dir <- tempfile()
  dir.create(tmp_dir)
  new_file <- file.path(tmp_dir, basename(facts_file))
  file.copy(from = facts_file, to = new_file)
  paths <- get_mono_flfll_paths(facts_file)
  args <- c(
    paths$flfll,
    arg_character("-file", new_file),
    arg_character("-outputPath", output_path),
    arg_character("-logPath", log_path),
    # Disables auto packetization. The engine controls num sims, not FLFLL.
    arg_integer("-nSim", 1),
    arg_integer("-nBurn", n_burn),
    arg_integer("-nMCMC", n_mcmc),
    arg_integer("-nWeeksFiles", n_weeks_files),
    arg_integer("-nSubjectFiles", n_patients_files),
    arg_integer("-nMCMCFiles", n_mcmc_files),
    arg_integer("-nMCMCThin", n_mcmc_thin),
    arg_integer("-seed", flfll_seed),
    arg_integer("-offset", flfll_offset)
  )
  run_linux(command = paths$mono, args = args, verbose = verbose)
  output_path
}
