% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/occ_data.R
\name{occ_data}
\alias{occ_data}
\title{Search for GBIF occurrences - simplified for speed}
\usage{
occ_data(taxonKey = NULL, scientificName = NULL, country = NULL,
  publishingCountry = NULL, hasCoordinate = NULL, typeStatus = NULL,
  recordNumber = NULL, lastInterpreted = NULL, continent = NULL,
  geometry = NULL, recordedBy = NULL, basisOfRecord = NULL,
  datasetKey = NULL, eventDate = NULL, catalogNumber = NULL,
  year = NULL, month = NULL, decimalLatitude = NULL,
  decimalLongitude = NULL, elevation = NULL, depth = NULL,
  institutionCode = NULL, collectionCode = NULL,
  hasGeospatialIssue = NULL, issue = NULL, search = NULL,
  mediaType = NULL, limit = 500, start = 0, ...)
}
\arguments{
\item{taxonKey}{A taxon key from the GBIF backbone. All included and synonym taxa
are included in the search, so a search for aves with taxononKey=212
(i.e. /occurrence/search?taxonKey=212) will match all birds, no matter which
species. You can pass many keys by passing occ_search in a call to an
lapply-family function (see last example below).}

\item{scientificName}{A scientific name from the GBIF backbone. All included and synonym
taxa are included in the search.}

\item{country}{The 2-letter country code (as per ISO-3166-1) of the country in
which the occurrence was recorded. See here
\url{http://en.wikipedia.org/wiki/ISO_3166-1_alpha-2}}

\item{publishingCountry}{The 2-letter country code (as per ISO-3166-1) of the
country in which the occurrence was recorded.}

\item{hasCoordinate}{(logical) Return only occurence records with lat/long data (TRUE) or
all records (FALSE, default).}

\item{typeStatus}{Type status of the specimen. One of many options. See ?typestatus}

\item{recordNumber}{Number recorded by collector of the data, different from GBIF record
number. See \url{http://rs.tdwg.org/dwc/terms/#recordNumber} for more info}

\item{lastInterpreted}{Date the record was last modified in GBIF, in ISO 8601 format:
yyyy, yyyy-MM, yyyy-MM-dd, or MM-dd.  Supports range queries, smaller,larger (e.g.,
'1990,1991', whereas '1991,1990' wouldn't work)}

\item{continent}{Continent. One of africa, antarctica, asia, europe, north_america
(North America includes the Caribbean and reachies down and includes Panama), oceania,
or south_america}

\item{geometry}{Searches for occurrences inside a polygon described in Well Known
Text (WKT) format. A WKT shape written as either POINT, LINESTRING, LINEARRING
or POLYGON. Example of a polygon: ((30.1 10.1, 20, 20 40, 40 40, 30.1 10.1))
 would be queried as \url{http://bit.ly/1BzNwDq}.}

\item{recordedBy}{The person who recorded the occurrence.}

\item{basisOfRecord}{Basis of record, as defined in our BasisOfRecord enum here
\url{http://gbif.github.io/gbif-api/apidocs/org/gbif/api/vocabulary/BasisOfRecord.html}
Acceptable values are:
\itemize{
  \item FOSSIL_SPECIMEN An occurrence record describing a fossilized specimen.
  \item HUMAN_OBSERVATION An occurrence record describing an observation made by
  one or more people.
  \item LITERATURE An occurrence record based on literature alone.
  \item LIVING_SPECIMEN An occurrence record describing a living specimen, e.g.
  \item MACHINE_OBSERVATION An occurrence record describing an observation made
  by a machine.
  \item OBSERVATION An occurrence record describing an observation.
  \item PRESERVED_SPECIMEN An occurrence record describing a preserved specimen.
  \item UNKNOWN Unknown basis for the record.
}}

\item{datasetKey}{The occurrence dataset key (a uuid)}

\item{eventDate}{Occurrence date in ISO 8601 format: yyyy, yyyy-MM, yyyy-MM-dd, or
MM-dd. Supports range queries, smaller,larger (e.g., '1990,1991', whereas '1991,1990'
wouldn't work)}

\item{catalogNumber}{An identifier of any form assigned by the source within a
physical collection or digital dataset for the record which may not unique,
but should be fairly unique in combination with the institution and collection code.}

\item{year}{The 4 digit year. A year of 98 will be interpreted as AD 98. Supports range queries,
smaller,larger (e.g., '1990,1991', whereas '1991,1990' wouldn't work)}

\item{month}{The month of the year, starting with 1 for January. Supports range queries,
smaller,larger (e.g., '1,2', whereas '2,1' wouldn't work)}

\item{decimalLatitude}{Latitude in decimals between -90 and 90 based on WGS 84.
Supports range queries, smaller,larger (e.g., '25,30', whereas '30,25' wouldn't work)}

\item{decimalLongitude}{Longitude in decimals between -180 and 180 based on WGS 84.
Supports range queries (e.g., '-0.4,-0.2', whereas '-0.2,-0.4' wouldn't work).}

\item{elevation}{Elevation in meters above sea level. Supports range queries, smaller,larger
(e.g., '5,30', whereas '30,5' wouldn't work)}

\item{depth}{Depth in meters relative to elevation. For example 10 meters below a
lake surface with given elevation. Supports range queries, smaller,larger (e.g., '5,30',
whereas '30,5' wouldn't work)}

\item{institutionCode}{An identifier of any form assigned by the source to identify
the institution the record belongs to. Not guaranteed to be que.}

\item{collectionCode}{An identifier of any form assigned by the source to identify
the physical collection or digital dataset uniquely within the text of an institution.}

\item{hasGeospatialIssue}{(logical) Includes/excludes occurrence records which contain spatial
issues (as determined in our record interpretation), i.e. \code{hasGeospatialIssue=TRUE}
returns only those records with spatial issues while \code{hasGeospatialIssue=FALSE} includes
only records without spatial issues. The absence of this parameter returns any
record with or without spatial issues.}

\item{issue}{(character) One or more of many possible issues with each occurrence record. See
Details. Issues passed to this parameter filter results by the issue.}

\item{search}{Query terms. The value for this parameter can be a simple word or a phrase.}

\item{mediaType}{Media type. Default is NULL, so no filtering on mediatype. Options:
NULL, 'MovingImage', 'Sound', and 'StillImage'.}

\item{limit}{Number of records to return. Default: 500. Note that the per request maximum is
300, but since we set it at 500 for the function, we do two requests to get you the 500
records (if there are that many). Note that there is a hard maximum of 200,000, which is
calculated as the \code{limit+start}, so \code{start=199,000} and \code{limit=2000} won't work}

\item{start}{Record number to start at. Use in combination with limit to page through results.
Note that we do the paging internally for you, but you can manually set the \code{start}
parameter}

\item{...}{Further named parameters, such as \code{query}, \code{path}, etc, passed on to
\code{\link[httr]{modify_url}} within \code{\link[httr]{GET}} call. Unnamed parameters will
be combined with \code{\link[httr]{config}}.}
}
\value{
A data.frame or list
}
\description{
Search for GBIF occurrences - simplified for speed
}
\details{
Note that you can pass in a vector to one of taxonkey, datasetKey, and
catalogNumber parameters in a function call, but not a vector >1 of the three
parameters at the same time

\bold{Hierarchies:} hierarchies are returned wih each occurrence object. There is no
option no to return them from the API. However, within the \code{occ_search}
function you can select whether to return just hierarchies, just data, all of
data and hiearchies and metadata, or just metadata. If all hierarchies are the
same we just return one for you.

\bold{Data:} By default only three data fields are returned: name (the species name),
decimallatitude, and decimallongitude. Set parameter minimal=FALSE if you want more data.

\bold{Nerds:} You can pass parameters not defined in this function into the call to
the GBIF API to control things about the call itself using \code{...}. See an example below
that passes in the \code{verbose} function to get details on the http call.

\bold{Scientific names vs. taxon keys:} In the previous GBIF API and the version of rgbif that
wrapped that API, you could search the equivalent of this function with a species name, which
was convenient. However, names are messy right. So it sorta makes sense to sort out the species
key numbers you want exactly, and then get your occurrence data with this function. GBIF has
added a parameter scientificName to allow searches by scientific names in this function - which
includes synonym taxa. \emph{Note:} that if you do use the scientificName parameter, we will
check internally that it's not a synonym of an accepted name, and if it is, we'll search on the
accepted name. If you want to force searching by a synonym do so by finding the GBIF identifier
first with any \code{name_*} functions, then pass that ID to the \code{taxonKey} parameter.

\bold{WKT:} Examples of valid WKT objects:
\itemize{
 \item 'POLYGON((30.1 10.1, 10 20, 20 60, 60 60, 30.1 10.1))'
 \item 'POINT(30.1 10.1)'
 \item 'LINESTRING(3 4,10 50,20 25)'
 \item 'LINEARRING' ???' - Not sure how to specify this. Anyone?
}

\bold{Range queries:} A range query is as it sounds - you query on a range of values defined by
a lower and upper limit. Do a range query by specifying the lower and upper limit in a vector
like \code{depth='50,100'}. It would be more R like to specify the range in a vector like
\code{c(50,100)}, but that sort of syntax allows you to do many searches, one for each element in
the vector - thus range queries have to differ. The following parameters support range queries.
\itemize{
 \item decimalLatitude
 \item decimalLongitude
 \item depth
 \item elevation
 \item eventDate
 \item lastInterpreted
 \item month
 \item year
}

\bold{Issue:} The options for the issue parameter (from
http://gbif.github.io/gbif-api/apidocs/org/gbif/api/vocabulary/OccurrenceIssue.html):
\itemize{
 \item BASIS_OF_RECORD_INVALID The given basis of record is impossible to interpret or seriously
 different from the recommended vocabulary.
 \item CONTINENT_COUNTRY_MISMATCH The interpreted continent and country do not match up.
 \item CONTINENT_DERIVED_FROM_COORDINATES The interpreted continent is based on the coordinates,
 not the verbatim string information.
 \item CONTINENT_INVALID Uninterpretable continent values found.
 \item COORDINATE_INVALID Coordinate value given in some form but GBIF is unable to interpret it.
 \item COORDINATE_OUT_OF_RANGE Coordinate has invalid lat/lon values out of their decimal max
 range.
 \item COORDINATE_REPROJECTED The original coordinate was successfully reprojected from a
 different geodetic datum to WGS84.
 \item COORDINATE_REPROJECTION_FAILED The given decimal latitude and longitude could not be
 reprojected to WGS84 based on the provided datum.
 \item COORDINATE_REPROJECTION_SUSPICIOUS Indicates successful coordinate reprojection according
 to provided datum, but which results in a datum shift larger than 0.1 decimal degrees.
 \item COORDINATE_ROUNDED Original coordinate modified by rounding to 5 decimals.
 \item COUNTRY_COORDINATE_MISMATCH The interpreted occurrence coordinates fall outside of the
 indicated country.
 \item COUNTRY_DERIVED_FROM_COORDINATES The interpreted country is based on the coordinates, not
 the verbatim string information.
 \item COUNTRY_INVALID Uninterpretable country values found.
 \item COUNTRY_MISMATCH Interpreted country for dwc:country and dwc:countryCode contradict each
 other.
 \item DEPTH_MIN_MAX_SWAPPED Set if supplied min>max
 \item DEPTH_NON_NUMERIC Set if depth is a non numeric value
 \item DEPTH_NOT_METRIC Set if supplied depth is not given in the metric system, for example
 using feet instead of meters
 \item DEPTH_UNLIKELY Set if depth is larger than 11.000m or negative.
 \item ELEVATION_MIN_MAX_SWAPPED Set if supplied min > max elevation
 \item ELEVATION_NON_NUMERIC Set if elevation is a non numeric value
 \item ELEVATION_NOT_METRIC Set if supplied elevation is not given in the metric system, for
 example using feet instead of meters
 \item ELEVATION_UNLIKELY Set if elevation is above the troposphere (17km) or below 11km
 (Mariana Trench).
 \item GEODETIC_DATUM_ASSUMED_WGS84 Indicating that the interpreted coordinates assume they are
 based on WGS84 datum as the datum was either not indicated or interpretable.
 \item GEODETIC_DATUM_INVALID The geodetic datum given could not be interpreted.
 \item IDENTIFIED_DATE_INVALID The date given for dwc:dateIdentified is invalid and cant be
 interpreted at all.
 \item IDENTIFIED_DATE_UNLIKELY The date given for dwc:dateIdentified is in the future or before
 Linnean times (1700).
 \item MODIFIED_DATE_INVALID A (partial) invalid date is given for dc:modified, such as a non
 existing date, invalid zero month, etc.
 \item MODIFIED_DATE_UNLIKELY The date given for dc:modified is in the future or predates unix
 time (1970).
 \item MULTIMEDIA_DATE_INVALID An invalid date is given for dc:created of a multimedia object.
 \item MULTIMEDIA_URI_INVALID An invalid uri is given for a multimedia object.
 \item PRESUMED_NEGATED_LATITUDE Latitude appears to be negated, e.g. 32.3 instead of -32.3
 \item PRESUMED_NEGATED_LONGITUDE Longitude appears to be negated, e.g. 32.3 instead of -32.3
 \item PRESUMED_SWAPPED_COORDINATE Latitude and longitude appear to be swapped.
 \item RECORDED_DATE_INVALID A (partial) invalid date is given, such as a non existing date,
 invalid zero month, etc.
 \item RECORDED_DATE_MISMATCH The recording date specified as the eventDate string and the
 individual year, month, day are contradicting.
 \item RECORDED_DATE_UNLIKELY The recording date is highly unlikely, falling either into the
 future or represents a very old date before 1600 that predates modern taxonomy.
 \item REFERENCES_URI_INVALID An invalid uri is given for dc:references.
 \item TAXON_MATCH_FUZZY Matching to the taxonomic backbone can only be done using a fuzzy, non
 exact match.
 \item TAXON_MATCH_HIGHERRANK Matching to the taxonomic backbone can only be done on a higher
 rank and not the scientific name.
 \item TAXON_MATCH_NONE Matching to the taxonomic backbone cannot be done cause there was no
 match at all or several matches with too little information to keep them apart (homonyms).
 \item TYPE_STATUS_INVALID The given type status is impossible to interpret or seriously
 different from the recommended vocabulary.
 \item ZERO_COORDINATE Coordinate is the exact 0/0 coordinate, often indicating a bad null
 coordinate.
}

\bold{Counts:} There is a slight difference in the way records are counted here vs.
results from \code{\link{occ_count}}. For equivalent outcomes, in this function
use \code{hasCoordinate=TRUE}, and \code{hasGeospatialIssue=FALSE} to have the
same outcome using \code{\link{occ_count}} with \code{isGeoreferenced=TRUE}.

This does nearly the same thing as \code{\link{occ_search}}, but is a bit simplified
for speed, and is for the most common use case where user just wants the data, and not
other information like taxon hierarchies and media (e.g., images) information. Alot of time
in \code{\link{occ_search}} is used parsing data to be more useable downstream. We do
less of that in this function.
}
\examples{
\dontrun{
(key <- name_backbone(name='Encelia californica')$speciesKey)
occ_data(taxonKey = key, limit = 4)
(res <- occ_data(taxonKey = key, limit = 400))

# Return 20 results, this is the default by the way
(key <- name_suggest(q='Helianthus annuus', rank='species')$key[1])
occ_data(taxonKey=key, limit=20)

# Instead of getting a taxon key first, you can search for a name directly
## However, note that using this approach (with \\code{scientificName="..."})
## you are getting synonyms too. The results for using \\code{scientifcName} and
## \\code{taxonKey} parameters are the same in this case, but I wouldn't be surprised if for some
## names they return different results
library("httr")
occ_data(scientificName = 'Ursus americanus', config=verbose())
key <- name_backbone(name = 'Ursus americanus', rank='species')$usageKey
occ_data(taxonKey = key)

# Search by dataset key
occ_data(datasetKey='7b5d6a48-f762-11e1-a439-00145eb45e9a', limit=10)

# Search by catalog number
occ_data(catalogNumber="49366", limit=10)
occ_data(catalogNumber=c("49366","Bird.27847588"), limit=10)

# Use paging parameters (limit and start) to page. Note the different results
# for the two queries below.
occ_data(datasetKey='7b5d6a48-f762-11e1-a439-00145eb45e9a',start=10,limit=5)
occ_data(datasetKey='7b5d6a48-f762-11e1-a439-00145eb45e9a',start=20,limit=5)

# Many dataset keys
occ_data(datasetKey=c("50c9509d-22c7-4a22-a47d-8c48425ef4a7",
   "7b5d6a48-f762-11e1-a439-00145eb45e9a"), limit=20)

# Search by recorder
occ_data(recordedBy="smith", limit=20)

# Many collector names
occ_data(recordedBy=c("smith","BJ Stacey"), limit=20)

# Pass in curl options for extra fun
library('httr')
occ_data(taxonKey=key, limit=20, config=verbose())
x <- occ_data(taxonKey=key, limit=50, config=progress())
# occ_data(taxonKey=key, limit=20, config=timeout(0.01))

# Search for many species
splist <- c('Cyanocitta stelleri', 'Junco hyemalis', 'Aix sponsa')
keys <- sapply(splist, function(x) name_suggest(x)$key[1], USE.NAMES=FALSE)
occ_data(taxonKey=keys, limit=5)

# Search using a synonym name
#  Note that you'll see a message printing out that the accepted name will be used
occ_data(scientificName = 'Pulsatilla patens', limit=5)

# Search on latitidue and longitude
occ_data(decimalLatitude=40, decimalLongitude=-120, limit = 10)

# Search on a bounding box
## in well known text format
occ_data(geometry='POLYGON((30.1 10.1, 10 20, 20 40, 40 40, 30.1 10.1))', limit=20)
key <- name_suggest(q='Aesculus hippocastanum')$key[1]
occ_data(taxonKey=key, geometry='POLYGON((30.1 10.1, 10 20, 20 40, 40 40, 30.1 10.1))',
   limit=20)
## or using bounding box, converted to WKT internally
occ_data(geometry=c(-125.0,38.4,-121.8,40.9), limit=20)

# Search on country
occ_data(country='US', limit=20)
isocodes[grep("France", isocodes$name),"code"]
occ_data(country='FR', limit=20)
occ_data(country='DE', limit=20)
occ_data(country=c('US','DE'), limit=20)

# Get only occurrences with lat/long data
occ_data(taxonKey=key, hasCoordinate=TRUE, limit=20)

# Get only occurrences that were recorded as living specimens
occ_data(taxonKey=key, basisOfRecord="LIVING_SPECIMEN", hasCoordinate=TRUE, limit=20)

# Get occurrences for a particular eventDate
occ_data(taxonKey=key, eventDate="2013", limit=20)
occ_data(taxonKey=key, year="2013", limit=20)
occ_data(taxonKey=key, month="6", limit=20)

# Get occurrences based on depth
key <- name_backbone(name='Salmo salar', kingdom='animals')$speciesKey
occ_data(taxonKey=key, depth=1, limit=20)

# Get occurrences based on elevation
key <- name_backbone(name='Puma concolor', kingdom='animals')$speciesKey
occ_data(taxonKey=key, elevation=50, hasCoordinate=TRUE, limit=20)

# Get occurrences based on institutionCode
occ_data(institutionCode="TLMF", limit=20)
occ_data(institutionCode=c("TLMF","ArtDatabanken"), limit=20)

# Get occurrences based on collectionCode
occ_data(collectionCode="Floristic Databases MV - Higher Plants", limit=20)
occ_data(collectionCode=c("Floristic Databases MV - Higher Plants","Artport"))

# Get only those occurrences with spatial issues
occ_data(taxonKey=key, hasGeospatialIssue=TRUE, limit=20)

# Search using a query string
occ_data(search="kingfisher", limit=20)

# Range queries
## See Detail for parameters that support range queries
occ_data(depth='50,100') # this is a range depth, with lower/upper limits in character string
occ_data(depth=c(50,100)) # this is not a range search, but does two searches for each depth

## Range search with year
occ_data(year='1999,2000', limit=20)

## Range search with latitude
occ_data(decimalLatitude='29.59,29.6')

# Search by specimen type status
## Look for possible values of the typeStatus parameter looking at the typestatus dataset
head(occ_data(typeStatus = 'allotype')$data[,c('name','typeStatus')])

# Search by specimen record number
## This is the record number of the person/group that submitted the data, not GBIF's numbers
## You can see that many different groups have record number 1, so not super helpful
head(occ_data(recordNumber = 1)$data[,c('name','recordNumber','recordedBy')])

# Search by last time interpreted: Date the record was last modified in GBIF
## The lastInterpreted parameter accepts ISO 8601 format dates, including
## yyyy, yyyy-MM, yyyy-MM-dd, or MM-dd. Range queries are accepted for lastInterpreted
occ_data(lastInterpreted = '2015-09-02')

# Search for occurrences with images
occ_data(mediaType = 'StillImage')
occ_data(mediaType = 'MovingImage')
occ_data(mediaType = 'Sound')

# Search by continent
## One of africa, antarctica, asia, europe, north_america, oceania, or south_america
occ_data(continent = 'south_america')$meta
occ_data(continent = 'africa')$meta
occ_data(continent = 'oceania')$meta
occ_data(continent = 'antarctica')$meta

# Query based on issues - see Details for options
## one issue
x <- occ_data(taxonKey=1, issue='DEPTH_UNLIKELY')
x$data[,c('name','key','decimalLatitude','decimalLongitude','depth')]
## two issues
occ_data(taxonKey=1, issue=c('DEPTH_UNLIKELY','COORDINATE_ROUNDED'))
# Show all records in the Arizona State Lichen Collection that cant be matched to the GBIF
# backbone properly:
occ_data(datasetKey='84c0e1a0-f762-11e1-a439-00145eb45e9a',
   issue=c('TAXON_MATCH_NONE','TAXON_MATCH_HIGHERRANK'))

# Parsing output by issue
(res <- occ_data(geometry='POLYGON((30.1 10.1, 10 20, 20 40, 40 40, 30.1 10.1))', limit = 50))
## what do issues mean, can print whole table, or search for matches
head(gbif_issues())
gbif_issues()[ gbif_issues()$code \%in\% c('cdround','cudc','gass84','txmathi'), ]
## or parse issues in various ways
### remove data rows with certain issue classes
library('magrittr')
res \%>\% occ_issues(gass84)
### split issues into separate columns
res \%>\% occ_issues(mutate = "split")
### expand issues to more descriptive names
res \%>\% occ_issues(mutate = "expand")
### split and expand
res \%>\% occ_issues(mutate = "split_expand")
### split, expand, and remove an issue class
res \%>\% occ_issues(-cudc, mutate = "split_expand")
}
}
\references{
\url{http://www.gbif.org/developer/occurrence#search}
}
\seealso{
\code{\link{downloads}}, \code{\link{occ_search}}
}

