% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/R6Map.R
\docType{data}
\name{Map}
\alias{Map}
\title{R6 object (Map) to display Earth Engine (EE) spatial objects}
\format{
An object of class environment with the
following functions:
\itemize{
\item  \strong{addLayer(eeObject, visParams, name = NULL, shown = TRUE,
opacity = 1, legend = FALSE)}: Adds a given EE object to the map as a layer. \cr
\itemize{
\item \strong{eeObject:} The object to add to the interactive map.\cr
\item \strong{visParams:} List of parameters for visualization.
See details.\cr
\item \strong{name:} The name of the layer.\cr
\item \strong{shown:} A flag indicating whether the
layer should be on by default. \cr
\item \strong{opacity:} The layer's opacity is represented as a number
between 0 and 1. Defaults to 1. \cr
\item \strong{legend:} Should a legend be plotted?. Ignore if \code{eeObject}
is not a single-band ee$Image.
}
\item  \strong{addLayers(eeObject, visParams, name = NULL, shown = TRUE,
opacity = 1, legend = FALSE)}: Adds a given ee$ImageCollection to the map
as multiple layers. \cr
\itemize{
\item \strong{eeObject:} The ee$ImageCollection to add to the interactive map.\cr
\item \strong{visParams:} List of parameters for visualization.
See details.\cr
\item \strong{name:} The name of layers.\cr
\item \strong{shown:} A flag indicating whether
layers should be on by default. \cr
\item \strong{opacity:} The layer's opacity is represented as a number
between 0 and 1. Defaults to 1. \cr
\item \strong{nmax:} Numeric. The maximum number of images to display.
By default 5. \cr
\item \strong{legend:} Should a legend be plotted?. Only the legend of
the first image is displayed.
}
\item \strong{setCenter(lon = 0, lat = 0, zoom = NULL)}: Centers the map
view at the given coordinates with the given zoom level. If no zoom level
is provided, it uses 1 by default.
\itemize{
\item \strong{lon:} The longitude of the center, in degrees.\cr
\item \strong{lat:} The latitude of the center, in degrees.\cr
\item \strong{zoom:} The zoom level, from 1 to 24.
}
\item \strong{setZoom(zoom = NULL)}: Sets the zoom level of the map.
\itemize{
\item \strong{zoom:} The zoom level, from 1 to 24.
}
\item \strong{centerObject(eeObject, zoom = NULL,
maxError = ee$ErrorMargin(1))}: Centers the
map view on a given object. If no zoom level is provided, it will
be predicted according to the bounds of the Earth Engine object specified.
\itemize{
\item \strong{eeObject:} EE object.\cr
\item \strong{zoom:} The zoom level, from 1 to 24.
\item \strong{maxError:} 	Max error when input
image must be reprojected to an explicitly
requested result projection or geodesic state.
}
}
}
\usage{
Map
}
\value{
Object of class leaflet, with the following extra parameters: tokens, name,
opacity, shown, min, max, palette, and legend. Use the $ method to retrieve
the data (e.g. m$rgee$min).
}
\description{
Create interactive visualizations of spatial EE objects
(ee$FeatureCollection, ee$ImageCollection, ee$Geometry, ee$Feature, and
ee$Image.) using \code{leaflet} in the backend.
}
\details{
\code{Map} use the Earth Engine method
\href{https://developers.google.com/earth-engine/api_docs#ee.data.getmapid/}{
getMapId} to fetch and return an ID dictionary being used to create
layers in a \code{leaflet} object. Users can specify visualization
parameters to Map$addLayer by using the visParams argument. Each Earth
Engine spatial object has a specific format. For
\code{ee$Image}, the
\href{https://developers.google.com/earth-engine/guides/image_visualization}{
parameters} available are:

\tabular{lll}{
\strong{Parameter}\tab \strong{Description}  \tab \strong{Type}\cr
\strong{bands}    \tab  Comma-delimited list of three band (RGB) \tab  list \cr
\strong{min}      \tab  Value(s) to map to 0 \tab  number or list of three
numbers, one for each band \cr
\strong{max}      \tab  Value(s) to map to 1 \tab  number or list of three
numbers, one for each band \cr
\strong{gain}     \tab  Value(s) by which to multiply each pixel value \tab
number or list of three numbers, one for each band \cr
\strong{bias}     \tab  Value(s) to add to each Digital Number
value \tab number or list of three numbers, one for each band \cr
\strong{gamma}    \tab  Gamma correction factor(s) \tab  number or list of
three numbers, one for each band \cr
\strong{palette}  \tab  List of CSS-style color strings
(single-band only) \tab  comma-separated list of hex strings \cr
\strong{opacity}   \tab  The opacity of the layer (from 0 to 1)  \tab  number \cr
}

If you add an \code{ee$Image} to Map$addLayer without any additional
parameters, by default it assigns the first three bands to red,
green, and blue bands, respectively. The default stretch is based on the
min-max range. On the other hand, the available parameters for
\code{ee$Geometry}, \code{ee$Feature}, and \code{ee$FeatureCollection}
are:

\itemize{
\item \strong{color}: A hex string in the format RRGGBB specifying the
color to use for drawing the features. By default #000000.
\item \strong{pointRadius}: The radius of the point markers. By default 3.
\item \strong{strokeWidth}: The width of lines and polygon borders. By
default 3.
}
}
\examples{
\dontrun{
library(rgee)
library(sf)
ee_Initialize()

# Case 1: Geometry*
geom1 <- ee$Geometry$Point(list(-73.53, -15.75))
Map$centerObject(geom1, zoom = 8)
m1 <- Map$addLayer(
  eeObject = geom1,
  visParams = list(
    pointRadius = 10,
    color = "FF0000"
  ),
  name = "Geometry-Arequipa"
)

# Case 2: Feature
feature_arq <- ee$Feature(ee$Geometry$Point(list(-72.53, -15.75)))
m2 <- Map$addLayer(
  eeObject = feature_arq,
  name = "Feature-Arequipa"
)
m2 + m1

# Case 4: Image
image <- ee$Image("LANDSAT/LC08/C01/T1/LC08_044034_20140318")
Map$centerObject(image)
m4 <- Map$addLayer(
  eeObject = image,
  visParams = list(
    bands = c("B4", "B3", "B2"),
    max = 10000
  ),
  name = "SF"
)

# Case 5: ImageCollection
nc <- st_read(system.file("shape/nc.shp", package = "sf")) \%>\%
  st_transform(4326) \%>\%
  sf_as_ee()

ee_s2 <- ee$ImageCollection("COPERNICUS/S2")$
  filterDate("2016-01-01", "2016-01-31")$
  filterBounds(nc) \%>\%
  ee_get(0:4)
Map$centerObject(nc$geometry())
m5 <- Map$addLayers(ee_s2, legend = TRUE)
m5

# Case 6: Map comparison
image <- ee$Image("LANDSAT/LC08/C01/T1/LC08_044034_20140318")
Map$centerObject(image)
m_ndvi <- Map$addLayer(
  eeObject = image$normalizedDifference(list("B5", "B4")),
  visParams = list(max = 0.7),
  name = "SF_NDVI",
  legend = TRUE
)
m6 <- m4 | m_ndvi
m6

# Case 7: digging up the metadata
m6$rgee$tokens
m5$rgee$tokens
}
}
\keyword{datasets}
