#' Define Function To Sample From Coda Output
#' 
#' A utility function which takes a matrix-like object that can be coerced to 
#' class \code{\link[coda]{mcmc}} and creates a function which samples from the 
#' posterior distribution for the parameters of the model.
#' 
#' @param modelfit An object that can be coerced to an \code{\link[coda]{mcmc}} 
#'   object. For instance, \code{modelfit} can be a matrix of samples with one 
#'   column per variable, or an object of type \code{rjags} generated by fitting
#'   a model in the program \code{JAGS} (Plummer, 2003).
#' @param sampler.name A string giving the desired name for the function to be 
#'   defined.
#' @param order A vector specifying which parameters to extract from 
#'   \code{modelfit}, and in which order. Can either be a vector of strings 
#'   naming the parameters to extract, or a numeric vector of the indices to 
#'   extract.
#' @param envir The environment in which to define the sampling function.
#'   Defaults to the global environment.
#' @return A function is defined in \code{envir} which randomly samples from the
#'   posterior distribution for the parameters. Note that this function does not
#'   take any arguments. A coda-sampling function generated in this way is 
#'   suitable for passing to the \code{rjmcmcpost} function.
#'   
#' @references Plummer, M. (2003) JAGS: A program for analysis of Bayesian 
#'   graphical models using Gibbs sampling. \emph{Proceedings of the 3rd 
#'   international workshop on distributed statistical computing (Vol. 124, p. 
#'   125)}.
#'   
#' @seealso \code{\link{rjmcmcpost}}
#' @examples
#' troutmodel = function(){
#'  for(i in 1:n){
#'    y[i] ~ dbern(p)
#'  }
#'  p <- 1 / (1 + exp(-b0))
#'  b0 ~ dnorm(0, 2*V)
#'  u1 ~ dnorm(0, 2*V)
#'  u2 ~ dnorm(0, 2*V)
#'  u3 ~ dnorm(0, 2*V)
#'  V ~ dgamma(3.29, 7.80)
#' }
#' 
#' data(brunner, package="rjmcmc")
#' y=brunner[,1]; s=brunner[,2]; l=brunner[,3]; n=length(brunner[,1])
#' 
#' inits=function(){list("b0"=rnorm(0,0.01),"u1"=rnorm(0,0.01),"u2"=rnorm(0,0.01),
#'                       "u3"=rnorm(0,0.01),"V"=rgamma(0.1,0.1))}
#' params=c("b0","u1","u2","u3","V")
#' jagsfit1=R2jags::jags(data=c("y", "n"), inits=inits,params,n.iter=1000,
#'                       model.file=troutmodel)
#' 
#' getsampler(modelfit=jagsfit1, sampler.name="posterior1")
#' set.seed(100)
#' posterior1()
#' 
#' ## Alternatively
#' posterior1b = getsampler(modelfit=jagsfit1)  # this successfully defines a function named
#' # \code{posterior1b} but also defines an identical function corresponding to the value 
#' # of \code{sampler.name}, i.e. \code{"post.draw"} in this case.
#' set.seed(100)
#' posterior1b()
#' set.seed(100)
#' post.draw()
#' 
#' @importFrom R2jags jags
#' @export
getsampler = function(modelfit, sampler.name="post.draw", order="default", envir=.GlobalEnv){
  C = as.matrix(coda::as.mcmc(modelfit))
  assign(sampler.name, function(){
    temp = C[sample(dim(C)[1],1,replace=T),-which(colnames(C)=="deviance")]
    if(any(order=="default")){return(temp)}
    else return(temp[order])}, envir=envir)
}
