% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/robustbetareg.R
\name{waldtypetest}
\alias{waldtypetest}
\title{Robust Wald-type Tests}
\usage{
waldtypetest(object, FUN, ...)
}
\arguments{
\item{object}{fitted model object of class \code{robustbetareg} (see \code{\link[robustbetareg:robustbetareg]{robustbetareg}}).}

\item{FUN}{function representing the null hypothesis to be tested.}

\item{...}{further arguments to be passed to the \code{FUN} function.}
}
\value{
\code{waldtypetest} returns an output for the Wald-type test containing
the value of the test statistic, degrees-of-freedom and p-value.
}
\description{
\code{waldtypetest} provides Wald-type tests for both simple and composite
hypotheses for beta regression based on the robust estimators
(LSMLE, LMDPDE, SMLE, and MDPDE).
}
\details{
The function provides a robust Wald-type test for a general hypothesis
    \eqn{m(\theta) = \eta_0}, for a fixed \eqn{\eta_0 \in R^d}, against
    a two sided alternative; see Maluf et al. (2022) for details.
    The argument \code{FUN} specifies the function \eqn{m(\theta) - \eta_0},
    which defines the null hypothesis. For instance, consider a model with
    \eqn{\theta = (\beta_1, \beta_2, \beta_3, \gamma_1)^\top} and let the
     null hypothesis be \eqn{\beta_2 + \beta_3 = 4}. The \code{FUN} argument can be
    \code{FUN = function(theta) \{theta[2] + theta[3] - 4\}}. It is also possible to
    define the function as \code{FUN = function(theta, B) \{theta[2] + theta[3] - B\}},
    and pass the \code{B} argument through the \code{waldtypetest} function.
    If no function is specified, that is, \code{FUN=NULL}, the \code{waldtypetest}
    returns a test in which the null hypothesis is that all the coefficients are
     zero.
}
\examples{
\donttest{
# generating a dataset
set.seed(2022)
n <- 40
beta.coef <- c(-1, -2)
gamma.coef <- c(5)
X <- cbind(rep(1, n), x <- runif(n))
mu <- exp(X\%*\%beta.coef)/(1 + exp(X\%*\%beta.coef))
phi <- exp(gamma.coef) #Inverse Log Link Function
y <- rbeta(n, mu*phi, (1 - mu)*phi)
y[26] <- rbeta(1, ((1 + mu[26])/2)*phi, (1 - ((1 + mu[26])/2))*phi)
SimData <- as.data.frame(cbind(y, x))
colnames(SimData) <- c("y", "x")

# Fitting the MLE and the LSMLE
fit.mle <- robustbetareg(y ~ x | 1, data = SimData, alpha = 0)
fit.lsmle <- robustbetareg(y ~ x | 1, data = SimData)

# Hypothesis to be tested: (beta_1, beta_2) = c(-1, -2) against a two
# sided alternative
h0 <- function(theta){theta[1:2] - c(-1, -2)}
waldtypetest(fit.mle, h0)
waldtypetest(fit.lsmle, h0)
# Alternative way:
h0 <- function(theta, B){theta[1:2] - B}
waldtypetest(fit.mle, h0, B = c(-1, -2))
waldtypetest(fit.lsmle, h0, B = c(-1, -2))
}
}
\references{
Maluf, Y. S., Ferrari, S. L. P., and Queiroz, F. F. (2022). Robust
    beta regression through the logit transformation. \emph{arXiv}:2209.11315.\cr \cr
    Basu, A., Ghosh, A., Martin, N., and Pardo, L. (2018). Robust Wald-type
    tests for  non-homogeneous observations based on the minimum density
    power divergence estimator. \emph{Metrika}, 81:493–522. \cr \cr
    Ribeiro, K. A. T. and Ferrari, S. L. P. (2022). Robust estimation in beta
    regression via maximum Lq-likelihood. \emph{Statistical Papers}.
}
\seealso{
\code{\link{robustbetareg}}
}
\author{
Yuri S. Maluf (\email{yurimaluf@gmail.com}),
Francisco F. Queiroz (\email{ffelipeq@outlook.com}) and Silvia L. P. Ferrari.
}
