% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rsvd.R
\name{rsvd}
\alias{rsvd}
\title{Randomized Singular Value Decomposition (rsvd).}
\usage{
rsvd(A, k = NULL, nu = NULL, nv = NULL, p = 10, q = 2,
  sdist = "normal")
}
\arguments{
\item{A}{array_like; \cr
a real/complex \eqn{(m, n)} input matrix (or data frame) to be decomposed.}

\item{k}{integer; \cr
specifies the target rank of the low-rank decomposition. \eqn{k} should satisfy \eqn{k << min(m,n)}.}

\item{nu}{integer, optional; \cr
number of left singular vectors to be returned. \eqn{nu} must be between \eqn{0} and \eqn{k}.}

\item{nv}{integer, optional; \cr
number of right singular vectors to be returned. \eqn{nv} must be between \eqn{0} and \eqn{k}.}

\item{p}{integer, optional; \cr
oversampling parameter (by default \eqn{p=10}).}

\item{q}{integer, optional; \cr
number of additional power iterations (by default \eqn{q=2}).}

\item{sdist}{string \eqn{c( 'unif', 'normal', 'rademacher')}, optional; \cr
specifies the sampling distribution of the random test matrix: \cr
        \eqn{'unif'} :  Uniform `[-1,1]`. \cr
        \eqn{'normal}' (default) : Normal `~N(0,1)`. \cr
        \eqn{'rademacher'} : Rademacher random variates. \cr}
}
\value{
\code{rsvd} returns a list containing the following three components:
\describe{
\item{d}{  array_like; \cr
          singular values; vector of length \eqn{(k)}.
}

\item{u}{  array_like; \cr
          left singular vectors; \eqn{(m, k)} or \eqn{(m, nu)} dimensional array.
}

\item{v}{  array_like; \cr
          right singular vectors; \eqn{(n, k)} or \eqn{(n, nv)} dimensional array. \cr
}
}
}
\description{
The randomized SVD computes the near-optimal low-rank approximation of a rectangular matrix
using a fast probablistic algorithm.
}
\details{
The singular value decomposition (SVD) plays an important role in data analysis, and scientific computing.
Given a rectangular \eqn{(m,n)} matrix \eqn{A}, and a target rank \eqn{k << min(m,n)}, 
the SVD factors the input matrix \eqn{A} as

\deqn{ A  =  U_{k} diag(d_{k}) V_{k}^\top }{ A = U diag(d) t(V)}

The \eqn{k} left singular vectors are the columns of the
real or complex unitary matrix \eqn{U}. The \eqn{k} right singular vectors are the columns
of the real or complex unitary matrix \eqn{V}. The \eqn{k} dominant singular values are the 
entries of \eqn{d}, and non-negative and real numbers.

\eqn{p} is an oversampling parameter to improve the approximation.
A value of at least 10 is recommended, and \eqn{p=10} is set by default.

The parameter \eqn{q} specifies the number of power (subspace) iterations
to reduce the approximation error. The power scheme is recommended,
if the singular values decay slowly. In practice, 2 or 3 iterations
achieve good results, however, computing power iterations increases the
computational costs. The power scheme is set to \eqn{q=2} by default.

If \eqn{k > (min(n,m)/4)}, a deterministic partial or truncated \code{\link{svd}}
algorithm might be faster.
}
\note{
The singular vectors are not unique and only defined up to sign
(a constant of modulus one in the complex case). If a left singular vector
has its sign changed, changing the sign of the corresponding right vector
gives an equivalent decomposition.
}
\examples{
library('rsvd')

# Create a n x n Hilbert matrix of order n,
# with entries H[i,j] = 1 / (i + j + 1).
hilbert <- function(n) { i <- 1:n; 1 / outer(i - 1, i, "+") }
H <- hilbert(n=50)

# Low-rank (k=10) matrix approximation using rsvd
k=10
s <- rsvd(H, k=k)
Hre <- s$u \%*\% diag(s$d) \%*\% t(s$v) # matrix approximation
print(100 * norm( H - Hre, 'F') / norm( H,'F')) # percentage error
# Compare to truncated base svd
s <- svd(H)
Hre <- s$u[,1:k] \%*\% diag(s$d[1:k]) \%*\% t(s$v[,1:k]) # matrix approximation
print(100 * norm( H - Hre, 'F') / norm( H,'F')) # percentage error

}
\references{
\itemize{
 \item [1] N. B. Erichson, S. Voronin, S. L. Brunton and J. N. Kutz. 2019.
       Randomized Matrix Decompositions Using {R}. 
       Journal of Statistical Software, 89(11), 1-48.
       \url{http://doi.org/10.18637/jss.v089.i11}.

 \item  [2] N. Halko, P. Martinsson, and J. Tropp.
         "Finding structure with randomness: probabilistic
         algorithms for constructing approximate matrix
         decompositions" (2009).
         (available at arXiv \url{http://arxiv.org/abs/0909.4061}).
}
}
\seealso{
\code{\link{svd}}, \code{\link{rpca}}
}
\author{
N. Benjamin Erichson, \email{erichson@berkeley.edu}
}
