% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/box_cox_functions_rcpp.R
\name{find_lambda_one_d_rcpp}
\alias{find_lambda_one_d_rcpp}
\title{Selecting the Box-Cox parameter in the 1D case using Rcpp}
\usage{
find_lambda_one_d_rcpp(
  logf,
  ...,
  ep_bc = 1e-04,
  min_phi = ep_bc,
  max_phi = 10,
  num = 1001L,
  xdiv = 100,
  probs = seq(0.01, 0.99, by = 0.01),
  lambda_range = c(-3, 3),
  phi_to_theta = NULL,
  log_j = NULL,
  user_args = list()
)
}
\arguments{
\item{logf}{A pointer to a compiled C++ function returning the log
of the target density \eqn{f}.}

\item{...}{further arguments to be passed to \code{logf} and related
functions.}

\item{ep_bc}{A (positive) numeric scalar. Smallest possible value of
\code{phi} to consider.  Used to avoid negative values of \code{phi}.}

\item{min_phi, max_phi}{Numeric scalars.  Smallest and largest values
of \code{phi} at which to evaluate \code{logf}, i.e., the range of values
of \code{phi} over which to evaluate \code{logf}. Any components in
\code{min_phi} that are not positive are set to \code{ep_bc}.}

\item{num}{A numeric scalar. Number of values at which to evaluate
\code{logf}.}

\item{xdiv}{A numeric scalar.  Only values of \code{phi} at which the
density \eqn{f} is greater than the (maximum of \eqn{f}) / \code{xdiv} are
used.}

\item{probs}{A numeric scalar. Probabilities at which to estimate the
quantiles of that will be used as data to find lambda.}

\item{lambda_range}{A numeric vector of length 2.  Range of lambda over
which to optimise.}

\item{phi_to_theta}{A pointer to a compiled C++ function returning
(the inverse) of the transformation from \code{theta} to \code{phi} used
to ensure positivity of \code{phi} prior to Box-Cox transformation.  The
argument is \code{phi} and the returned value is \code{theta}.  If
\code{phi_to_theta} is undefined at the input value then the function
should return \code{NA}.}

\item{log_j}{A pointer to a compiled C++ function returning the log of
the Jacobian of the transformation from \code{theta} to \code{phi}, i.e.,
based on derivatives of \eqn{phi} with respect to \eqn{theta}. Takes
\code{theta} as its argument.
If this is not supplied then a constant Jacobian is used.}

\item{user_args}{A list of numeric components providing arguments to
the user-supplied functions \code{phi_to_theta} and \code{log_j}.}
}
\value{
A list containing the following components
  \item{lambda}{A numeric scalar.  The value of \code{lambda}.}
  \item{gm}{A numeric scalar.  Box-cox scaling parameter, estimated by the
    geometric mean of the quantiles used in the optimisation to find the
    value of lambda.}
  \item{init_psi}{A numeric scalar.  An initial estimate of the mode of the
    Box-Cox transformed density}
  \item{sd_psi}{A numeric scalar.  Estimates of the marginal standard
    deviations of the Box-Cox transformed variables.}
 \item{phi_to_theta}{as detailed above (only if \code{phi_to_theta} is
   supplied)}
 \item{log_j}{as detailed above (only if \code{log_j} is supplied)}
 \item{user_args}{as detailed above (only if \code{user_args} is supplied)}
}
\description{
Finds a value of the Box-Cox transformation parameter lambda for which
the (positive univariate) random variable with log-density
\eqn{\log f}{log f} has a density closer to that of a Gaussian random
variable. Works by estimating a set of quantiles of the distribution implied
by \eqn{\log f}{log f} and treating those quantiles as data in a standard
Box-Cox analysis.  In the following we use \code{theta} (\eqn{\theta}) to
denote the argument of \eqn{\log f}{log f} on the original scale and
\code{phi} (\eqn{\phi}) on the Box-Cox transformed scale.
}
\details{
The general idea is to estimate quantiles of \eqn{f} corresponding
  to a set of equally-spaced probabilities in \code{probs} and to use these
  estimated quantiles as data in a standard estimation of the Box-Cox
  transformation parameter \code{lambda}.

  The density \eqn{f} is first evaluated at \code{num} points equally spaced
  over the interval (\code{min_phi}, \code{max_phi}).  The continuous
  density \eqn{f} is approximated by attaching trapezium-rule estimates of
  probabilities to the midpoints of the intervals between the points.  After
  standardizing to account for the fact that \eqn{f} may not be normalized,
  (\code{min_phi}, \code{max_phi}) is reset so that values with small
  estimated probability (determined by \code{xdiv}) are excluded and the
  procedure is repeated on this new range.  Then the required quantiles are
  estimated by inferring them from a weighted empirical distribution
  function based on treating the midpoints as data and the estimated
  probabilities at the midpoints as weights.
}
\examples{

# Log-normal density ===================

# Note: the default value of max_phi = 10 is OK here but this will not
# always be the case.

ptr_lnorm <- create_xptr("logdlnorm")
mu <- 0
sigma <- 1
lambda <- find_lambda_one_d_rcpp(logf = ptr_lnorm, mu = mu, sigma = sigma)
lambda
x <- ru_rcpp(logf = ptr_lnorm, mu = mu, sigma = sigma, log = TRUE, d = 1,
             n = 1000, trans = "BC", lambda = lambda)

# Gamma density ===================

alpha <- 1
# Choose a sensible value of max_phi
max_phi <- qgamma(0.999, shape = alpha)
# [I appreciate that typically the quantile function won't be available.
# In practice the value of lambda chosen is quite insensitive to the choice
# of max_phi, provided that max_phi is not far too large or far too small.]

ptr_gam <- create_xptr("logdgamma")
lambda <- find_lambda_one_d_rcpp(logf = ptr_gam, alpha = alpha,
                                 max_phi = max_phi)
lambda
x <- ru_rcpp(logf = ptr_gam, alpha = alpha, d = 1, n = 1000, trans = "BC",
             lambda = lambda)

alpha <- 0.1
# NB. for alpha < 1 the gamma(alpha, beta) density is not bounded
# So the ratio-of-uniforms emthod can't be used but it may work after a
# Box-Cox transformation.
# find_lambda_one_d() works much better than find_lambda() here.

max_phi <- qgamma(0.999, shape = alpha)
lambda <- find_lambda_one_d_rcpp(logf = ptr_gam, alpha = alpha,
                                 max_phi = max_phi)
lambda
x <- ru_rcpp(logf = ptr_gam, alpha = alpha, d = 1, n = 1000, trans = "BC",
             lambda = lambda)
\donttest{
plot(x)
plot(x, ru_scale = TRUE)
}
}
\references{
Box, G. and Cox, D. R. (1964) An Analysis of Transformations.
 Journal of the Royal Statistical Society. Series B (Methodological), 26(2),
 211-252.

Andrews, D. F. and Gnanadesikan, R. and Warner, J. L. (1971)
 Transformations of Multivariate Data, Biometrics, 27(4).

Eddelbuettel, D. and Francois, R. (2011). Rcpp: Seamless
 R and C++ Integration. \emph{Journal of Statistical Software},
 \strong{40}(8), 1-18. \doi{10.18637/jss.v040.i08}

Eddelbuettel, D. (2013). \emph{Seamless R and C++ Integration
 with Rcpp}, Springer, New York. ISBN 978-1-4614-6867-7.
}
\seealso{
\code{\link{ru_rcpp}} to perform ratio-of-uniforms sampling.

\code{\link{find_lambda_rcpp}} to produce (somewhat) automatically
  a list for the argument \code{lambda} of \code{ru} for any value of
  \code{d}.
}
