% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vinecop.R
\name{vinecop}
\alias{vinecop}
\title{Fitting vine copula models}
\usage{
vinecop(
  data,
  var_types = rep("c", ncol(data)),
  family_set = "all",
  structure = NA,
  par_method = "mle",
  nonpar_method = "constant",
  mult = 1,
  selcrit = "bic",
  weights = numeric(),
  psi0 = 0.9,
  presel = TRUE,
  trunc_lvl = Inf,
  tree_crit = "tau",
  threshold = 0,
  keep_data = FALSE,
  show_trace = FALSE,
  cores = 1
)
}
\arguments{
\item{data}{a matrix or data.frame with at least two columns, containing the
(pseudo-)observations for the two variables (copula data should have
approximately uniform margins). More columns are required for discrete
models, see \emph{Details}.}

\item{var_types}{variable types, a length d vector; e.g., \code{c("c", "c")} for
two continuous variables, or \code{c("c", "d")} for first variable continuous
and second discrete.}

\item{family_set}{a character vector of families; see \code{\link[=bicop]{bicop()}} for
additional options.}

\item{structure}{an \code{rvine_structure} object, namely a compressed
representation of the vine structure, or an object that can be coerced into
one (see \code{\link[=rvine_structure]{rvine_structure()}} and \code{\link[=as_rvine_structure]{as_rvine_structure()}}). The dimension
must be \code{length(pair_copulas[[1]]) + 1}; \code{structure = NA} performs
automatic selection based on Dissman's algorithm. See \emph{Details} for partial
selection of the structure.}

\item{par_method}{the estimation method for parametric models, either \code{"mle"}
for maximum likelihood or \code{"itau"} for inversion of Kendall's tau (only
available for one-parameter families and \code{"t"}.}

\item{nonpar_method}{the estimation method for nonparametric models, either
\code{"constant"} for the standard transformation estimator, or
\code{"linear"}/\code{"quadratic"} for the local-likelihood approximations of order
one/two.}

\item{mult}{multiplier for the smoothing parameters of nonparametric
families. Values larger than 1 make the estimate more smooth, values less
than 1 less smooth.}

\item{selcrit}{criterion for family selection, either \code{"loglik"}, \code{"aic"},
\code{"bic"}, \code{"mbic"}. For \code{vinecop()} there is the additional option
\code{"mbicv"}.}

\item{weights}{optional vector of weights for each observation.}

\item{psi0}{prior probability of a non-independence copula (only used for
\code{selcrit = "mbic"} and \code{selcrit = "mbicv"}).}

\item{presel}{whether the family set should be thinned out according to
symmetry characteristics of the data.}

\item{trunc_lvl}{the truncation level of the vine copula; \code{Inf} means no
truncation, \code{NA} indicates that the truncation level should be selected
automatically by \code{\link[=mBICV]{mBICV()}}.}

\item{tree_crit}{the criterion for tree selection, one of \code{"tau"}, \code{"rho"},
\code{"hoeffd"}, \code{"mcor"}, or \code{"joe"} for Kendall's \eqn{\tau}, Spearman's
\eqn{\rho}, Hoeffding's \eqn{D}, maximum correlation, or logarithm of
the partial correlation, respectively.}

\item{threshold}{for thresholded vine copulas; \code{NA} indicates that the
threshold should be selected automatically by \code{\link[=mBICV]{mBICV()}}.}

\item{keep_data}{whether the data should be stored (necessary for using
\code{\link[=fitted]{fitted()}}).}

\item{show_trace}{logical; whether a trace of the fitting progress should be
printed.}

\item{cores}{number of cores to use; if more than 1, estimation of pair
copulas within a tree is done in parallel.}
}
\value{
Objects inheriting from \code{vinecop} and \code{vinecop_dist} for \code{\link[=vinecop]{vinecop()}}. In
addition to the entries provided by \code{\link[=vinecop_dist]{vinecop_dist()}}, there are:
\itemize{
\item \code{threshold}, the (set or estimated) threshold used for thresholding the
vine.
\item \code{data} (optionally, if \code{keep_data = TRUE} was used), the dataset that was
passed to \code{\link[=vinecop]{vinecop()}}.
\item \code{controls}, a \code{list} with fit controls that was passed to \code{\link[=vinecop]{vinecop()}}.
\item \code{nobs}, the number of observations that were used to fit the model.
}
}
\description{
Automated fitting and model selection for vine copula models with continuous
or discrete data.
}
\details{
\subsection{Discrete variables}{

Let \code{n} be the number of observations and \code{d} the number of variables.
When at least one variable is discrete, two types of
"observations" are required in \code{data}: the first \verb{n  x  d} block
contains realizations of \eqn{F_{X_j}(X_j)}. The second \verb{n  x  d}
block contains realizations of \eqn{F_{X_j}(X_j^-)}. The minus indicates a
left-sided limit of the cdf. For, e.g., an integer-valued variable, it holds
\eqn{F_{X_j}(X_j^-) = F_{X_j}(X_j - 1)}. For continuous variables the left
limit and the cdf itself coincide. Respective columns can be omitted in the
second block.
}

\subsection{Partial structure selection}{

It is possible to fix the vine structure only in the first trees and select
the remaining ones automatically. To specify only the first \code{k} trees, supply
a \code{k}-truncated \code{rvine_structure()} or \code{rvine_matrix()}. All trees up to
\code{trunc_lvl} will then be selected automatically.
}
}
\examples{
## simulate dummy data
x <- rnorm(30) * matrix(1, 30, 5) + 0.5 * matrix(rnorm(30 * 5), 30, 5)
u <- pseudo_obs(x)

## fit and select the model structure, family and parameters
fit <- vinecop(u)
summary(fit)
plot(fit)
contour(fit)

## select by log-likelihood criterion from one-paramter families
fit <- vinecop(u, family_set = "onepar", selcrit = "bic")
summary(fit)

## Gaussian D-vine
fit <- vinecop(u, structure = dvine_structure(1:5), family = "gauss")
plot(fit)
contour(fit)

## Partial structure selection with only first tree specified
structure <- rvine_structure(order = 1:5, list(rep(5, 4)))
structure
fit <- vinecop(u, structure = structure, family = "gauss")
plot(fit)

## 1-truncated model with random structure
fit <- vinecop(u, structure = rvine_structure_sim(5), trunc_lvl = 1)
contour(fit)

## Model for discrete data
x <- qpois(u, 1)  # transform to Poisson margins
# we require two types of observations (see Details)
u_disc <- cbind(ppois(x, 1), ppois(x - 1, 1))
fit <- vinecop(u_disc, var_types = rep("d", 5))

## Model for mixed data
x <- qpois(u[, 1], 1)  # transform first variable to Poisson margin
# we require two types of observations (see Details)
u_disc <- cbind(ppois(x, 1), u[, 2:5], ppois(x - 1, 1))
fit <- vinecop(u_disc, var_types = c("d", rep("c", 4)))
}
\seealso{
\code{\link[=vinecop]{vinecop()}}, \code{\link[=dvinecop]{dvinecop()}}, \code{\link[=pvinecop]{pvinecop()}}, \code{\link[=rvinecop]{rvinecop()}},
\code{\link[=plot.vinecop]{plot.vinecop()}}, \code{\link[=contour.vinecop]{contour.vinecop()}}
}
