% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/projection.R
\name{projection}
\alias{projection}
\title{Projection Estimator}
\usage{
projection(
  formula,
  id,
  weight,
  strata = NULL,
  domain,
  fun = "mean",
  model,
  data_model,
  data_proj,
  model_metric,
  kfold = 3,
  grid = 10,
  parallel_over = "resamples",
  seed = 1,
  est_y = FALSE,
  ...
)
}
\arguments{
\item{formula}{An object of class formula that contains a description of the model to be fitted. The variables included in the formula must be contained in the \code{data_model} dan \code{data_proj}.}

\item{id}{Column name specifying cluster ids from the largest level to the smallest level, where ~0 or ~1 represents a formula indicating the absence of clusters.}

\item{weight}{Column name in data_proj representing the survey weight.}

\item{strata}{Column name specifying strata, use NULL for no strata}

\item{domain}{Column names in data_model and data_proj representing specific domains for which disaggregated data needs to be produced.}

\item{fun}{A function taking a formula and survey design object as its first two arguments (default = "mean", "total", "varians").}

\item{model}{The working model to be used in the projection estimator. Refer to the details for the available working models.}

\item{data_model}{A data frame or a data frame extension (e.g., a tibble) representing the second survey, characterized by a much smaller sample, provides information on both the variable of interest and the auxiliary variables.}

\item{data_proj}{A data frame or a data frame extension (e.g., a tibble) representing the first survey, characterized by a large sample that collects only auxiliary information or general-purpose variables.}

\item{model_metric}{A yardstick::metric_set(), or NULL to compute a standard set of metrics (rmse for regression and f1-score for classification).}

\item{kfold}{The number of partitions of the data set (k-fold cross validation).}

\item{grid}{A data frame of tuning combinations or a positive integer. The data frame should have columns for each parameter being tuned and rows for tuning parameter candidates. An integer denotes the number of candidate parameter sets to be created automatically.}

\item{parallel_over}{A single string containing either "resamples" or "everything" describing how to use parallel processing. Alternatively, NULL is allowed, which chooses between "resamples" and "everything" automatically.
If "resamples", then tuning will be performed in parallel over resamples alone. Within each resample, the preprocessor (i.e. recipe or formula) is processed once, and is then reused across all models that need to be fit.
If "everything", then tuning will be performed in parallel at two levels. An outer parallel loop will iterate over resamples. Additionally, an inner parallel loop will iterate over all unique combinations of preprocessor and model tuning parameters for that specific resample. This will result in the preprocessor being re-processed multiple times, but can be faster if that processing is extremely fast.}

\item{seed}{A single value, interpreted as an integer}

\item{est_y}{A logical value indicating whether to return the estimation of \code{y} in \code{data_model}. If \code{TRUE}, the estimation is returned; otherwise, it is not.}

\item{...}{Further argument to the \code{\link[survey]{svydesign}}.}
}
\value{
The function returns a list with the following objects (\code{model}, \code{prediction} and \code{df_result}):
\code{model} The working model used in the projection.
\code{prediction} A vector containing the prediction results from the working model.
\code{df_result} A data frame with the following columns:
\itemize{
\item \code{domain} The name of the domain.
\item \code{ypr} The estimation results of the projection for each domain.
\item \code{var_ypr} The sample variance of the projection estimator for each domain.
\item \code{rse_ypr} The Relative Standard Error (RSE) in percentage (\%).
}
}
\description{
The function addresses the problem of combining information from two or more independent surveys, a common challenge in survey sampling. It focuses on cases where: \cr
\itemize{
\item \strong{Survey 1:} A large sample collects only auxiliary information.
\item \strong{Survey 2:} A much smaller sample collects both the variables of interest and the auxiliary variables.
}
The function implements a model-assisted projection estimation method based on a working model. The working models that can be used include several machine learning models that can be seen in the details section
}
\details{
The available working models include:
\itemize{
\item Linear Regression \code{linear_reg()}
\item Logistic Regression \code{logistic_reg()}
\item Poisson Regression \code{poisson_reg()}
\item Decision Tree \code{decision_tree()}
\item KNN \code{nearest_neighbor()}
\item Naive Bayes \code{naive_bayes()}
\item Multi Layer Perceptron \code{mlp()}
\item Random Forest \code{rand_forest()}
\item Accelerated Oblique Random Forests (Jaeger et al. 2022, Jaeger et al. 2024) \code{rand_forest(engine = 'aorsf')}
\item XGBoost \code{boost_tree(engine = 'xgboost')}
\item LightGBM \code{boost_tree(engine = 'lightgbm')}
}
A complete list of models can be seen at the following link \href{https://www.tmwr.org/pre-proc-table}{Tidy Modeling With R}
}
\examples{
\dontrun{
library(sae.projection)
library(dplyr)
library(bonsai)

df_svy22_income <- df_svy22 \%>\% filter(!is.na(income))
df_svy23_income <- df_svy23 \%>\% filter(!is.na(income))

# Linear regression
lm_proj <- projection(
  income ~ age + sex + edu + disability,
  id = "PSU", weight = "WEIGHT", strata = "STRATA",
  domain = c("PROV", "REGENCY"),
  model = linear_reg(),
  data_model = df_svy22_income,
  data_proj = df_svy23_income,
  nest = TRUE
)

# Random forest regression with hyperparameter tunning
rf_proj <- projection(
  income ~ age + sex + edu + disability,
  id = "PSU", weight = "WEIGHT", strata = "STRATA",
  domain = c("PROV", "REGENCY"),
  model = rand_forest(mtry = tune(), trees = tune(), min_n = tune()),
  data_model = df_svy22_income,
  data_proj = df_svy23_income,
  kfold = 3,
  grid = 10,
  nest = TRUE
)

df_svy22_neet <- df_svy22 \%>\% filter(between(age, 15, 24))
df_svy23_neet <- df_svy23 \%>\% filter(between(age, 15, 24))

# Logistic regression
lr_proj <- projection(
  formula = neet ~ sex + edu + disability,
  id = ~ PSU,
  weight = ~ WEIGHT,
  strata = ~ STRATA,
  domain = ~ PROV + REGENCY,
  model = logistic_reg(),
  data_model = df_svy22_neet,
  data_proj = df_svy23_neet,
  nest = TRUE
)

# LightGBM regression with hyperparameter tunning
show_engines("boost_tree")
lgbm_model <- boost_tree(
  mtry = tune(), trees = tune(), min_n = tune(),
  tree_depth = tune(), learn_rate = tune(),
  engine = "lightgbm"
)

lgbm_proj <- projection(
  formula = neet ~ sex + edu + disability,
  id = "PSU",
  weight = "WEIGHT",
  strata = "STRATA",
  domain = c("PROV", "REGENCY"),
  model = lgbm_model,
  data_model = df_svy22_neet,
  data_proj = df_svy23_neet,
  kfold = 3,
  grid = 10,
  nest = TRUE
)
}
}
\references{
\enumerate{
\item Kim, J. K., & Rao, J. N. (2012). Combining data from two independent surveys: a model-assisted approach. Biometrika, 99(1), 85-100.
}
}
