#' @title Small Area Estimation Using Hierarchical Bayesian Method under Generalized Poisson Distribution
#'
#' @description This function is implemented to variable of interest \eqn{(y)} that assumed to be a Generalized Poisson Distribution. The range of data is \eqn{0 < y < \infty}. Generalized Distribution model can be used to handle underdispersion and overdispersion in count data.
#'
#' @param formula Formula that describe the fitted model
#' @param iter.update Number of updates with default \code{3}
#' @param coef Regression coefficient for variable of interest \eqn{(y)}
#' @param var.coef Variance of coefficient
#' @param iter.mcmc Number of total iterations per chain with default \code{2000}
#' @param thin Thinning rate, must be a positive integer with default \code{1}
#' @param burn.in Number of iterations to discard at the beginning with default \code{1000}
#' @param tau.u Variance of random effect area for non-zero count of variable interest with default \code{1}
#' @param data The data frame
#'
#' @import stringr
#' @import coda
#' @import rjags
#' @import stats
#' @import grDevices
#' @import graphics
#'
#' @return This function returns a list of the following objects:
#'    \item{Est}{A vector with the values of Small Area mean Estimates using Hierarchical Bayesian method}
#'    \item{refVar}{Estimated random effect variances}
#'    \item{coefficient}{A dataframe contains the estimated model coefficient}
#'    \item{alpha}{Parameter dispersion of Generalized Poisson distribution}
#'    \item{plot}{Trace, Density, Autocorrelation Function Plot of MCMC samples}
#'
#'@examples
#' ##For data without any non-sampled area
#' data(dataGPois)     # Load dataset
#'
#' result <- GPois(y ~ x1 + x2, data = dataGPois)
#'
#' result$Est           # Small Area mean estimates
#' result$refVar        # Estimated random effect variances
#' result$coefficient   # Estimated model coefficient
#' result$alpha         # Paramter of dispersion
#'
#' # Load library 'coda' to execute the plot
#' # autocorr.plot(result$plot[[3]])  # Generate ACF Plot
#' # plot(result$plot[[3]])           # Generate Density and Trace Plot
#'
#' ## For data with non-sampled area use dataGPoisNs
#'
#' @export

GPois <- function(formula,iter.update=3, iter.mcmc=2000,
                  coef, var.coef, thin = 1, burn.in =1000, tau.u = 1, data){

  result <- list(Est = NA, refVar = NA, coefficient = NA, alpha = NA, plot= NA)

  formuladata <- model.frame(formula,data,na.action=NULL)
  if (any(is.na(formuladata[,-1])))
    stop("Auxiliary Variables contains NA values.")
  auxVar <- as.matrix(formuladata[,-1])
  nvar <- ncol(auxVar) + 1

  if (!missing(var.coef)){

    if( length(var.coef) != nvar ){
      stop("length of vector var.coef does not match the number of regression coefficients, the length must be ",nvar)
    }

    tau.b.value = 1/var.coef
  } else {
    tau.b.value = 1/rep(1,nvar)
  }

  if (!missing(coef)){
    if( length(coef) != nvar ){
      stop("length of vector coef does not match the number of regression coefficients, the length must be ",nvar)
    }
    mu.b.value = coef
  } else {
    mu.b.value = rep(0,nvar)
  }

  if (iter.update < 3){
    stop("the number of iteration updates at least 3 times")
  }

  if (!any(is.na(formuladata[,1]))){

    formuladata <- as.matrix(na.omit(formuladata))
    x <- model.matrix(formula,data = as.data.frame(formuladata))
    n <- nrow(formuladata)
    mu.b = mu.b.value
    tau.b = tau.b.value
    tau.ua = tau.ub = a.var = 1
    alpha.a = -1
    alpha.b = 1
    for (i in 1:iter.update){
      dat <- list("Zeros" = rep(0, n), "C" = 10000, "n"= n,  "nvar"= nvar, "y" = formuladata[,1], "x"=as.matrix(x[,-1]),
                  "mu.b"=mu.b, "tau.b"=tau.b, "tau.ua"=tau.ua, "tau.ub"=tau.ub, "alpha.a" = alpha.a, "alpha.b" = alpha.b)
      inits <- list(u = rep(0, n),b = mu.b, tau.u =tau.u)
      cat("model {
					for (i in 1:n) {
					    Zeros[i] ~ dpois(Zeros.mean[i])
              Zeros.mean[i] <- -L[i] + C

              l1[i] <- log((1-alpha)*mu[i]) + (y[i]-1)*log((1-alpha)*mu[i] + alpha*y[i])
              l2[i] <- (1-alpha)*mu[i] + alpha*y[i] + loggam(y[i] + 1)
              L[i] <- l1[i] - l2[i]

              log(mu[i]) <- b[1] + sum(b[2:nvar]*x[i,]) + u[i]
              u[i] ~ dnorm(0,tau.u)
					}

          # prior
          for (k in 1:nvar){
					    b[k] ~ dnorm(mu.b[k],tau.b[k])
          }
					alpha ~ dunif(alpha.a, alpha.b)
					tau.u~dgamma(tau.ua, tau.ub)
          a.var <- 1/tau.u

		  	}", file="gpois.txt")
      jags.m <- jags.model( file = "gpois.txt", data=dat, inits=inits, n.chains=1, n.adapt=500 )
      file.remove("gpois.txt")
      params <- c("mu","a.var","b", "tau.u", "alpha")
      samps <- coda.samples( jags.m, params, n.iter=iter.mcmc, thin=thin)
      samps1 <- window(samps, start=burn.in+1, end=iter.mcmc)
      result_samps=summary(samps1)
      a.var=result_samps$statistics[1]
      beta=result_samps$statistics[3:(nvar+2),1:2]
      for (i in 1:nvar){
        mu.b[i]  = beta[i,1]
        tau.b[i] = 1/(beta[i,2]^2)
      }
      tau.ua = result_samps$statistics[3+nvar+n,1]^2/result_samps$statistics[3+nvar+n,2]^2
      tau.ub = result_samps$statistics[3+nvar+n,1]/result_samps$statistics[3+nvar+n,2]^2
    }
    result_samps=summary(samps1)
    b.varnames <- list()
    for (i in 1:(nvar)) {
      idx.b.varnames <- as.character(i-1)
      b.varnames[i] <-str_replace_all(paste("b[",idx.b.varnames,"]"),pattern=" ", replacement="")
    }

    result_mcmc <- samps1[,c(3:(nvar+2))]
    colnames(result_mcmc[[1]]) <- b.varnames

    a.var=result_samps$statistics[1]
    alpha=result_samps$statistics[2]

    beta=result_samps$statistics[3:(nvar+2),1:2]
    rownames(beta) <- b.varnames

    mu=result_samps$statistics[(nvar+3):(2+nvar+n),1:2]

    Estimation=data.frame(mu)

    Quantiles <- as.data.frame(result_samps$quantiles[1:(3+nvar+n),])
    q_mu <- Quantiles[(nvar+3):(nvar+2+n),]
    q_beta <- (Quantiles[3:(nvar+2),])
    rownames(q_beta) <- b.varnames
    beta <- cbind(beta,q_beta)
    Estimation <- data.frame(Estimation,q_mu)
    colnames(Estimation) <- c("MEAN","SD","2.5%","25%","50%","75%","97.5%")

  } else {

    formuladata <- as.data.frame(formuladata)

    x <- as.matrix(formuladata[,2:nvar])
    n <- nrow(formuladata)

    mu.b =mu.b.value
    tau.b = tau.b.value
    tau.ua = tau.ub = a.var = 1
    alpha.a = -1
    alpha.b = 1

    formuladata$idx <- rep(1:n)
    data_sampled <- na.omit(formuladata)
    data_nonsampled <- formuladata[-data_sampled$idx,]

    r=data_nonsampled$idx
    n1=nrow(data_sampled)
    n2=nrow(data_nonsampled)
    for (i in 1:iter.update){
      dat <- list("Zeros" = rep(0, n1), "C" = 10000,"n1"= n1, "n2"=n2,"nvar"=nvar, "y_sampled" = data_sampled[,1],
                  "x_sampled"=as.matrix(data_sampled[,2:nvar]),
                  "x_nonsampled"=as.matrix(data_nonsampled[,2:nvar]),
                  "mu.b"=mu.b,"tau.b"=tau.b, "tau.ua"=tau.ua,"tau.ub"=tau.ub, "alpha.a" = alpha.a, "alpha.b" = alpha.b)
      inits <- list(u = rep(0,n1), v = rep(0,n2), b = mu.b, tau.u = tau.u)
      cat("model {
					for (i in 1:n1) {
					    Zeros[i] ~ dpois(Zeros.mean[i])
              Zeros.mean[i] <- -L[i] + C

              l1[i] <- log((1-alpha)*mu[i]) + (y_sampled[i]-1)*log((1-alpha)*mu[i] + alpha*y_sampled[i])
              l2[i] <- (1-alpha)*mu[i] + alpha*y_sampled[i] + loggam(y_sampled[i] + 1)
              L[i] <- l1[i] - l2[i]

              log(mu[i]) <- b[1] + sum(b[2:nvar]*x_sampled[i,]) + u[i]
              u[i] ~ dnorm(0,tau.u)
					}
					alpha ~ dunif(alpha.a, alpha.b)
					tau.u~dgamma(tau.ua, tau.ub)
          a.var <- 1/tau.u

					for(j in 1:n2) {
              v[j]~dnorm(0, tau.u)
              log(mu.nonsampled[j]) <-  mu.b[1] + sum(mu.b[2:nvar]*x_nonsampled[j,]) +v[j]
					}

					# prior
          for (k in 1:nvar){
					    b[k] ~ dnorm(mu.b[k],tau.b[k])
          }
			}", file="gpois.txt")
      jags.m <- jags.model( file = "gpois.txt", data=dat, inits=inits, n.chains=1, n.adapt=500 )
      file.remove("gpois.txt")
      params <- c("mu","mu.nonsampled","a.var","b", "tau.u", "alpha")
      samps <- coda.samples( jags.m, params, n.iter=iter.mcmc, thin=thin)
      samps1 <- window(samps, start=burn.in+1, end=iter.mcmc)
      result_samps=summary(samps1)
      a.var=result_samps$statistics[1]
      beta=result_samps$statistics[3:(nvar+2),1:2]
      for (i in 1:nvar){
        mu.b[i]  = beta[i,1]
        tau.b[i] = 1/(beta[i,2]^2)
      }
      tau.ua = result_samps$statistics[3+nvar+n,1]^2/result_samps$statistics[3+nvar+n,2]^2
      tau.ub = result_samps$statistics[3+nvar+n,1]/result_samps$statistics[3+nvar+n,2]^2
    }
    result_samps=summary(samps1)
    b.varnames <- list()
    for (i in 1:(nvar)) {
      idx.b.varnames <- as.character(i-1)
      b.varnames[i] <-str_replace_all(paste("b[",idx.b.varnames,"]"),pattern=" ", replacement="")
    }
    result_mcmc <- samps1[,c(3:(nvar+2))]
    colnames(result_mcmc[[1]]) <- b.varnames

    a.var=result_samps$statistics[1]
    alpha=result_samps$statistics[2]

    beta=result_samps$statistics[3:(nvar+2),1:2]
    rownames(beta) <- b.varnames

    mu=result_samps$statistics[(nvar+3):(2+nvar+n1),1:2]
    mu.nonsampled =result_samps$statistics[(3+nvar+n1):(2+nvar+n),1:2]

    Estimation=matrix(rep(0,n),n,2)
    Estimation[r,]=mu.nonsampled
    Estimation[-r,]=mu
    Estimation = as.data.frame(Estimation)

    Quantiles <- as.data.frame(result_samps$quantiles[1:(3+nvar+n),])
    q_beta <- (Quantiles[3:(nvar+2),])
    q_mu <- (Quantiles[(nvar+3):(nvar+2+n1),])
    q_mu.nonsampled <- (Quantiles[(3+nvar+n1):(2+nvar+n),])
    q_Estimation <- matrix(0,n,5)
    for (i in 1:5){
      q_Estimation[r,i] <- q_mu.nonsampled[,i]
      q_Estimation[-r,i] <- q_mu[,i]
    }

    rownames(q_beta) <- b.varnames
    beta <- cbind(beta,q_beta)
    Estimation <- data.frame(Estimation,q_Estimation)
    colnames(Estimation) <- c("MEAN","SD","2.5%","25%","50%","75%","97.5%")
  }

  result$Est         = Estimation
  result$refVar      = a.var
  result$alpha       = alpha
  result$coefficient = beta
  result$plot        = list(graphics.off() ,par(mar=c(2,2,2,2)),
                            autocorr.plot(result_mcmc,col="brown2",lwd=2),
                            plot(result_mcmc,col="brown2",lwd=2))
  return(result)

}
