#' semPower: Power analyses for structural equation models (SEM).
#'
#' semPower allows for performing a-priori, post-hoc, and compromise power-analyses for structural equation models (SEM).
#'
#' \itemize{
#' \item A-priori power analysis \code{\link{semPower.aPriori}} computes the required N, given an effect, alpha, power, and the model df
#' \item Post-hoc power analysis \code{\link{semPower.postHoc}} computes the achieved power, given an effect, alpha, N, and the model df
#' \item Compromise power analysis \code{\link{semPower.compromise}} computes the implied alpha and power, given an effect, the alpha/beta ratio, N, and the model df
#' }
#'
#' In SEM, the discrepancy between H0 and H1 (the magnitude of effect) refers to the difference in fit between two models. If only one model is defined (which is the default), power refers to the global chi-square test. If both models are explicitly defined, power is computed for nested model tests.  semPower allows for expressing the magnitude of effect by one of the following measures: F0, RMSEA, Mc, GFI, or AGFI.
#'
#' Alternatively, the implied effect can also be computed from the discrepancy between the population (or a certain model-implied) covariance matrix defining H0 and the hypothesized (model-implied) covariance matrix from a nested model defining H1. See the examples below how to use this feature in conjunction with lavaan.
#'
#'
#'
#' @docType package
#' @name semPower
#' @author  Morten Moshagen \email{morten.moshagen@@uni-ulm.de}
#' @Citation
#' If you use \code{semPower} in publications, please cite the package as follows:
#' 
#' Moshagen, M., & Erdfelder, E. (2016). A new strategy for testing structural equation models. \emph{Structural Equation Modeling, 23}, 54-60. doi: 10.1080/10705511.2014.950896
#'
#'  @examples
#' # a-priori power analyses using rmsea = .05 a target power (1-beta) of .80
#' ap1 <- semPower.aPriori(0.05, 'RMSEA', alpha = .05, beta = .20, df = 200)
#' summary(ap1)
#' # a-priori power analyses using f0 = .75 and a target power of .95
#' ap2 <- semPower.aPriori(0.75, 'F0', alpha = .05, power = .95, df = 200)
#' summary(ap2)
#' # create a plot showing how power varies by N (given a certain effect)
#' semPower.powerPlot.byN(.05, 'RMSEA', alpha=.05, df=200, power.min=.05, power.max=.99)
#' # post-hoc power analyses using rmsea = .08
#' ph <- semPower.postHoc(.08, 'RMSEA', alpha = .05, N = 250, df = 50)
#' summary(ph)
#' # create a plot showing how power varies by the magnitude of effect (given a certain N)
#' semPower.powerPlot.byEffect('RMSEA', alpha=.05, N = 100, df=200, effect.min=.001, effect.max=.10)
#' # compromise power analyses using rmsea = .08 and an abratio of 2
#' cp <- semPower.compromise(.08, 'RMSEA', abratio = 2, N = 1000, df = 200)
#' summary(cp)
#'
#' # use lavaan to define effect through covariance matrices:
#' \dontrun{
#' library(lavaan)
#'
#' # define population model (= H1)
#' model.pop <- '
#' f1 =~ .8*x1 + .7*x2 + .6*x3
#' f2 =~ .7*x4 + .6*x5 + .5*x6
#' f1 ~~ 1*f1
#' f2 ~~ 1*f2
#' f1 ~~ 0.5*f2
#' '
#' # define (wrong) H0 model
#' model.h0 <- '
#' f1 =~ x1 + x2 + x3
#' f2 =~ x4 + x5 + x6
#' f1 ~~ 0*f2
#' '
#'
#' # get population covariance matrix; equivalent to a perfectly fitting H1 model
#' cov.h1 <- fitted(sem(model.pop))$cov
#' # get covariance matrix as implied by H0 model
#' res.h0 <- sem(model.h0, sample.cov = cov.h1, sample.nobs = 1000, likelihood='wishart', sample.cov.rescale = F)
#' df <- res.h0@test[[1]]$df
#' cov.h0 <- fitted(res.h0)$cov
#'
#' ## do power analyses
#'
#' # post-hoc
#' ph <- semPower.postHoc(SigmaHat = cov.h0, Sigma = cov.h1, alpha = .05, N = 1000, df = df)
#' summary(ph)
#' # => Power to reject the H1 model is > .9999 (1-beta = 1-1.347826e-08) with N = 1000 at alpha=.05
#'
#' # compare:
#' ph$fmin * (ph$N-1)
#' fitmeasures(res.h1, 'chisq')
#' # => expected chi-square matches empirical chi-square
#'
#' # a-priori
#' ap <- semPower.aPriori(SigmaHat = cov.h0, Sigma = cov.h1, alpha = .05, power = .80, df = df)
#' summary(ap)
#' # => N = 194 gives a power of ~80% to reject the H1 model at alpha = .05
#'
#' # compromise
#' cp <- semPower.compromise(SigmaHat = cov.h0, Sigma = cov.h1, abratio = 1, N = 1000, df = df)
#' summary(cp)
#' # => A critical Chi-Squared of 33.999 gives balanced alpha-beta
#' #    error probabilities of alpha=beta=0.000089 with N = 1000.
#'
#' }
#'
"_PACKAGE"

#library(devtools)
#devtools::document()

