#' Interaction Functions
#'
#' \code{interactions} creates interaction measurement items by multipying all combination of factor items.
#'
#' This function automatically generates interaction measurement items for a PLS SEM.
#'
#' @param ... Interaction Combinations as generated by the \code{interaction_scaled} or
#' \code{interaction_ortho} methods.
#'
#' @usage
#'
#' interactions(...)
#'
#' @examples
#' data(mobi)
#'
#' # seminr syntax for creating measurement model
#' mobi_mm <- constructs(
#'   composite("Image",        multi_items("IMAG", 1:5)),
#'   composite("Expectation",  multi_items("CUEX", 1:3)),
#'   composite("Value",        multi_items("PERV", 1:2)),
#'   composite("Satisfaction", multi_items("CUSA", 1:3))
#' )
#' mobi_xm <- interactions(
#'   interaction_ortho("Image", "Expectation"),
#'   interaction_ortho("Image", "Value")
#' )
#'
#' #  structural model: note that name of the interactions factor should be
#' #  the names of its two main factors joined by a '.' in between.
#' mobi_sm <- relationships(
#'   paths(to = "Satisfaction",
#'         from = c("Image", "Expectation", "Value",
#'                  "Image.Expectation", "Image.Value"))
#' )
#'
#' mobi_pls <- estimate_pls(mobi, mobi_mm, mobi_xm, mobi_sm)
#' summary(mobi_pls)
#'
#' @export
interactions <- function(...) {
  function(data, mm, all_intxns=list(...)) {
    create_interaction <- function(intxn_function) { intxn_function(data, mm) }
    intxns_list <- lapply(all_intxns, create_interaction)
    return(intxns_list)
  }
}

#' \code{interaction_ortho} creates interaction measurement items by using the orthogonalized approach..
#'
#' This function automatically generates interaction measurement items for a PLS SEM using the orthogonalized approach..
#'
#' @param factor1 The first factor which is subject to the interaction.
#' @param factor2 The second factor which is subject to the interaction.
#'
#' @usage
#'  # orthogonalization approach as per Henseler & CHin (2010):
#'  interaction_ortho(factor1, factor2)
#'
#' @references Henseler & Chin (2010), A comparison of approaches for the analysis of interaction effects
#' between latent variables using partial least squares path modeling. Structural Equation Modeling, 17(1),82-109.
#'
#' @examples
#' data(mobi)
#'
#' # seminr syntax for creating measurement model
#' mobi_mm <- constructs(
#'   composite("Image",        multi_items("IMAG", 1:5)),
#'   composite("Expectation",  multi_items("CUEX", 1:3)),
#'   composite("Value",        multi_items("PERV", 1:2)),
#'   composite("Satisfaction", multi_items("CUSA", 1:3))
#' )
#' mobi_xm <- interactions(
#'   interaction_ortho("Image", "Expectation"),
#'   interaction_ortho("Image", "Value")
#' )
#'
#' #  structural model: note that name of the interactions factor should be
#' #  the names of its two main factors joined by a '.' in between.
#' mobi_sm <- relationships(
#'   paths(to = "Satisfaction",
#'         from = c("Image", "Expectation", "Value",
#'                  "Image.Expectation", "Image.Value"))
#' )
#'
#' mobi_pls <- estimate_pls(mobi, mobi_mm, mobi_xm, mobi_sm)
#' summary(mobi_pls)
#'
#' @export
interaction_ortho <- function(factor1, factor2) {
  function(data, mm) {
    interaction_name <- paste(factor1, factor2, sep=".")
    iv1_items <- mm[mm[, "latent"] == factor1, ][, "measurement"]
    iv2_items <- mm[mm[, "latent"] == factor2, ][, "measurement"]

    iv1_data <- as.data.frame(scale(data[iv1_items]))
    iv2_data <- as.data.frame(scale(data[iv2_items]))

    mult <- function(col) {
      iv2_data*col
    }

    multiples_list <- lapply(iv1_data, mult)
    interaction_data <- do.call("cbind", multiples_list)

    # Create formula
    frmla <- stats::as.formula(paste("interaction_data[,i]",paste(as.vector(c(iv1_items,iv2_items)), collapse ="+"), sep = " ~ "))

    # iterate and orthogonalize
    for(i in 1:ncol(interaction_data)) {
      interaction_data[,i] <- stats::lm(formula = frmla, data = data)$residuals
    }
    return(list(name = interaction_name, data = interaction_data))
  }
}

#' \code{interaction_scaled} creates interaction measurement items by scaled product indicator approach..
#'
#' This function automatically generates interaction measurement items for a PLS SEM using scaled product indicator approach.
#'
#' @param factor1 The first factor which is subject to the interaction.
#' @param factor2 The second factor which is subject to the interaction.
#'
#' @usage
#'  # standardized product indicator approach as per Henseler & Chin (2010):
#'  interaction_scaled("factor1", "factor2")
#'
#' @references Henseler & Chin (2010), A comparison of approaches for the analysis of interaction effects
#' between latent variables using partial least squares path modeling. Structural Equation Modeling, 17(1),82-109.
#'
#' @examples
#' data(mobi)
#'
#' # seminr syntax for creating measurement model
#' mobi_mm <- constructs(
#'   composite("Image",        multi_items("IMAG", 1:5)),
#'   composite("Expectation",  multi_items("CUEX", 1:3)),
#'   composite("Value",        multi_items("PERV", 1:2)),
#'   composite("Satisfaction", multi_items("CUSA", 1:3))
#' )
#' mobi_xm <- interactions(
#'   interaction_scaled("Image", "Expectation"),
#'   interaction_scaled("Image", "Value")
#' )
#'
#' #  structural model: note that name of the interactions factor should be
#' #  the names of its two main factors joined by a '.' in between.
#' mobi_sm <- relationships(
#'   paths(to = "Satisfaction",
#'         from = c("Image", "Expectation", "Value",
#'                  "Image.Expectation", "Image.Value"))
#' )
#'
#' mobi_pls <- estimate_pls(mobi, mobi_mm, mobi_xm, mobi_sm)
#' summary(mobi_pls)
#'
#' @export
interaction_scaled <- function(factor1, factor2) {
  function(data, mm) {
    interaction_name <- paste(factor1, factor2, sep=".")
    iv1_items <- mm[mm[, "latent"] == factor1, ][, "measurement"]
    iv2_items <- mm[mm[, "latent"] == factor2, ][, "measurement"]

    iv1_data <- as.data.frame(scale(data[iv1_items]))
    iv2_data <- as.data.frame(scale(data[iv2_items]))

    mult <- function(col) {
      iv2_data*col
    }

    multiples_list <- lapply(iv1_data, mult)
    interaction_data <- do.call("cbind", multiples_list)

    return(list(name = interaction_name, data = interaction_data))
  }
}

