% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/influ_physig.R
\name{influ_physig}
\alias{influ_physig}
\title{Influential species detection - Phylogenetic signal}
\usage{
influ_physig(trait.col, data, phy, method = "K", cutoff = 2, track = TRUE,
  ...)
}
\arguments{
\item{trait.col}{The name of a column in the provided data frame with trait 
to be analyzed  (e.g. "Body_mass").}

\item{data}{Data frame containing species traits with row names matching tips
in \code{phy}.}

\item{phy}{A phylogeny (class 'phylo') matching \code{data}.}

\item{method}{Method to compute signal: can be "K" or "lambda".}

\item{cutoff}{The cutoff value used to identify for influential species
(see Details)}

\item{track}{Print a report tracking function progress (default = TRUE)}

\item{...}{Further arguments to be passed to \code{\link[phytools]{phylosig}}}
}
\value{
The function \code{influ_physig} returns a list with the following
components:

\code{cutoff}: The value selected for \code{cutoff}

\code{trait.col}: Column name of the trait analysed

\code{full.data.estimates}: Phylogenetic signal estimate (K or lambda)
and the P value (for the full data).

\code{influential_species}: List of influential species,
based on standardised difference in K or lambda. 
Species are ordered from most influential to less influential and
only include species with a standardised difference > \code{cutoff}.

\code{influ.physig.estimates}: A data frame with all simulation
estimates. Each row represents a deleted species 
Columns report the calculated signal estimate (\code{k}) or (\code{lambda}),
 difference between signal estimation of the reduced and full data 
 (\code{DF}), the percentage of change in signal compared
to the full data signal (\code{perc}) and p-value for the phylogenetic signal
test (\code{pval})

\code{data}: Original full dataset.
}
\description{
Performs leave-one-out deletion analysis for phylogenetic signal estimates,
and detects influential species for K or lambda.
}
\details{
This function sequentially removes one species at a time, ans estimates phylogenetic
signal (K or lambda) using \code{\link[phytools]{phylosig}}, stores the
results and detects the most influential species.

\code{influ_physig} detects influential species based on the standardised
difference in signal estimate (K or lambda) when removing a given species compared
to the full data estimate (with all species). Species with a standardised difference
above the value of \code{cutoff} are identified as influential. The default
value for the cutoff is 2 standardised differences in signal estimate.

Output can be visualised using \code{sensi_plot}.
}
\note{
The argument "se" from \code{\link[phytools]{phylosig}} is not available in this function. Use the 
argument "V" instead with \code{\link{intra_physig}} to indicate the name of the column containing the standard 
deviation or the standard error of the trait variable instead.
}
\examples{
\dontshow{
# Load data:
data(alien)
alien.data<-alien$data
alien.phy<-alien$phy
# Logtransform data
alien.data$logMass <- log(alien.data$adultMass) 
# Run sensitivity analysis:
influ <- influ_physig("logMass", data = alien.data[1:20,],
phy = alien.phy[[1]])
# To check summary results:
summary(influ)
}
\dontrun{
# Load data:
data(alien)
# Logtransform data
alien.data$logMass <- log(alien.data$adultMass) 
# Run sensitivity analysis:
influ <- influ_physig("logMass", data = alien.data, phy = alien.phy[[1]])
# To check summary results:
summary(influ)
# Most influential speciesL
influ$influential.species
# Visual diagnostics
sensi_plot(influ)
# You can specify which graph to print: 
sensi_plot(influ, graphs = 1)
sensi_plot(influ, graphs = 2)
}
}
\references{
Blomberg, S. P., T. Garland Jr., A. R. Ives (2003) 
Testing for phylogenetic signal in comparative data: 
Behavioral traits are more labile. Evolution, 57, 717-745.

Pagel, M. (1999) Inferring the historical patterns of biological evolution. 
Nature, 401, 877-884.

Kamilar, J. M., & Cooper, N. (2013). Phylogenetic signal in primate behaviour,
 ecology and life history. Philosophical Transactions of the Royal Society 
 B: Biological Sciences, 368: 20120341.
}
\seealso{
\code{\link[phytools]{phylosig}},
\code{\link{influ_phylm}},\code{\link{sensi_plot}}
}
\author{
Gustavo Paterno
}
