% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sglg.R
\name{sglg}
\alias{sglg}
\title{Fitting semi-parametric generalized log-gamma regression models}
\usage{
sglg(formula, npc, basis, data, shape, Tolerance, Maxiter)
}
\arguments{
\item{formula}{a symbolic description of the systematic component of the model to be fitted. See details for further information.}

\item{npc}{a data frame with potential nonparametric variables of the systematic part of the model to be fitted.}

\item{basis}{a name of the cubic spline basis to be used in the model. Supported basis include deBoor and Gu basis
which are a B-spline basis and a natural cubic spline basis, respectively.}

\item{data}{an optional data frame, list containing the variables in the model.}

\item{shape}{an optional value for the shape parameter of the error distribution of a generalized log-gamma distribution. Default value is 1.}

\item{Tolerance}{an optional positive value, which represents the convergence criterion. Default value is 1e-04.}

\item{Maxiter}{an optional positive integer giving the maximal number of iterations for the estimating process. Default value is 1e03.}
}
\value{
mu a vector of parameter estimates asociated with the location parameter.

sigma estimate of the scale parameter associated with the model.

lambda estimate of the shape parameter associated with the model.

interval estimate of a 95\% confidence interval for each estimate parameters associated with the model.

Deviance the deviance associated with the model.
}
\description{
\code{sglg} is used to fit a semi-parametric regression model suitable for analysis of data sets in which the response variable is continuous, strictly positive, and asymmetric.
In this setup, the location parameter of the response variable is explicitly modeled by semi-parametric functions, whose nonparametric components may be approximated by
natural cubic splines or cubic P-splines.
}
\examples{
rows <- 240 # Number of observations
columns <- 2 # Number of parametric components
t_beta  <- c(0.5, 2)
t_sigma <- 1
t_lambda <- 1
set.seed(8142031)
library(ssym)
x1 <- rbinom(rows, 1, 0.5)
x2 <- runif(columns, 0, 1)
X <- cbind(x1,x2)
t_knot1 <- 6
ts1 <- seq(0, 1, length = t_knot1)
t_g1 <- 0.4 * sin(pi * ts1)

BasisN <- function(n, knot) {
          N <- matrix(0, n, knot)
          m <- n/knot
          block <- matrix(1, m, 1)
          for (i in 1:knot) {
          l <- (i - 1) * m + 1
          r <- i * m
          N[l:r, i] <- block }
          return(N)
          }
s_N1 <- BasisN(rows, length(ts1))
x3 <- s_N1 \%*\% ts1
colnames(x3) <- 'x3'
error <- robustloggamma::rloggamma(rows, 0, 1, t_lambda)
y2 <- X \%*\%t_beta + + s_N1 \%*\% t_g1 + t_sigma * error
data.example <- data.frame(y2,X,x3)
fit2 <- sglg(y2 ~ x1 + x2, npc=x3, data=data.example)
}
\references{
Carlos A. Cardozo, G. Paula and L. Vanegas. Semi-parametric generalized log-gamma regression models. In preparation.
}
\author{
Carlos Alberto Cardozo Delgado <cardozorpackages@gmail.com>, G. Paula and L. Vanegas.
}
