% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/components.R, R/documentation.R, R/examples.R
\name{Modal}
\alias{Modal}
\title{Modal}
\usage{
Modal(...)
}
\arguments{
\item{...}{Props to pass to the component.
The allowed props are listed below in the \bold{Details} section.}
}
\value{
Object with \code{shiny.tag} class suitable for use in the UI of a Shiny app.
}
\description{
Modals are temporary pop-ups that take focus from the page or app and require people to interact with them. Unlike a dialog box (\code{Dialog}), a modal should be used for hosting lengthy content, such as privacy statements or license agreements, or for asking people to perform complex or multiple actions, such as changing settings.

For more details and examples visit the \href{https://developer.microsoft.com/en-us/fluentui#/controls/web/Modal}{official docs}.
The R package cannot handle each and every case, so for advanced use cases
you need to work using the original docs to achieve the desired result.
}
\details{
\itemize{
\item {\bold{ allowTouchBodyScroll }} {\code{boolean} \cr Allow body scroll on content and overlay on touch devices. Changing after mounting has no effect.}
\item {\bold{ className }} {\code{string} \cr Optional class name to be added to the root class}
\item {\bold{ componentRef }} {\verb{IRefObject<IModal>} \cr Optional callback to access the IDialog interface. Use this instead of ref for accessing the public methods and properties of the component.}
\item {\bold{ containerClassName }} {\code{string} \cr Optional override for container class}
\item {\bold{ dragOptions }} {\code{IDragOptions} \cr The options to make the modal draggable}
\item {\bold{ enableAriaHiddenSiblings }} {\code{boolean} \cr Puts aria-hidden=true on all non-ancestors of the current modal, for screen readers. This is an experimental feature that will be graduated to default behavior after testing. This flag will be removed with the next major release.}
\item {\bold{ isBlocking }} {\code{boolean} \cr Whether the dialog can be light dismissed by clicking outside the dialog (on the overlay).}
\item {\bold{ isDarkOverlay }} {\code{boolean} \cr Whether the overlay is dark themed.}
\item {\bold{ isModeless }} {\code{boolean} \cr Whether the dialog should be modeless (e.g. not dismiss when focusing/clicking outside of the dialog). if true: isBlocking is ignored, there will be no overlay (isDarkOverlay is ignored), isClickableOutsideFocusTrap is true, and forceFocusInsideTrap is false}
\item {\bold{ isOpen }} {\code{boolean} \cr Whether the dialog is displayed.}
\item {\bold{ layerProps }} {\code{ILayerProps} \cr Defines an optional set of props to be passed through to Layer}
\item {\bold{ onDismiss }} {\verb{(ev?: React.MouseEvent<HTMLButtonElement>) => any} \cr A callback function for when the Modal is dismissed light dismiss, before the animation completes.}
\item {\bold{ onDismissed }} {\verb{() => any} \cr A callback function which is called after the Modal is dismissed and the animation is complete.}
\item {\bold{ onLayerDidMount }} {\verb{() => void} \cr A callback function for when the Modal content is mounted on the overlay layer}
\item {\bold{ overlay }} {\code{IOverlayProps} \cr Defines an optional set of props to be passed through to Overlay}
\item {\bold{ scrollableContentClassName }} {\code{string} \cr Optional override for scrollable content class}
\item {\bold{ styles }} {\verb{IStyleFunctionOrObject<IModalStyleProps, IModalStyles>} \cr Call to provide customized styling that will layer on top of the variant rules.}
\item {\bold{ subtitleAriaId }} {\code{string} \cr ARIA id for the subtitle of the Modal, if any}
\item {\bold{ theme }} {\code{ITheme} \cr Theme provided by High-Order Component.}
\item {\bold{ titleAriaId }} {\code{string} \cr ARIA id for the title of the Modal, if any}
\item {\bold{ topOffsetFixed }} {\code{boolean} \cr Whether the modal should have top offset fixed once opened and expand from the bottom only when the content changes dynamically.}
}
}
\section{Best practices}{
\subsection{Layout}{
\itemize{
\item Use a variable width with a minimum width of 288 pixels.
\item Use a variable height with a minimum height of 172 pixels.
\item Center vertically and horizontally in the available space.
\item Always have at least one focusable element inside a modal.
\item Blocking modals (\verb{Modeless Modal}) should be used very sparingly, only when it's critical for people to make a choice or provide information before they can proceed.
\item Provide a clear way for people to dismiss the control, such as a Close button, which should always go in the upper right corner.
}
}

\subsection{Content}{
\itemize{
\item Use sentence-style capitalization—only capitalize the first word. For more info, see \href{https://docs.microsoft.com/style-guide/capitalization}{Capitalization} in the Microsoft Writing Style Guide.
}
}
}

\examples{
library(shiny.fluent)

if (interactive()) {
  shinyApp(
    ui = tagList(
      reactOutput("modal"),
      PrimaryButton.shinyInput("showModal", text = "Show modal"),
    ),
    server = function(input, output) {
      modalVisible <- reactiveVal(FALSE)
      observeEvent(input$showModal, modalVisible(TRUE))
      observeEvent(input$hideModal, modalVisible(FALSE))
      output$modal <- renderReact({
        Modal(isOpen = modalVisible(),
          Stack(tokens = list(padding = "15px", childrenGap = "10px"),
            div(style = list(display = "flex"),
              Text("Title", variant = "large"),
              div(style = list(flexGrow = 1)),
              IconButton.shinyInput("hideModal", iconProps = list(iconName = "Cancel")),
            ),
            div(
              p("A paragraph of text."),
              p("Another paragraph.")
            )
          )
        )
      })
    }
  )
}
}
