% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/metareactive.R
\name{expandChain}
\alias{expandChain}
\alias{newExpansionContext}
\title{Expand code objects}
\usage{
newExpansionContext()

expandChain(..., .expansionContext = newExpansionContext())
}
\arguments{
\item{...}{All arguments must be unnamed, and must be one of: 1) calls to
meta-reactive objects, 2) comment string (e.g. \code{"# A comment"}), 3)
language object (e.g. \code{quote(print(1 + 1))}), or 4) \code{NULL} (which will be
ignored). Calls to meta-reactive objects can optionally be \code{\link[=invisible]{invisible()}},
see Details.}

\item{.expansionContext}{Accept the default value if calling \code{expandChain} a
single time to generate a corpus of code; or create an expansion context
object using \code{newExpansionContext()} and pass it to multiple related calls
of \code{expandChain}. See Details.}
}
\value{
The return value of \code{expandChain()} is a code object that's suitable for
printing or passing to \code{\link[=displayCodeModal]{displayCodeModal()}}, \code{\link[=buildScriptBundle]{buildScriptBundle()}}, or
\code{\link[=buildRmdBundle]{buildRmdBundle()}}.

The return value of \code{newExpansionContext} is an object that should be
passed to multiple \code{expandChain()} calls.
}
\description{
Use \code{expandChain} to write code out of one or more metaReactive objects.
Each meta-reactive object (expression, observer, or renderer) will cause not
only its own code to be written, but that of its dependencies as well.
}
\details{
There are two ways to extract code from meta objects (i.e. \code{\link[=metaReactive]{metaReactive()}},
\code{\link[=metaObserve]{metaObserve()}}, and \code{\link[=metaRender]{metaRender()}}): \code{withMetaMode()} and \code{expandChain()}.
The simplest is \code{withMetaMode(obj())}, which crawls the tree of meta-reactive
dependencies and expands each \code{..()} in place.

For example, consider these meta objects:\preformatted{    nums <- metaReactive(\{ runif(100) \})
    obs <- metaObserve(\{
      summary(..(nums()))
      hist(..(nums()))
    \})
}

When code is extracted using \code{withMetaMode}:\preformatted{    withMetaMode(obs())
}

The result looks like this:\preformatted{    summary(runif(100))
    plot(runif(100))
}

Notice how \code{runif(100)} is inlined wherever \code{..(nums())}
appears, which is not desirable if we wish to reuse the same
values for \code{summary()} and \code{plot()}.

The \code{expandChain} function helps us workaround this issue
by assigning return values of \code{metaReactive()} expressions to
a name, then replaces relevant expansion (e.g., \code{..(nums())})
with the appropriate name (e.g. \code{nums}).\preformatted{    expandChain(obs())
}

The result looks like this:\preformatted{    nums <- runif(100)
    summary(nums)
    plot(nums)
}

You can pass multiple meta objects and/or comments to \code{expandChain}.\preformatted{    expandChain(
      "# Generate values",
      nums(),
      "# Summarize and plot",
      obs()
    )
}

Output:\preformatted{    # Load data
    nums <- runif(100)
    nums
    # Inspect data
    summary(nums)
    plot(nums)
}

You can suppress the printing of the \code{nums} vector in the previous example by
wrapping the \code{nums()} argument to \code{expandChain()} with \code{invisible(nums())}.
}
\section{Preserving dependencies between \code{expandChain()} calls}{


Sometimes we may have related meta objects that we want to generate code for,
but we want the code for some objects in one code chunk, and the code for
other objects in another code chunk; for example, you might be constructing
an R Markdown report that has a specific place for each code chunk.

Within a single \code{expandChain()} call, all \code{metaReactive} objects are
guaranteed to only be declared once, even if they're declared on by multiple
meta objects; but since we're making two \code{expandChain()} calls, we will end
up with duplicated code. To remove this duplication, we need the second
\code{expandChain} call to know what code was emitted in the first \code{expandChain}
call.

We can achieve this by creating an "expansion context" and sharing it between
the two calls.\preformatted{    exp_ctx <- newExpansionContext()
    chunk1 <- expandChain(.expansionContext = exp_ctx,
      invisible(nums())
    )
    chunk2 <- expandChain(.expansionContext = exp_ctx,
      obs()
    )
}

After this code is run, \code{chunk1} contains only the definition of \code{nums} and
\code{chunk2} contains only the code for \code{obs}.
}

\section{Substituting \code{metaReactive} objects}{


Sometimes, when generating code, we want to completely replace the
implementation of a \code{metaReactive}. For example, our Shiny app might contain
this logic, using \code{\link[shiny:fileInput]{shiny::fileInput()}}:\preformatted{    data <- metaReactive2(\{
      req(input$file_upload)
      metaExpr(read.csv(..(input$file_upload$datapath)))
    \})
    obs <- metaObserve(\{
      summary(..(data()))
    \})
}

Shiny's file input works by saving uploading files to a temp directory. The
file referred to by \code{input$file_upload$datapath} won't be available when
another user tries to run the generated code.

You can use the expansion context object to swap out the implementation of
\code{data}, or any other \code{metaReactive}:\preformatted{    ec <- newExpansionContext()
    ec$substituteMetaReactive(data, function() \{
      metaExpr(read.csv("data.csv"))
    \})

    expandChain(.expansionContext = ec, obs())
}

Result:\preformatted{    data <- read.csv("data.csv")
    summary(data)
}

Just make sure this code ends up in a script or Rmd bundle that includes the
uploaded file as \code{data.csv}, and the user will be able to reproduce your
analysis.

The \code{substituteMetaReactive} method takes two arguments: the \code{metaReactive}
object to substitute, and a function that takes zero arguments and returns a
quoted expression (for the nicest looking results, use \code{metaExpr} to create
the expression). This function will be invoked the first time the
\code{metaReactive} object is encountered (or if the \code{metaReactive} is defined
with \code{inline = TRUE}, then every time it is encountered).
}

\examples{
input <- list(dataset = "cars")

# varname is only required if srcref aren't supported
# (R CMD check disables them for some reason?)
mr <- metaReactive({
  get(..(input$dataset), "package:datasets")
})

top <- metaReactive({
  head(..(mr()))
})

bottom <- metaReactive({
  tail(..(mr()))
})

obs <- metaObserve({
  message("Top:")
  summary(..(top()))
  message("Bottom:")
  summary(..(bottom()))
})

# Simple case
expandChain(obs())

# Explicitly print top
expandChain(top(), obs())

# Separate into two code chunks
exp_ctx <- newExpansionContext()
expandChain(.expansionContext = exp_ctx,
  invisible(top()),
  invisible(bottom()))
expandChain(.expansionContext = exp_ctx,
  obs())

}
\references{
\url{https://rstudio.github.io/shinymeta/articles/code-generation.html}
}
