\name{Bclust}
\alias{Bclust}
\title{Bootstrapped hclust}

\description{Bootstraps (or jacknifes) hierarchical clustering}

\usage{
Bclust(data, method.d="euclidean", method.c="ward.D",
FUN=function(.x) hclust(dist(.x, method=method.d), method=method.c),
iter=1000, mc.cores=1, monitor=TRUE, bootstrap=TRUE)
}

\arguments{

  \item{data}{Data suitable for the chosen distance method}

  \item{method.d}{Method for dist()}

  \item{method.c}{Method for hclust()}

  \item{FUN}{Function to make 'hclust' objects}

  \item{iter}{Number of replicates}

  \item{mc.cores}{\code{integer}, number of processes to
  run in parallel}

  \item{monitor}{If TRUE (default), prints a dot for each replicate}

  \item{bootstrap}{If FALSE (not default), performs jacknife (and makes 'iter=ncol(data)')}
}

\value{
  List with the following components: 'values' for bootstrapped
  frequencies of each node, 'hcl' for original 'hclust' object,
  'consensus' which is a sum of all Hcl2mat() matrices.
}

\details{

  This function provides bootstrapping for hierarchical clustering
  (\code{\link{hclust}} objects). Internally, it uses Hcl2mat() which
  converts 'hclust' objects into binary matrix of cluster memberships.

  Bclust() and companion functions were based on functions from the
  'bootstrap' package of Sebastian Gibb.

  Please note that this method frequently underestimates the cluster
  stability when number of characters is relatively small. One of
  possible remedies is to use hyber-binding (like "cbind(data, data,
  data)") to reach the reliable number of characters.

}

\references{

  Felsenstein J. 1985. Confidence limits on phylogenies: an approach
  using the bootstrap. Evolution. 39 (4): 783--791.

  Efron B., Halloran E., Holmes S. 1996. Bootstrap confidence levels for
  phylogenetic trees. Proceedings of the National Academy of Sciences. 93
  (23): 13429--13429.

}

\seealso{
  \code{\link{Jclust}}, \code{\link{BootA}}, \code{\link{Hcl2mat}},
  \code{\link{Bclabels}}, \code{\link{Hcoords}}
}

\examples{
(bb <- Bclust(t(atmospheres))) # specify 'mc.cores=4' or similar to speed up the process

plot(bb$hclust)
Bclabels(bb$hclust, bb$values, col="blue", pos=3, offset=0.1)

plot(bb$hclust)
Bclabels(bb$hclust, bb$values, col="blue", pos=3, offset=0.1, threshold=0.9)

plot(bb$hclust)
Bclabels(bb$hclust, bb$values, method="points", threshold=0.9, pch=19, cex=2)

plot(bb$hclust)
Bclabels(bb$hclust, bb$values, method="points", pch=19, cex=bb$values*3)

coords1 <- Hcoords(bb$hclust)
plot(bb$hclust)
Bclabels(bb$hclust, bb$values, coords=coords1, method="points", pch=19,
 cex=bb$values*3)

oldpar <- par(mar=c(2,1,0,4))
Ploth(bb$hclust, horiz=TRUE)
Bclabels(bb$hclust, bb$values, col="blue", pos=3, offset=0.1, horiz=TRUE)
par(oldpar)

plot(hclust(dist(bb$consensus)), main="Net consensus tree") # net consensus
## majority rule is 'consensus >= 0.5', strict is like 'round(consensus) == 1'

bb1 <- Bclust(t(atmospheres), FUN=function(.x) hclust(Gower.dist(.x)), monitor=FALSE)
plot(bb1$hclust)
Bclabels(bb1$hclust, bb1$values, col="green", pos=3, offset=0.1)

Bclust(t(atmospheres), bootstrap=FALSE) # jacknife
}

\keyword{cluster}
