#' @title Access information from model objects
#' @name pred_vars
#'
#' @description Several functions to retrieve information from model objects,
#'    like variable names, link-inverse function, model frame,
#'    model family etc., in a tidy and consistent way.
#'
#' @param x A fitted model; for \code{var_names()}, \code{x} may also be a
#'    character vector.
#' @param fe.only Logical, if \code{TRUE} and \code{x} is a mixed effects
#'    model, \code{model_frame()} returns the model frame for fixed effects only,
#'    and \code{pred_vars()} returns only fixed effects terms. Note that the default
#'    for \code{model_frame()} is \code{fe.only = TRUE}, while for \code{pred_vars()}
#'    the default is \code{fe.only = FALSE}.
#' @param mv,multi.resp Logical, if \code{TRUE} and model is a multivariate response
#'    model from a \code{brmsfit} object or of class \code{stanmvreg}, then a
#'    list of values (one for each regression) is returned.
#' @param combine Logical, if \code{TRUE} and the response is a matrix-column,
#'    the name of the response matches the notation in formula, and would for
#'    instance also contain patterns like \code{"cbind(...)"}. Else, the original
#'    variable names from the matrix-column are returned. See 'Examples'.
#' @param zi Logical, if \code{TRUE} and model has a zero-inflation-formula,
#'    the variable(s) used in this formula are also returned.
#' @param disp Logical, if \code{TRUE} and model is of class \code{glmmTMB} and
#'    has a dispersion-formula, the variable(s) used in the dispersion-formula
#'    are also returned.
#' @param ... Currently not used.
#'
#' @return For \code{pred_vars()} and \code{resp_var()}, the name(s) of the
#'    response or predictor variables from \code{x} as character vector.
#'    \code{resp_val()} returns the values from \code{x}'s response vector.
#'    \code{re_grp_var()} returns the group factor of random effects in
#'    mixed models, or \code{NULL} if \code{x} has no such random effects term
#'    (\code{grp_var()} is an alias for \code{re_grp_var()}).
#'    \cr \cr
#'    \code{link_inverse()} returns, if known, the inverse link function from
#'    \code{x}; else \code{NULL} for those models where the inverse link function
#'    can't be identified.
#'    \cr \cr
#'    \code{model_frame()} is similar to \code{model.frame()},
#'    but should also work for model objects that don't have a S3-generic for
#'    \code{model.frame()}.
#'    \cr \cr
#'    \code{var_names()} returns the "cleaned" variable
#'    names, i.e. things like \code{s()} for splines or \code{log()} are
#'    removed.
#'    \cr \cr
#'    \code{model_family()} returns a list with information about the
#'    model family (see 'Details').
#'
#' @details \code{model_family()} returns a list with information about the
#'    model family for many different model objects. Following information
#'    is returned, where all values starting with \code{is_} are logicals.
#'    \itemize{
#'      \item \code{is_bin}: family is binomial (but not negative binomial)
#'      \item \code{is_pois}: family is either poisson or negative binomial
#'      \item \code{is_negbin}: family is negative binomial
#'      \item \code{is_count}: model is a count model (i.e. family is either poisson or negative binomial)
#'      \item \code{is_beta}: family is beta
#'      \item \code{is_logit}: model has logit link
#'      \item \code{is_linear}: family is gaussian
#'      \item \code{is_ordinal}: family is ordinal or cumulative link
#'      \item \code{is_categorical}: family is categorical link
#'      \item \code{is_zeroinf}: model has zero-inflation component
#'      \item \code{is_multivariate}: model is a multivariate response model (currently only works for \emph{brmsfit} objects)
#'      \item \code{is_trial}: model response contains additional information about the trials
#'      \item \code{link.fun}: the link-function
#'      \item \code{family}: the family-object
#'    }
#'    \code{model_frame()} slighty differs from \code{model.frame()}, especially
#'    for spline terms and matrix-variables created with \code{cbind()} (for example
#'    in binomial models, where the response is a combination of successes and
#'    trials) . Where \code{model.frame()} returns a matrix for splines,
#'    \code{model_frame()} returns the data of the original variable and uses
#'    the same column name as in the \code{data}-argument from the model-function.
#'    This makes it easier, for instance, to get data that should be used as new
#'    data in \code{predict()}. For matrix-variables created with \code{cbind()},
#'    \code{model_frame()} returns the original variable as matrix and
#'    \emph{additionally} each column as own variable. See 'Examples'.
#'
#' @examples
#' data(efc)
#' fit <- lm(neg_c_7 ~ e42dep + c161sex, data = efc)
#'
#' pred_vars(fit)
#' resp_var(fit)
#' resp_val(fit)
#'
#' link_inverse(fit)(2.3)
#'
#' # example from ?stats::glm
#' counts <- c(18, 17, 15, 20, 10, 20, 25, 13, 12)
#' outcome <- gl(3, 1, 9)
#' treatment <- gl(3, 3)
#' m <- glm(counts ~ outcome + treatment, family = poisson())
#'
#' link_inverse(m)(.3)
#' # same as
#' exp(.3)
#'
#' outcome <- as.numeric(outcome)
#' m <- glm(counts ~ log(outcome) + as.factor(treatment), family = poisson())
#' var_names(m)
#'
#' # model.frame and model_frame behave slightly different
#' library(splines)
#' m <- lm(neg_c_7 ~ e42dep + ns(c160age, knots = 2), data = efc)
#' head(model.frame(m))
#' head(model_frame(m))
#'
#' library(lme4)
#' data(cbpp)
#' cbpp$trials <- cbpp$size - cbpp$incidence
#' m <- glm(cbind(incidence, trials) ~ period, data = cbpp, family = binomial)
#' head(model.frame(m))
#' head(model_frame(m))
#'
#' resp_var(m, combine = TRUE)
#' resp_var(m, combine = FALSE)
#'
#' # get random effects grouping factor from mixed models
#' library(lme4)
#' data(sleepstudy)
#' m <- lmer(Reaction ~ Days + (1 + Days | Subject), data = sleepstudy)
#' re_grp_var(m)
#'
#' # get model predictors, with and w/o dispersion formula
#' \dontrun{
#' library(glmmTMB)
#' data("Salamanders")
#' m <- glmmTMB(
#'   count ~ spp + cover + mined + poly(DOP, 3) + (1 | site),
#'   ziformula = ~spp + mined,
#'   dispformula = ~DOY,
#'   data = Salamanders,
#'   family = nbinom2
#' )
#'
#' pred_vars(m)
#' pred_vars(m, fe.only = TRUE)
#' pred_vars(m, disp = TRUE)}
#'
#' @importFrom purrr flatten_chr map
#' @importFrom stats formula terms
#' @export
pred_vars <- function(x, ...) {
  UseMethod("pred_vars")
}


#' @rdname pred_vars
#' @export
pred_vars.default <- function(x, fe.only = FALSE, ...) {
  fm <- stats::formula(x)
  av <- all.vars(fm[[3L]])

  if (length(av) == 1 && av == ".")
    av <- all.vars(stats::terms(x)[[3L]])

  # remove random effects from formula

  if (fe.only) {
    re <- re_grp_var(x)
    if (!sjmisc::is_empty(re)) {
      re <- unique(sjmisc::trim(unlist(strsplit(re, ":", fixed = TRUE))))
      pos <- match(re, av)
      av <- av[-pos]
    }
  }

  unique(av)
}


#' @export
pred_vars.clm2 <- function(x, ...) {
  fm <- attr(x$location, "terms", exact = TRUE)
  av <- all.vars(fm[[3L]])

  if (length(av) == 1 && av == ".")
    av <- all.vars(stats::terms(x)[[3L]])

  unique(av)
}


#' @rdname pred_vars
#' @export
pred_vars.glmmTMB <- function(x, fe.only = FALSE, zi = FALSE, disp = FALSE, ...) {
  fm <- stats::formula(x)
  av <- all.vars(fm[[3L]])

  if (length(av) == 1 && av == ".")
    av <- all.vars(stats::terms(x)[[3L]])

  # remove random effects from formula

  if (fe.only) {
    re <- re_grp_var(x)
    if (!sjmisc::is_empty(re)) {
      re <- unique(sjmisc::trim(unlist(strsplit(re, ":", fixed = TRUE))))
      pos <- match(re, av)
      av <- av[-pos]
    }
  }

  # add variables from zero-inflation

  if (isTRUE(zi)) {
    dp <- tryCatch(
      {all.vars(x$modelInfo$allForm$ziformula[[2L]])},
      error = function(x) { NULL}
    )

    if (!is.null(dp)) av <- c(av, dp)
  }

  # for glmmtmb, check dispersion formula

  if (isTRUE(disp)) {
    dp <- tryCatch(
      {all.vars(x$modelInfo$allForm$dispformula[[2L]])},
      error = function(x) { NULL}
    )

    if (!is.null(dp)) av <- c(av, dp)
  }

  unique(av)
}


#' @export
pred_vars.brmsfit <- function(x, fe.only = FALSE, ...) {
  fm <- stats::formula(x)

  if (!is.null(fm$responses)) {
    av <- fm$forms %>%
      purrr::map(~ all.vars(stats::formula(.x)[[3L]])) %>%
      purrr::flatten_chr() %>%
      unique()
  } else
    av <- all.vars(fm$formula[[3L]])

  if (length(av) == 1 && av == ".")
    av <- all.vars(stats::terms(x)[[3L]])

  # remove random effects from formula

  if (fe.only) {
    re <- re_grp_var(x)
    if (!sjmisc::is_empty(re)) {
      re <- unique(sjmisc::trim(unlist(strsplit(re, ":", fixed = TRUE))))
      pos <- match(re, av)
      av <- av[-pos]
    }
  }

  unique(av)
}


#' @export
pred_vars.MCMCglmm <- function(x, fe.only = FALSE, ...) {
  fm <- x$Fixed
  av <- all.vars(fm$formula[[3L]])

  if (!fe.only) {
    fmr <- x$Random
    avr <- all.vars(fmr$formula[[2L]])
    av <- c(av, avr)
  }

  unique(av)
}


#' @rdname pred_vars
#' @export
pred_vars.MixMod <- function(x, fe.only = FALSE, zi = FALSE, ...) {
  fm <- stats::formula(x)
  av <- all.vars(fm[[3L]])

  if (!fe.only) {
    av <- c(av, x$id_name)
    avrandom <- all.vars(stats::formula(x, type = "random")[[2L]])
    if (!sjmisc::is_empty(avrandom)) av <- c(av, avrandom)
  }

  if (isTRUE(zi)) {
    avzi <- all.vars(stats::formula(x, type = "zi_fixed")[[2L]])
    if (!sjmisc::is_empty(avzi)) av <- c(av, avzi)
  }

  if (length(av) == 1 && av == ".")
    av <- all.vars(stats::terms(x)[[3L]])

  unique(av)
}


#' @export
pred_vars.gam <- function(x, ...) {
  fm <- stats::formula(x)
  if (is.list(fm)) fm <- fm[[1]]
  av <- all.vars(fm[[3L]])

  if (length(av) == 1 && av == ".")
    av <- all.vars(stats::terms(x)[[3L]])

  unique(av)
}


#' @export
pred_vars.stanmvreg <- function(x, fe.only = FALSE, ...) {
  fm <- stats::formula(x)

  av <- fm %>%
    purrr::map(~ all.vars(.x[[3L]])) %>%
    purrr::flatten_chr() %>%
    unique()

  if (length(av) == 1 && av == ".")
    av <- all.vars(stats::terms(x)[[3L]])

  # remove random effects from formula

  if (fe.only) {
    re <- re_grp_var(x)
    if (!sjmisc::is_empty(re)) {
      re <- unique(sjmisc::trim(unlist(strsplit(re, ":", fixed = TRUE))))
      pos <- match(re, av)
      av <- av[-pos]
    }
  }

  unique(av)
}


#' @export
pred_vars.felm <- function(x, ...) {
  fm <- stats::formula(x)
  av <- all.vars(fm[[2L]])

  if (length(av) == 1 && av == ".")
    av <- all.vars(stats::terms(x)[[3L]])

  unique(av)
}
