% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/blus.R
\name{blus}
\alias{blus}
\title{Compute Best Linear Unbiased Scalar-Covariance (BLUS) residuals from a linear model}
\usage{
blus(
  mainlm,
  omit = c("first", "last", "random"),
  keepNA = TRUE,
  exhaust = NA,
  seed = 1234
)
}
\arguments{
\item{mainlm}{Either an object of \code{\link[base]{class}} \code{"lm"}
(e.g., generated by \code{\link[stats]{lm}}), or
a list of two objects: a response vector and a design matrix. The objects
are assumed to be in that order, unless they are given the names
\code{"X"}  and \code{"y"} to distinguish them. The design matrix passed
in a list must begin with a column of ones if an intercept is to be
included in the linear model. The design matrix passed in a list should
not contain factors, as all columns are treated 'as is'. For tests that
use ordinary least squares residuals, one can also pass a vector of
residuals in the list, which should either be the third object or be
named \code{"e"}.}

\item{omit}{A numeric vector of length \eqn{p} (the number of columns in the
linear model design matrix) giving the indices of \eqn{p} observations to omit in
the BLUS residual vector; or a character partially matching \code{"first"}
(for the first \eqn{p}) observations, \code{"last"} (for the last \eqn{p}
observations), or \code{"random"} (for a random sample of \eqn{p} indices
between 1 and \eqn{n}). Defaults to \code{"first"}.}

\item{keepNA}{A logical. Should BLUS residuals for omitted observations be
returned as \code{NA_real_} to preserve the length of the residual vector?}

\item{exhaust}{An integer. If singular matrices are encountered
using the passed value of \code{omit}, how many random combinations
of \eqn{p} indices should be attempted before an error is thrown? If
\code{NA} (the default), all possible combinations are attempted
provided that \eqn{{n \choose p} \le 10^4}; otherwise up to
\eqn{10^4} random samples of size \code{p} from \code{1:n} are
attempted (with replacement). Integer values of \code{exhaust}
greater than \code{1e4L} are treated as \code{NA}.}

\item{seed}{An integer specifying a seed to pass to
\code{\link[base]{set.seed}} for random number generation. This allows
reproducibility of bootstrap results. The default value, \code{NA},
results in not setting a seed.}
}
\value{
A double vector of length \eqn{n} containing the BLUS residuals
   (with \code{NA_real_}) for omitted observations), or a double vector
   of length \eqn{n-p} containing the BLUS residuals only (if \code{keepNA}
   is set to \code{FALSE})
}
\description{
This function computes the Best Linear Unbiased Scalar-Covariance (BLUS)
   residuals from a linear model, as defined in
   \insertCite{Theil65;textual}{skedastic} and explained further in
   \insertCite{Theil68;textual}{skedastic}.
}
\details{
Under the ideal linear model conditions, the BLUS residuals have a scalar
   covariance matrix \eqn{\sigma^2 I} (meaning they have a constant variance
   and are mutually uncorrelated), unlike the OLS residuals, which have
   covariance matrix \eqn{\sigma^2 M} where \eqn{M} is a function of the
   design matrix. Use of BLUS residuals could improve the performance of
   tests for heteroskedasticity and/or autocorrelation in the linear model.
   A linear model with \eqn{n} observations and an \eqn{n\times p} design
   matrix yields only \eqn{n-p} BLUS residuals. The choice of which \eqn{p}
   observations will not be represented in the BLUS residuals is specified
   within the algorithm.
}
\examples{
mtcars_lm <- lm(mpg ~ wt + qsec + am, data = mtcars)
blus(mtcars_lm)
plot(mtcars_lm$residuals, blus(mtcars_lm))
# Same as first example
mtcars_list <- list("y" = mtcars$mpg, "X" = cbind(1, mtcars$wt, mtcars$qsec, mtcars$am))
blus(mtcars_list)
# Again same as first example
mtcars_list2 <- list("e" = mtcars_lm$residuals, "X" = cbind(1, mtcars$wt, mtcars$qsec, mtcars$am))
blus(mtcars_list2)
# BLUS residuals cannot be computed with `omit = "last"` in this example, so
# omitted indices are randomised:
blus(mtcars_lm, omit = "last")

}
\references{
{\insertAllCited{}}
}
\seealso{
H. D. Vinod's online article,
   \href{https://papers.ssrn.com/sol3/papers.cfm?abstract_id=2412740}{Theil's
   BLUS Residuals and R Tools for Testing and Removing Autocorrelation and
   Heteroscedasticity}, for an alternative function for computing BLUS
   residuals.
}
